package dev.bg.bikebridge.ui.screens.bike.components

import androidx.compose.animation.core.Animatable
import androidx.compose.animation.core.tween
import androidx.compose.foundation.Canvas
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.size
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.remember
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.geometry.CornerRadius
import androidx.compose.ui.geometry.Offset
import androidx.compose.ui.geometry.Size
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.drawscope.Stroke
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import dev.bg.bikebridge.R
import dev.bg.bikebridge.data.state.BatteryState
import dev.bg.bikebridge.util.Constants

@Composable
fun BatteryInfo(
    batteryState: BatteryState
) {
    Column(
        Modifier.fillMaxWidth(),
        verticalArrangement = Arrangement.spacedBy(12.dp)
    ) {
        Spacer(Modifier.size(6.dp))
        Text(
            stringResource(R.string.battery),
            fontSize = 24.sp,
            fontWeight = FontWeight.W400
        )
        Row(
            Modifier.fillMaxSize(),
            verticalAlignment = Alignment.CenterVertically,
            horizontalArrangement = Arrangement.SpaceBetween
        ) {
            if (batteryState.level != null) {
                Text(stringResource(R.string.battery_level, batteryState.level))

                val factor = remember { Animatable(0f) }

                LaunchedEffect(batteryState.level) {
                    factor.animateTo(
                        targetValue = batteryState.level.toFloat(),
                        animationSpec = tween(1500)
                    )
                }

                Canvas(
                    Modifier.size(
                        width = 40.dp,
                        height = 20.dp
                    )
                ) {
                    val w1 = size.width * 0.9f
                    val w2 = size.width - w1
                    val h1 = size.height * 0.33f
                    val h2 = size.height - h1
                    val s = 10f

                    drawRoundRect(
                        topLeft = Offset(x = s / 2f, y = 0f),
                        size = Size(width = (w1 - (s / 2f)) * (factor.value / 100f), height = size.height),
                        cornerRadius = CornerRadius(x = 5f, y = 5f),
                        color = when {
                            batteryState.level <= 33f -> Constants.IndicatorColors.Low
                            33f < batteryState.level && batteryState.level <= 66f -> Constants.IndicatorColors.Medium
                            66f < batteryState.level -> Constants.IndicatorColors.High
                            else -> Constants.IndicatorColors.Low
                        },
                    )
                    drawRoundRect(
                        topLeft = Offset(x = 0f, 0f),
                        size = Size(width = w1, height = size.height),
                        cornerRadius = CornerRadius(x = 5f, y = 5f),
                        style = Stroke(width = s),
                        color = Color.Gray
                    )
                    drawRoundRect(
                        topLeft = Offset(x = w1, y = h1),
                        size = Size(width = w2, height = h2 - h1),
                        cornerRadius = CornerRadius(x = 5f, y = 5f),
                        color = Color.Gray
                    )
                }
            }
        }
    }
}
