package dev.bg.bikebridge.ui.screens.bike

import android.bluetooth.BluetoothGattCharacteristic
import androidx.annotation.StringRes
import androidx.compose.animation.AnimatedVisibility
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Edit
import androidx.compose.material3.AlertDialog
import androidx.compose.material3.Button
import androidx.compose.material3.DropdownMenuItem
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.ExposedDropdownMenuBox
import androidx.compose.material3.ExposedDropdownMenuDefaults
import androidx.compose.material3.FloatingActionButton
import androidx.compose.material3.HorizontalDivider
import androidx.compose.material3.Icon
import androidx.compose.material3.Text
import androidx.compose.material3.TextField
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.unit.dp
import androidx.lifecycle.compose.collectAsStateWithLifecycle
import dev.bg.bikebridge.LocalGattBinder
import dev.bg.bikebridge.R
import dev.bg.bikebridge.ble.BleConstants
import dev.bg.bikebridge.data.state.GattServiceState
import dev.bg.bikebridge.ui.components.OutlinedTextFieldInputType
import dev.bg.bikebridge.ui.components.SettingsSwitch
import dev.bg.bikebridge.ui.components.TypedOutlinedTextFieldWithError
import dev.bg.bikebridge.ui.screens.bike.components.BatteryInfo
import dev.bg.bikebridge.ui.screens.bike.components.DeviceInfo
import dev.bg.bikebridge.ui.screens.bike.components.ShimanoInfo
import dev.bg.bikebridge.ui.screens.bike.components.ShimanoSettings
import dev.bg.bikebridge.ui.screens.bike.components.SramInfo
import dev.bg.bikebridge.util.BikeUtil
import dev.bg.bikebridge.util.Preferences
import dev.bg.bikebridge.util.ktx.toHexString

@Composable
fun BikeScreen() {
    val binder = LocalGattBinder.current

    binder ?: return

    val state by binder.state.collectAsStateWithLifecycle()

    BikeScreenImpl(state)
}

@Composable
private fun BikeScreenImpl(
    state: GattServiceState
) {
    var hudVisible by remember { mutableStateOf(false) }
    var customWriteDialog by remember { mutableStateOf(false) }

    val writeMode by Preferences.writeEnabledFlow.collectAsStateWithLifecycle()

    Box(Modifier.fillMaxSize()) {
        Column(
            Modifier
                .fillMaxSize()
                .padding(horizontal = 16.dp)
                .verticalScroll(rememberScrollState()),
            verticalArrangement = Arrangement.spacedBy(16.dp)
        ) {
            DeviceInfo(
                deviceName = state.deviceName,
                manufacturer = stringResource(BikeUtil.getManufacturerResource(state.manufacturer ?: -1)),
                firmwareVersion = state.firmwareVersion,
                rssi = state.rssi,
                bonded = state.bonded
            )
            AnimatedVisibility(state.batteryState != null) {
                if (state.batteryState != null) {
                    HorizontalDivider()
                    BatteryInfo(state.batteryState)
                }
            }
            AnimatedVisibility(state.shimanoState != null) {
                if (state.shimanoState != null) {
                    HorizontalDivider()
                    ShimanoInfo(state.shimanoState)
                }
            }
            AnimatedVisibility(
                state.manufacturer == BleConstants.Manufacturers.SHIMANO &&
                state.writableCharacteristics.isNotEmpty()
            ) {
                if (writeMode && state.writableCharacteristics.isNotEmpty()) {
                    HorizontalDivider()
                    ShimanoSettings(state)
                }
            }
            AnimatedVisibility(state.sramAxsState != null) {
                if (state.sramAxsState != null) {
                    HorizontalDivider()
                    SramInfo(state.sramAxsState)
                }
            }
        }
        if (writeMode) {
            FloatingActionButton(
                modifier = Modifier
                    .align(Alignment.BottomEnd)
                    .padding(bottom = 16.dp, end = 16.dp),
                onClick = {
                    customWriteDialog = true
                }
            ) {
                Icon(
                    Icons.Filled.Edit,
                    contentDescription = stringResource(R.string.edit)
                )
            }
        } else {
            FloatingActionButton(
                modifier = Modifier
                    .align(Alignment.BottomEnd)
                    .padding(bottom = 16.dp, end = 16.dp),
                onClick = {
                    hudVisible = true
                }
            ) {
                Icon(
                    painterResource(R.drawable.baseline_fullscreen_24),
                    contentDescription = null
                )
            }
        }
    }
    CustomWriteDialog(
        visible = customWriteDialog,
        writableCharacteristics = state.writableCharacteristics,
        onDismissRequest = { customWriteDialog = false }
    )
    BikeHud(
        visible = hudVisible,
        state = state,
        onDismissRequest = {
            hudVisible = false
        }
    )
}

sealed class InferredType(
    @StringRes val res: Int
) {
    data object None: InferredType(R.string.app_name)
    data object Hexadecimal: InferredType(R.string.hexadecimal)
    data object ASCII: InferredType(R.string.ascii)
}

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun CustomWriteDialog(
    visible: Boolean,
    writableCharacteristics: List<BluetoothGattCharacteristic>,
    onDismissRequest: () -> Unit
) {
    val ctx = LocalContext.current
    val binder = LocalGattBinder.current

    binder ?: return

    var characteristic by remember { mutableStateOf<BluetoothGattCharacteristic?>(null) }
    var payload by remember { mutableStateOf("") }
    var littleEndian by remember { mutableStateOf(true) }
    var characteristicSuggestionExpanded by remember { mutableStateOf(false) }
    var inferredType by remember { mutableStateOf<InferredType>(InferredType.None) }
    var txPayload by remember { mutableStateOf(byteArrayOf()) }

    LaunchedEffect(payload) {
        val strippedPayload = payload.filterNot { it.isWhitespace() }
        inferredType = when {
            strippedPayload.isEmpty() -> InferredType.None
            strippedPayload.matches(Regex("[0-9a-f]+")) -> InferredType.Hexadecimal
            strippedPayload.matches(Regex("[a-zA-z0-9\\s]*")) -> InferredType.ASCII
            else -> InferredType.None
        }
        txPayload = when (inferredType) {
            InferredType.Hexadecimal -> {
                strippedPayload
                    .apply { if (this.length % 2 != 0) { plus("0") } }
                    .chunked(2)
                    .map { it.toInt(16).toByte() }
                    .toByteArray()
            }
            InferredType.ASCII -> {
                strippedPayload.toByteArray()
            }
            InferredType.None -> {
                byteArrayOf()
            }
        }
    }

    LaunchedEffect(littleEndian) {
        txPayload = txPayload.reversedArray()
    }

    if (visible) {
        AlertDialog(
            onDismissRequest = onDismissRequest,
            dismissButton = {
                Button(onClick = onDismissRequest) {
                    Text(stringResource(R.string.cancel))
                }
            },
            confirmButton = {
                Button(
                    enabled = characteristic != null && payload.isNotEmpty() && inferredType != InferredType.None,
                    onClick = {
                        // TODO test & add send snackbar (todo impl global)
                        binder.writeCharacteristic(characteristic!!.uuid, txPayload)
                        onDismissRequest()
                    }
                ) {
                    Text(stringResource(R.string.write))
                }
            },
            title = { Text(stringResource(R.string.send_custom_payload)) },
            text = {
                Column(
                    Modifier.fillMaxWidth()
                ) {
                    ExposedDropdownMenuBox(
                        expanded = characteristicSuggestionExpanded,
                        onExpandedChange = { characteristicSuggestionExpanded = !characteristicSuggestionExpanded }
                    ) {
                        TextField(
                            readOnly = true,
                            value = characteristic?.uuid?.toString() ?: "",
                            onValueChange = {},
                            label = { Text(stringResource(R.string.uuid)) },
                            trailingIcon = {
                                ExposedDropdownMenuDefaults.TrailingIcon(characteristicSuggestionExpanded)
                            },
                            colors = ExposedDropdownMenuDefaults.textFieldColors(),
                            modifier = Modifier.menuAnchor()
                        )
                        ExposedDropdownMenu(
                            expanded = characteristicSuggestionExpanded,
                            onDismissRequest = { characteristicSuggestionExpanded = false }
                        ) {
                            writableCharacteristics.forEach { c ->
                                DropdownMenuItem(
                                    text = {
                                        Text(c.uuid.toString())
                                    },
                                    onClick = {
                                        characteristic = c
                                        characteristicSuggestionExpanded = false
                                    }
                                )
                            }
                        }
                    }
                    TypedOutlinedTextFieldWithError(
                        value = payload,
                        onValueChange = { payload = it },
                        label = { Text(stringResource(R.string.payload)) },
                        inputType = OutlinedTextFieldInputType.Alphanumeric,
                        isError = false,
                        errorText = "",
                        modifier = Modifier
                            .fillMaxWidth()
                            .height(128.dp)
                    )
                    SettingsSwitch(
                        title = stringResource(R.string.little_endian),
                        checked = littleEndian,
                        onChange = { littleEndian = it }
                    )
                    if (inferredType != InferredType.None) {
                        Text(stringResource(R.string.inferred_type, ctx.getString(inferredType.res)))
                        Text(stringResource(R.string.tx_payload, txPayload.toHexString(), txPayload.size))
                    }
                }
            }
        )
    }
}
