package dev.bg.bikebridge.ui.screens.bike

import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.material3.Surface
import androidx.compose.runtime.Composable
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.unit.dp
import androidx.compose.ui.window.Dialog
import androidx.compose.ui.window.DialogProperties
import dev.bg.bikebridge.data.state.GattServiceState
import dev.bg.bikebridge.ui.screens.bike.components.HudCard

@Composable
fun BikeHud(
    visible: Boolean,
    state: GattServiceState,
    onDismissRequest: () -> Unit
) {
    val ctx = LocalContext.current

    if (visible) {
        Dialog(
            properties = DialogProperties(usePlatformDefaultWidth = false),
            onDismissRequest = onDismissRequest
        ) {
            Surface(
                Modifier
                    .fillMaxSize()
                    .clickable { onDismissRequest() },
            ) {
                Column(
                    Modifier
                        .fillMaxSize()
                        .padding(16.dp),
                    verticalArrangement = Arrangement.spacedBy(16.dp)
                ) {
                    TwoByRow {
                        HudCard(
                            title = "battery",
                            value = "%d%%".format(state.batteryState?.level ?: 0),
                            modifier = it
                        )
                        HudCard(
                            title = "speed",
                            value = "%.1f km/h".format(state.shimanoState?.speed ?: 0f),
                            modifier = it
                        )
                    }
                    TwoByRow {
                        HudCard(
                            title = "mode",
                            value = state.shimanoState?.mode?.res?.let { res -> ctx.getString(res) } ?: "",
                            modifier = it
                        )
                        HudCard(
                            title = "cadence",
                            value = "%d rpm".format(state.shimanoState?.cadence ?: 0),
                            modifier = it
                        )
                    }
                    HudCard(
                        title = "total distance",
                        value = "%.1f km".format(state.shimanoState?.totalDistance ?: 0f),
                        modifier = Modifier.fillMaxWidth()
                    )
                    HudCard(
                        title = "average speed",
                        value = "%.1f km/h".format(state.shimanoState?.averageSpeed ?: 0f),
                        modifier = Modifier.fillMaxWidth()
                    )
                    HudCard(
                        title = "max speed",
                        value = "%.1f km/h".format(state.shimanoState?.maxSpeed ?: 0f),
                        modifier = Modifier.fillMaxWidth()
                    )
                }
            }
        }
    }
}

@Composable
fun TwoByRow(
    content: @Composable (
        modifier: Modifier
    ) -> Unit
) {
    Row(
        Modifier.fillMaxWidth(),
        horizontalArrangement = Arrangement.spacedBy(16.dp)
    ) {
        content(
            Modifier
                .fillMaxWidth()
                .weight(1f)
        )
    }
}
