package dev.bg.bikebridge.ui.components

import androidx.compose.animation.core.LinearEasing
import androidx.compose.animation.core.Spring
import androidx.compose.animation.core.animateFloat
import androidx.compose.animation.core.animateFloatAsState
import androidx.compose.animation.core.infiniteRepeatable
import androidx.compose.animation.core.rememberInfiniteTransition
import androidx.compose.animation.core.spring
import androidx.compose.animation.core.tween
import androidx.compose.foundation.Canvas
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.aspectRatio
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.material3.MaterialTheme
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.drawWithCache
import androidx.compose.ui.geometry.Offset
import androidx.compose.ui.geometry.Rect
import androidx.compose.ui.graphics.Paint
import androidx.compose.ui.graphics.drawscope.drawIntoCanvas
import androidx.compose.ui.graphics.rotate
import androidx.compose.ui.layout.boundsInWindow
import androidx.compose.ui.layout.onGloballyPositioned
import androidx.compose.ui.layout.onPlaced
import androidx.compose.ui.unit.dp
import dev.bg.bikebridge.util.ktx.squarePythagoras
import kotlin.random.Random

const val RADAR_SIZE_DP = 144f

private const val NO_BLIPS = 5
private const val SWEEP = 6000
private const val H = 10f

data class Blip(
    val x: Float,
    val y: Float,
    val angle: Int
)

@Composable
fun Radar() {
    val inf = rememberInfiniteTransition("Radar")
    val degrees by inf.animateFloat(
        initialValue = 0f,
        targetValue = 360f,
        animationSpec = infiniteRepeatable(
            animation = tween(SWEEP, easing = LinearEasing)
        ),
        label = "Radar angle"
    )

    Box(
        Modifier.fillMaxWidth(),
        contentAlignment = Alignment.TopCenter
    ) {
        RadarAnimation(degrees.toInt())
    }
}

@Composable
private fun RadarAnimation(
    progress: Int
) {
    val c = MaterialTheme.colorScheme.primary
    val p = remember {
        Paint().apply {
            color = c
            strokeWidth = 5f
        }
    }

    var boundingBox by remember { mutableStateOf<Rect?>(null) }
    val blips = remember(boundingBox) {
        val r = Random // keep seed
        if (boundingBox != null) {
            MutableList(NO_BLIPS) {
                val h = (r.nextFloat() * H).coerceIn(0.8f * H, 0.95f * H)
                Blip(
                    x = boundingBox!!.center.x squarePythagoras h,
                    y = boundingBox!!.center.y squarePythagoras h,
                    angle = r.nextInt(0, 360),
                )
            }
        } else {
            mutableListOf()
        }
    }

    blips.forEach { blip ->
        val alpha by animateFloatAsState(
            targetValue = if (progress in (blip.angle..blip.angle + 30)) 1f else 0f,
            animationSpec = spring(
                stiffness = Spring.StiffnessLow
            ),
            label = "Blip fade animation"
        )

        Box(
            Modifier
                .fillMaxWidth()
                .height(RADAR_SIZE_DP.dp)
                .aspectRatio(1f),
        ) {
            Canvas(
                modifier = Modifier.fillMaxSize(),
                onDraw = {
                    drawIntoCanvas {
                        it.rotate(blip.angle.toFloat(), boundingBox!!.center.x, boundingBox!!.center.y)
                        drawCircle(
                            radius = 10f,
                            center = Offset(blip.x, blip.y),
                            color = c,
                            alpha = alpha
                        )
                    }
                }
            )
        }
    }
    Canvas(
        modifier = Modifier
            .fillMaxWidth()
            .height(RADAR_SIZE_DP.dp)
            .aspectRatio(1f)
            .onPlaced {
                boundingBox = it.boundsInWindow()
            }
            .onGloballyPositioned {
                boundingBox = it.boundsInWindow()
            }
            .drawWithCache {
                onDrawBehind { drawCircle(c, radius = 15f) }
            },
        onDraw = {
            val x = size.width / 2f
            val dx = x squarePythagoras H
            if (blips.isNotEmpty()) {
                drawIntoCanvas { canvas ->
                    canvas.rotate(progress.toFloat(), x, x)
                    canvas.drawLine(
                        Offset(x, x),
                        Offset(dx, dx),
                        p
                    )
                }
            }
        }
    )
}
