package dev.bg.bikebridge.ui.components

import android.graphics.PointF
import androidx.compose.animation.core.LinearEasing
import androidx.compose.animation.core.animateFloatAsState
import androidx.compose.animation.core.tween
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.size
import androidx.compose.material3.MaterialTheme
import androidx.compose.runtime.Composable
import androidx.compose.runtime.derivedStateOf
import androidx.compose.runtime.getValue
import androidx.compose.runtime.remember
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.drawWithCache
import androidx.compose.ui.draw.drawWithContent
import androidx.compose.ui.graphics.Path
import androidx.compose.ui.graphics.PathMeasure
import androidx.compose.ui.graphics.drawscope.DrawScope
import androidx.compose.ui.graphics.drawscope.Stroke
import androidx.compose.ui.graphics.drawscope.scale
import androidx.compose.ui.graphics.drawscope.translate
import androidx.compose.ui.graphics.graphicsLayer
import androidx.compose.ui.unit.dp
import androidx.core.graphics.plus
import androidx.core.graphics.times
import androidx.graphics.shapes.CornerRounding
import androidx.graphics.shapes.RoundedPolygon
import dev.bg.bikebridge.util.ktx.toPath
import dev.bg.bikebridge.util.ktx.toRadians
import kotlin.math.cos
import kotlin.math.sin

private const val PATH_TIME = 250
private const val ALPHA_TIME = 50
private val SIZE = 96.dp

@Composable
fun Bolt(
    on: Boolean,
    modifier: Modifier
) {
    val c = MaterialTheme.colorScheme.primary

    val vertices = remember {
        val limitY = 0.8f
        val padding = 0.2f
        floatArrayOf(
            radialToCartesian(1f, 280f.toRadians()).x,
            radialToCartesian(1f, 280f.toRadians()).y,
            radialToCartesian(limitY, 180f.toRadians()).x,
            radialToCartesian(padding, 90f.toRadians()).y,
            radialToCartesian(0f, 120f.toRadians()).x,
            radialToCartesian(padding, 90f.toRadians()).y,
            radialToCartesian(padding, 140f.toRadians()).x,
            radialToCartesian(1f, 90f.toRadians()).y,
            radialToCartesian(limitY, 0f.toRadians()).x,
            radialToCartesian(-padding, 90f.toRadians()).y,
            radialToCartesian(0f, 280f.toRadians()).x,
            radialToCartesian(-padding, 90f.toRadians()).y,
        )
    }

    val rounding = remember {
        vertices.map { CornerRounding(0.02f) }.take(vertices.size / 2)
    }

    val polygon = remember(vertices, rounding) {
        RoundedPolygon(
            vertices = vertices,
            perVertexRounding = rounding
        )
    }

    val pathProgress by animateFloatAsState(
        targetValue = if (on) 1f else 0f,
        animationSpec = tween(
            durationMillis = PATH_TIME,
            easing = LinearEasing,
            delayMillis = if (on) 0 else ALPHA_TIME
        ),
        label = "Lightning path progress"
    )
    val alpha by animateFloatAsState(
        targetValue = if (on) 1f else 0f,
        animationSpec = tween(
            ALPHA_TIME,
            easing = LinearEasing,
            delayMillis = if (on) PATH_TIME else 0
        ),
        label = "Lightning alpha"
    )
    val measure = remember { PathMeasure() }

    val roundedPolygonPath = polygon.cubics.toPath()

    val animPath by remember {
        derivedStateOf {
            val np = Path()
            measure.setPath(roundedPolygonPath, false)
            measure.getSegment(
                0f,
                pathProgress * measure.length,
                np
            )
            np
        }
    }

    Box(
        modifier,
        contentAlignment = Alignment.Center
    ) {
        Box(
            modifier = Modifier
                .drawWithCache {
                    onDrawBehind {
                        drawInCenter {
                            drawPath(animPath, color = c, style = Stroke(0.01f))
                        }
                    }
                }
                .size(SIZE)
        )
        Box(
            modifier = Modifier
                .graphicsLayer(alpha = alpha)
                .drawWithContent {
                    drawInCenter {
                        drawPath(roundedPolygonPath, color = c)
                    }
                }
                .size(SIZE)
        )
    }
}

private fun DrawScope.drawInCenter(
    block: DrawScope.() -> Unit
) = scale(size.width * 0.5f, size.width * 0.5f) {
    translate(size.width * 0.5f, size.height * 0.5f) {
        block()
    }
}

private fun vectorPoint(rads: Float) = PointF(cos(rads), sin(rads))

private fun radialToCartesian(
    radius: Float,
    rads: Float,
    center: PointF = PointF(0f, 0f)
) = vectorPoint(rads) * radius + center
