package dev.bg.bikebridge.ui.components

import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.items
import androidx.compose.foundation.lazy.itemsIndexed
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Check
import androidx.compose.material.icons.filled.Edit
import androidx.compose.material.icons.filled.Info
import androidx.compose.material3.AlertDialog
import androidx.compose.material3.Button
import androidx.compose.material3.CircularProgressIndicator
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.HorizontalDivider
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.RadioButton
import androidx.compose.material3.SegmentedButton
import androidx.compose.material3.SegmentedButtonDefaults
import androidx.compose.material3.SingleChoiceSegmentedButtonRow
import androidx.compose.material3.Switch
import androidx.compose.material3.SwitchDefaults
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.unit.dp
import dev.bg.bikebridge.R
import dev.bg.bikebridge.ble.Payload
import dev.bg.bikebridge.util.ktx.toHexString

@Composable
fun <T: Payload> BikeSettingsSwitch(
    title: String,
    options: List<T>,
    checked: Boolean,
    enabled: Boolean,
    waiting: Boolean,
    onChange: (Boolean) -> Unit
) {
    var payloadDialog by remember { mutableStateOf(false) }

    Row(
        Modifier.fillMaxWidth(),
        verticalAlignment = Alignment.CenterVertically,
        horizontalArrangement = Arrangement.SpaceBetween
    ) {
        Row(
            Modifier.weight(1f),
            verticalAlignment = Alignment.CenterVertically,
            horizontalArrangement = Arrangement.spacedBy(4.dp)
        ) {
            Text(title)
            IconButton(onClick = { payloadDialog = true }) {
                Icon(
                    Icons.Filled.Info,
                    contentDescription = stringResource(R.string.payload_info)
                )
            }
        }
        Switch(
            enabled = enabled || !waiting,
            checked = checked,
            onCheckedChange = onChange,
            thumbContent = if (checked) {
                {
                    Icon(
                        Icons.Filled.Check,
                        contentDescription = null,
                        modifier = Modifier.size(SwitchDefaults.IconSize)
                    )
                }
            } else if (waiting) {
                {
                    SmallCircularProgressIndicator(Modifier.fillMaxSize())
                }
            } else null,
            modifier = Modifier
                .padding(horizontal = 4.dp)
                .padding(start = 12.dp)
        )
    }

    PayloadDialog(
        payloadDialog,
        options.map { Pair(it.res, it.payload) }
    ) { payloadDialog = false }
}

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun <T: Payload> BikeSettingsSegmentedButtons(
    title: String,
    options: List<T>,
    selected: T?,
    enabled: Boolean,
    waiting: Boolean,
    onClick: (selected: T) -> Unit
) {
    var payloadDialog by remember { mutableStateOf(false) }

    Row(
        Modifier.fillMaxWidth(),
        verticalAlignment = Alignment.CenterVertically,
        horizontalArrangement = Arrangement.SpaceBetween
    ) {
        Row(
            Modifier.weight(1f),
            verticalAlignment = Alignment.CenterVertically,
            horizontalArrangement = Arrangement.spacedBy(4.dp)
        ) {
            Text(title)
            IconButton(onClick = { payloadDialog = true }) {
                Icon(
                    Icons.Filled.Info,
                    contentDescription = stringResource(R.string.payload_info)
                )
            }
        }
        if (waiting) {
            SmallCircularProgressIndicator(Modifier.size(24.dp))
            Spacer(Modifier.size(4.dp))
        }
        SingleChoiceSegmentedButtonRow {
            options.forEachIndexed { idx, o ->
                SegmentedButton(
                    shape = SegmentedButtonDefaults.itemShape(idx, options.size),
                    enabled = enabled || !waiting,
                    selected = selected == o,
                    onClick = { onClick(o) }
                ) {
                    Text(stringResource(o.res))
                }
            }
        }
    }

    PayloadDialog(
        payloadDialog,
        options.map { Pair(it.res, it.payload) }
    ) { payloadDialog = false }
}

@Composable
fun <T: Payload> BikeSettingsOptionDialog(
    title: String,
    options: List<T>,
    selected: T?,
    waiting: Boolean,
    onSelected: (selected: Payload) -> Unit
) {
    var payloadDialog by remember { mutableStateOf(false) }
    var editDialog by remember { mutableStateOf(false) }
    var internalSelected by remember { mutableStateOf(selected) }

    LaunchedEffect(selected) {
        internalSelected = selected
    }

    Row(
        Modifier.fillMaxWidth(),
        verticalAlignment = Alignment.CenterVertically,
        horizontalArrangement = Arrangement.SpaceBetween
    ) {
        Row(
            Modifier.weight(1f),
            verticalAlignment = Alignment.CenterVertically,
            horizontalArrangement = Arrangement.spacedBy(4.dp)
        ) {
            Text(title)
            IconButton(onClick = { payloadDialog = true }) {
                Icon(
                    Icons.Filled.Info,
                    contentDescription = stringResource(R.string.payload_info)
                )
            }
        }
        if (waiting) {
            SmallCircularProgressIndicator(Modifier.size(24.dp))
            Spacer(Modifier.size(4.dp))
        }
        Row(
            verticalAlignment = Alignment.CenterVertically
        ) {
            Text(stringResource(selected?.res ?: R.string.unknown))
            IconButton(onClick = { editDialog = true }) {
                Icon(
                    Icons.Filled.Edit,
                    contentDescription = stringResource(R.string.edit)
                )
            }
        }
    }

    if (editDialog) {
        AlertDialog(
            onDismissRequest = { editDialog = false },
            dismissButton = {
                Button(onClick = { editDialog = false }) {
                    Text(stringResource(R.string.cancel))
                }
            },
            confirmButton = {
                Button(
                    onClick = {
                        if (internalSelected != null) {
                            editDialog = false
                            onSelected(internalSelected!!)
                        }
                    }
                ) {
                    Text(stringResource(R.string.write))
                }
            },
            title = { Text(title) },
            text = {
                LazyColumn(
                    Modifier.fillMaxWidth(),
                    verticalArrangement = Arrangement.spacedBy(4.dp)
                ) {
                    itemsIndexed(options) { idx, option ->
                        Row(
                            verticalAlignment = Alignment.CenterVertically,
                            horizontalArrangement = Arrangement.SpaceBetween
                        ) {
                            RadioButton(
                                selected = internalSelected.hashCode() == option.hashCode(),
                                onClick = { internalSelected = option }
                            )
                            Text(stringResource(option.res))
                        }
                        if (idx != options.lastIndex) {
                            HorizontalDivider()
                        }
                    }
                }
            }
        )
    }

    PayloadDialog(
        payloadDialog,
        options.map { Pair(it.res, it.payload) }
    ) { payloadDialog = false }
}

@Composable
private fun PayloadDialog(
    visible: Boolean,
    payloadInfo: List<Pair<Int, ByteArray?>>,
    onDismissRequest: () -> Unit
) {
    if (visible) {
        AlertDialog(
            onDismissRequest = onDismissRequest,
            confirmButton = {
                Button(onClick = onDismissRequest) {
                    Text(stringResource(R.string.ok))
                }
            },
            title = { Text(stringResource(R.string.payload_info)) },
            text = {
                LazyColumn(
                    Modifier.fillMaxWidth(),
                    verticalArrangement = Arrangement.spacedBy(4.dp)
                ) {
                    item {
                        Row(
                            Modifier.padding(bottom = 4.dp),
                            horizontalArrangement = Arrangement.SpaceBetween
                        ) {
                            Text(stringResource(R.string.action))
                            Spacer(Modifier.weight(1f))
                            Text(stringResource(R.string.payload))
                        }
                        HorizontalDivider()
                    }
                    items(payloadInfo) {
                        val b = it.second
                        if (b != null) {
                            Row(
                                horizontalArrangement = Arrangement.SpaceBetween
                            ) {
                                MonospaceText(stringResource(it.first))
                                Spacer(Modifier.weight(1f))
                                MonospaceText(
                                    b.toHexString(),
                                    textAlign = TextAlign.End
                                )
                            }
                        }
                    }
                }
            }
        )
    }
}

@Composable
private fun SmallCircularProgressIndicator(
    modifier: Modifier
) {
    CircularProgressIndicator(
        modifier.then(Modifier.padding(4.dp)),
        strokeWidth = (2.5).dp
    )
}
