package dev.bg.bikebridge.ui

import android.Manifest
import android.os.Build
import androidx.compose.animation.AnimatedVisibility
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.displayCutoutPadding
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.systemBarsPadding
import androidx.compose.foundation.layout.waterfallPadding
import androidx.compose.foundation.text.KeyboardOptions
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.automirrored.filled.ArrowBack
import androidx.compose.material.icons.filled.Settings
import androidx.compose.material.icons.filled.Warning
import androidx.compose.material3.AlertDialog
import androidx.compose.material3.Button
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.OutlinedButton
import androidx.compose.material3.Scaffold
import androidx.compose.material3.Text
import androidx.compose.material3.TopAppBar
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.runtime.saveable.rememberSaveable
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.input.KeyboardType
import androidx.compose.ui.unit.dp
import androidx.lifecycle.compose.collectAsStateWithLifecycle
import androidx.navigation.compose.NavHost
import androidx.navigation.compose.composable
import androidx.navigation.compose.currentBackStackEntryAsState
import com.google.accompanist.permissions.ExperimentalPermissionsApi
import com.google.accompanist.permissions.isGranted
import com.google.accompanist.permissions.rememberMultiplePermissionsState
import dev.bg.bikebridge.LocalGattBinder
import dev.bg.bikebridge.LocalNavController
import dev.bg.bikebridge.LocalPasskeyDao
import dev.bg.bikebridge.R
import dev.bg.bikebridge.Route
import dev.bg.bikebridge.ScaffoldVisibility
import dev.bg.bikebridge.ble.BleConstants
import dev.bg.bikebridge.data.db.Passkey
import dev.bg.bikebridge.service.BluetoothGattService
import dev.bg.bikebridge.ui.components.OutlinedTextFieldInputType
import dev.bg.bikebridge.ui.components.TypedOutlinedTextFieldWithError
import dev.bg.bikebridge.ui.screens.bike.BikeScreen
import dev.bg.bikebridge.ui.screens.log.LogScreen
import dev.bg.bikebridge.ui.screens.permissions.EnableScreen
import dev.bg.bikebridge.ui.screens.permissions.PermissionsScreen
import dev.bg.bikebridge.ui.screens.scan.ScanScreen
import dev.bg.bikebridge.ui.screens.settings.SettingsScreen
import dev.bg.bikebridge.util.Constants
import dev.bg.bikebridge.util.Preferences
import kotlinx.coroutines.async
import kotlinx.coroutines.flow.collect
import kotlinx.coroutines.flow.onEach
import kotlinx.coroutines.launch
import timber.log.Timber

@OptIn(
    ExperimentalPermissionsApi::class,
    ExperimentalMaterial3Api::class
)
@Composable
fun Navigation(
    isBluetoothEnabled: Boolean,
    isLocationEnabled: Boolean
) {
    val navController = LocalNavController.current
    val binder = LocalGattBinder.current
    val passkeyDao = LocalPasskeyDao.current

    val scope = rememberCoroutineScope()
    val backStack by navController.currentBackStackEntryAsState()
    val scaffoldVisibility = Route.getScaffoldVisibility(backStack?.destination)

    val logging by Preferences.logEnabledFlow.collectAsStateWithLifecycle()

    val permissionsState = rememberMultiplePermissionsState(
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.S) {
            listOf(
                Manifest.permission.BLUETOOTH,
                Manifest.permission.BLUETOOTH_SCAN,
                Manifest.permission.BLUETOOTH_CONNECT,
                Manifest.permission.BLUETOOTH_ADMIN,
                Manifest.permission.ACCESS_COARSE_LOCATION,
                Manifest.permission.ACCESS_FINE_LOCATION
            )
        } else {
            listOf(
                Manifest.permission.BLUETOOTH,
                Manifest.permission.BLUETOOTH_ADMIN,
                Manifest.permission.ACCESS_COARSE_LOCATION,
                Manifest.permission.ACCESS_FINE_LOCATION
            )
        }
    )
    val permissions = if (Build.VERSION.SDK_INT < Build.VERSION_CODES.Q) {
        permissionsState.permissions
    } else {
        permissionsState
            .permissions
            .filterNot {
                it.permission == Manifest.permission.READ_EXTERNAL_STORAGE ||
                it.permission == Manifest.permission.BLUETOOTH ||
                it.permission == Manifest.permission.BLUETOOTH_ADMIN ||
                it.permission == Manifest.permission.ACCESS_COARSE_LOCATION ||
                it.permission == Manifest.permission.ACCESS_FINE_LOCATION
            }
    }

    var passkeyDialog by rememberSaveable { mutableStateOf(false) }
    var passkeyIncorrect by rememberSaveable { mutableStateOf(false) }

    LaunchedEffect(permissions, isBluetoothEnabled, isLocationEnabled) {
        if (permissions.any { !it.status.isGranted }) {
            navController.navigate(Route.Permissions)
        } else if (isBluetoothEnabled && isLocationEnabled) {
            navController.navigate(Route.Scan)
        } else {
            navController.navigate(Route.Enable)
        }
    }

    LaunchedEffect(Unit, binder) {
        binder?.events?.onEach { event ->
            when (event.action) {
                BluetoothGattService.GATT_DISCONNECTED -> {
                    if (!passkeyIncorrect) {
                        passkeyDialog = false
                        navController.navigate(Route.Scan)
                    }
                }
                BluetoothGattService.GATT_REQUEST_PASSKEY -> {
                    passkeyDialog = true
                }
                BluetoothGattService.GATT_INCORRECT_PASSKEY -> {
                    passkeyDialog = true
                    passkeyIncorrect = true
                }
            }
        }?.collect()
    }

    Scaffold(
        topBar = {
            AnimatedVisibility(scaffoldVisibility != ScaffoldVisibility.Gone) {
                TopAppBar(
                    title = {},
                    navigationIcon = {
                        AnimatedVisibility(scaffoldVisibility == ScaffoldVisibility.Top || scaffoldVisibility == ScaffoldVisibility.NavOnly) {
                            IconButton(
                                onClick = {
                                    if (backStack?.destination == Route.Bike) {
                                        Preferences.setAutoConnectFlow(false)
                                        binder?.disconnectFromDevice()
                                    }
                                    navController.navigateUp()
                                }
                            ) {
                                Icon(
                                    Icons.AutoMirrored.Filled.ArrowBack,
                                    contentDescription = null
                                )
                            }
                        }
                    },
                    actions = {
                        AnimatedVisibility(scaffoldVisibility == ScaffoldVisibility.Top || scaffoldVisibility == ScaffoldVisibility.ActionsOnly) {
                            Row {
                                if (logging) {
                                    IconButton(onClick = { navController.navigate(Route.Log) }) {
                                        Icon(
                                            painterResource(R.drawable.baseline_list_24),
                                            contentDescription = stringResource(R.string.log)
                                        )
                                    }
                                }
                                IconButton(onClick = { navController.navigate(Route.Settings) }) {
                                    Icon(
                                        Icons.Filled.Settings,
                                        contentDescription = stringResource(R.string.settings)
                                    )
                                }
                            }
                        }
                    }
                )
            }
        },
        bottomBar = {}
    ) { padding ->
        NavHost(
            navController,
            startDestination = if (permissions.all { it.status.isGranted }) Route.Scan else Route.Permissions,
            modifier = Modifier
                .fillMaxSize()
                .systemBarsPadding()
                .waterfallPadding()
                .displayCutoutPadding()
                .padding(padding)
        ) {
            composable<Route.Permissions> {
                PermissionsScreen(
                    permissions,
                    permissionsState::launchMultiplePermissionRequest
                )
            }
            composable<Route.Enable> {
                EnableScreen(
                    isBluetoothEnabled,
                    isLocationEnabled
                )
            }
            composable<Route.Scan> {
                ScanScreen()
            }
            composable<Route.Bike> {
                BikeScreen()
            }
            composable<Route.Log> {
                LogScreen()
            }
            composable<Route.Settings> {
                SettingsScreen()
            }
        }
        PasskeyDialog(
            visible = passkeyDialog,
            passkeyIncorrect = passkeyIncorrect,
            onPasskey = { passkey ->
                if (binder != null) {
                    passkeyDialog = false
                    passkeyIncorrect = false
                    val mac = binder.state.value.deviceAddress
                    if (mac != null) {
                        scope.launch {
                            async {
                                passkeyDao.insertPasskey(
                                    Passkey(
                                        mac = mac,
                                        passkey = passkey
                                    )
                                )
                            }.await()
                            if (binder.isConnected()) {
                                binder.entryPoint()
                            } else {
                                navController.navigate(Route.Scan)
                            }
                        }
                    }
                } else {
                    Timber.e("binder was null")
                }
            }
        )
    }
}

@Composable
private fun PasskeyDialog(
    visible: Boolean,
    passkeyIncorrect: Boolean,
    onPasskey: (String) -> Unit
) {
    val navController = LocalNavController.current

    var passkey by remember { mutableStateOf("") }
    var error by remember { mutableStateOf(false) }

    if (visible) {
        AlertDialog(
            onDismissRequest = {},
            dismissButton = {
                OutlinedButton(
                    onClick = {
                        navController.navigate(Route.Scan)
                    }
                ) {
                    Text(stringResource(R.string.cancel))
                }
            },
            confirmButton = {
                Button(
                    onClick = {
                        if (passkey.isNotEmpty()) {
                            onPasskey(passkey)
                        } else {
                            onPasskey(BleConstants.Characteristics.Shimano.DEFAULT_PASSKEY)
                        }
                    },
                    enabled = !error
                ) {
                    Text(stringResource(R.string.confirm))
                }
            },
            title = { Text(stringResource(R.string.passkey)) },
            text = {
                Column(
                    Modifier.fillMaxWidth(),
                    verticalArrangement = Arrangement.spacedBy(12.dp)
                ) {
                    Text(stringResource(R.string.passkey_description))
                    TypedOutlinedTextFieldWithError(
                        label = { Text(stringResource(R.string.passkey)) },
                        value = passkey,
                        onValueChange = {
                            if (it.length <= 6) {
                                passkey = it
                                error = passkey.length != 6 && passkey.isNotEmpty()
                            }
                        },
                        inputType = OutlinedTextFieldInputType.Numeric,
                        keyboardOptions = KeyboardOptions(keyboardType = KeyboardType.Number),
                        placeholder = {
                            Text(BleConstants.Characteristics.Shimano.DEFAULT_PASSKEY)
                        },
                        isError = error,
                        errorText = stringResource(R.string.passkey_error)
                    )
                    if (passkeyIncorrect) {
                        Row(
                            verticalAlignment = Alignment.CenterVertically,
                            horizontalArrangement = Arrangement.spacedBy(8.dp)
                        ) {
                            Icon(
                                Icons.Filled.Warning,
                                contentDescription = null,
                                tint = Constants.Colors.ErrorRed
                            )
                            Text(stringResource(R.string.passkey_incorrect))
                        }
                    }
                }
            }
        )
    }
}
