package dev.bg.bikebridge.service

import android.annotation.SuppressLint
import android.bluetooth.BluetoothAdapter
import android.bluetooth.BluetoothDevice
import android.bluetooth.BluetoothGatt
import android.bluetooth.BluetoothGattCallback
import android.bluetooth.BluetoothGattCharacteristic
import android.bluetooth.BluetoothGattDescriptor
import android.bluetooth.BluetoothManager
import android.bluetooth.BluetoothProfile
import android.bluetooth.le.ScanCallback
import android.bluetooth.le.ScanResult
import android.bluetooth.le.ScanSettings
import android.content.Context
import android.content.Intent
import android.os.Binder
import android.os.Build
import android.os.IBinder
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.LifecycleService
import androidx.lifecycle.lifecycleScope
import androidx.lifecycle.repeatOnLifecycle
import dagger.hilt.android.AndroidEntryPoint
import dev.bg.bikebridge.ble.BleConstants
import dev.bg.bikebridge.ble.BleOp
import dev.bg.bikebridge.ble.EnableNotification
import dev.bg.bikebridge.ble.ReadCharacteristic
import dev.bg.bikebridge.ble.WriteCharacteristic
import dev.bg.bikebridge.data.GattServiceEvent
import dev.bg.bikebridge.data.GattServiceEventAction
import dev.bg.bikebridge.data.ShimanoSettingsNotification
import dev.bg.bikebridge.data.db.Passkey
import dev.bg.bikebridge.data.state.BatteryState
import dev.bg.bikebridge.data.state.GattServiceState
import dev.bg.bikebridge.data.state.ShimanoState
import dev.bg.bikebridge.data.state.SramAxsState
import dev.bg.bikebridge.db.dao.PasskeyDao
import dev.bg.bikebridge.util.BikeUtil
import dev.bg.bikebridge.util.Preferences
import dev.bg.bikebridge.util.ktx.assertivelyGet
import dev.bg.bikebridge.util.ktx.censor
import dev.bg.bikebridge.util.ktx.findByUuid
import dev.bg.bikebridge.util.ktx.getCharacteristics
import dev.bg.bikebridge.util.ktx.getGattTable
import dev.bg.bikebridge.util.ktx.getManufacturerId
import dev.bg.bikebridge.util.ktx.getShortUuid
import dev.bg.bikebridge.util.ktx.isIndicatable
import dev.bg.bikebridge.util.ktx.isNotifiable
import dev.bg.bikebridge.util.ktx.isReadable
import dev.bg.bikebridge.util.ktx.isWritable
import dev.bg.bikebridge.util.ktx.isWritableWithoutResponse
import dev.bg.bikebridge.util.ktx.toHexString
import dev.bg.bikebridge.util.ktx.toShortHex
import dev.bg.bikebridge.util.ktx.toUtf8
import dev.bg.bikebridge.util.ktx.updateEntry
import kotlinx.coroutines.delay
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.SharedFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.update
import kotlinx.coroutines.launch
import timber.log.Timber
import java.util.LinkedList
import java.util.Queue
import java.util.UUID
import javax.inject.Inject
import kotlin.time.Duration.Companion.seconds

@SuppressLint("MissingPermission") // caller verifies
@OptIn(ExperimentalUnsignedTypes::class)
@AndroidEntryPoint
class BluetoothGattService: LifecycleService() {

    @Inject
    lateinit var passkeyDao: PasskeyDao

    private val opQueue: Queue<BleOp> = LinkedList()
    private var bluetoothAdapter: BluetoothAdapter? = null
    private var bluetoothGatt: BluetoothGatt? = null
    private var bikeCharacteristics: List<BluetoothGattCharacteristic> = emptyList()
    private var writeEnabled = Preferences.getBoolean(Preferences.WRITE)
    private var loggingEnabled = Preferences.getBoolean(Preferences.LOGGING)
    private var censorLog = Preferences.getBoolean(Preferences.CENSOR_LOG)
    private var autoConnect = Preferences.getBoolean(Preferences.AUTO_CONNECT)
    private var lastMac = Preferences.getString(Preferences.LAST_MAC)

    private lateinit var authNonce: ByteArray

    private val _state: MutableStateFlow<GattServiceState> = MutableStateFlow(
        GattServiceState(
            scanning = false,
            bikesFound = emptyList(),
            bonded = false,
            rssi = 0,
            writableCharacteristics = emptyList(),
            writableRequiresTrigger = null,
            manufacturer = null,
            deviceName = null,
            deviceAddress = null,
            firmwareVersion = null,
            batteryState = null,
            shimanoState = null,
            sramAxsState = null
        )
    )
    private val _events = MutableSharedFlow<GattServiceEvent>()
    private val _logEntries: MutableStateFlow<List<String>> = MutableStateFlow(emptyList())

    override fun onCreate() {
        super.onCreate()
        Timber.d("GattService onCreate()")
        bluetoothAdapter = (this.getSystemService(Context.BLUETOOTH_SERVICE) as BluetoothManager).adapter

        lifecycleScope.launch {
            lifecycle.repeatOnLifecycle(Lifecycle.State.STARTED) {
                launch {
                    Preferences.writeEnabledFlow.collect { writeEnabled = it }
                }
                launch {
                    Preferences.logEnabledFlow.collect { loggingEnabled = it }
                }
                launch {
                    Preferences.censorLogFlow.collect { censorLog = it }
                }
                launch {
                    Preferences.autoConnectFlow.collect { autoConnect = it }
                }
                launch {
                    Preferences.lastMacFlow.collect { lastMac = it }
                }
            }
        }
    }

    override fun onBind(intent: Intent): IBinder {
        super.onBind(intent)
        return GattBinder()
    }

    override fun onUnbind(intent: Intent?): Boolean {
        disconnectFromDevice()
        return super.onUnbind(intent)
    }

    private fun startScan() {
        if (_state.value.scanning) return
        log("Starting scan")
        _state.update {
            it.copy(
                scanning = true
            )
        }
        val scanSettings = ScanSettings.Builder()
            .setScanMode(ScanSettings.SCAN_MODE_LOW_LATENCY)
            .build()
        lifecycleScope.launch {
            bluetoothAdapter?.bluetoothLeScanner?.startScan(emptyList(), scanSettings, scanCallback)
        }
    }

    private fun stopScan() {
        if (!_state.value.scanning) return
        log("Stopping scan")
        _state.update {
            it.copy(scanning = false)
        }
        lifecycleScope.launch {
            bluetoothAdapter?.bluetoothLeScanner?.stopScan(scanCallback)
        }
    }

    private fun connectToDevice(scanResult: ScanResult) {
        with(scanResult) {
            device.connectGatt(this@BluetoothGattService, false, gattCallback)
            _state.update {
                it.copy(
                    deviceName = device.name,
                    bonded = device.bondState == BluetoothDevice.BOND_BONDED,
                    manufacturer = scanResult.getManufacturerId()
                )
            }
            if (autoConnect) {
                Preferences.setLastMac(device.address)
            }
        }
        lifecycleScope.launch {
            while (true) {
                bluetoothGatt?.readRemoteRssi()
                delay(5.seconds)
            }
        }
    }

    private fun disconnectFromDevice() {
        bikeCharacteristics = emptyList()
        opQueue.clear()
        bluetoothGatt?.disconnect()
        bluetoothGatt = null
        _state.update {
            it.copy(
                bonded = false,
                rssi = 0,
                manufacturer = null,
                deviceName = null,
                firmwareVersion = null,
                batteryState = null,
                shimanoState = null
            )
        }
        lifecycleScope.launch {
            _events.emit(GattServiceEventAction(GATT_DISCONNECTED))
        }
        log("Requested disconnect")
    }

    private fun consumeNextOp() { // TODO do we need this?
        lifecycleScope.launch {
            delay(100L)
            _consumeNextOp()
        }
    }

    @Synchronized
    private fun _consumeNextOp() {
        if (bluetoothGatt == null || opQueue.isEmpty()) return
        val peeked = opQueue.peek()
        val characteristic = peeked?.characteristic
        if (characteristic == null || !bikeHasCharacteristic(characteristic)) return
        when (peeked) {
            is ReadCharacteristic -> {
                bluetoothGatt!!.readCharacteristic(peeked.characteristic)
            }
            is WriteCharacteristic -> {
                // write = 0x12
                // write w/o response = 0x52
                val type = when {
                    characteristic.isWritable() -> BluetoothGattCharacteristic.WRITE_TYPE_DEFAULT
                    characteristic.isWritableWithoutResponse() -> BluetoothGattCharacteristic.WRITE_TYPE_NO_RESPONSE
                    else -> return
                }
                @Suppress("DEPRECATION")
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) {
                    bluetoothGatt!!.writeCharacteristic(
                        characteristic,
                        peeked.payload,
                        type
                    )
                } else {
                    characteristic.writeType = type
                    characteristic.value = peeked.payload
                    bluetoothGatt!!.writeCharacteristic(peeked.characteristic)
                }
                log("tx: ${characteristic.getShortUuid()} - payload: ${peeked.payload.toHexString()}")
            }
            is EnableNotification -> {
                val descriptor = characteristic.getDescriptor(BleConstants.Descriptors.CCCD)
                val payload = when {
                    characteristic.isNotifiable() -> BluetoothGattDescriptor.ENABLE_NOTIFICATION_VALUE
                    characteristic.isIndicatable() -> BluetoothGattDescriptor.ENABLE_INDICATION_VALUE
                    else -> return
                }
                @Suppress("DEPRECATION")
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) {
                    bluetoothGatt!!.writeDescriptor(descriptor, payload)
                } else {
                    bluetoothGatt!!.writeDescriptor(descriptor)
                    characteristic.value = payload
                }
                bluetoothGatt!!.setCharacteristicNotification(peeked.characteristic, true)
                log("Enabling ${if (characteristic.isNotifiable()) "notification" else "indication"} for ${characteristic.getShortUuid()}")
            }
        }
        opQueue.remove()
    }

    private fun txTriggers(
        enqueueOnly: Boolean = false
    ) {
        if (_state.value.writableRequiresTrigger != true) return
        _state.value.writableCharacteristics.forEach { c ->
            if (BleConstants.TRIGGERS.contains(c.uuid)) {
                BleConstants.TRIGGER_PAYLOAD_MAP.assertivelyGet(c.uuid).forEach { payload ->
                    opQueue.add(
                        WriteCharacteristic(
                            characteristic = c,
                            payload = payload
                        )
                    )
                }
            }
        }
        lifecycleScope.launch {
            _events.emit(GattServiceEventAction(GATT_ENQUEUED_TRIGGERS))
        }
        if (!enqueueOnly) {
            consumeNextOp()
        }
    }

    private fun writeCharacteristic(
        uuid: UUID,
        payload: ByteArray
    ) {
        opQueue.add(
            WriteCharacteristic(
                characteristic = _state.value.writableCharacteristics.find { c -> c.uuid == uuid } ?: return,
                payload = payload
            )
        )
        consumeNextOp()
    }

    private fun stateReducer(
        characteristic: BluetoothGattCharacteristic,
        bytes: ByteArray
    ) {
        when (characteristic.uuid) {
            BleConstants.Characteristics.Device.DEVICE_NAME -> {
                log("rx - Device name: ${bytes.toUtf8()}")
                _state.update {
                    it.copy(
                        deviceName = bytes.toUtf8()
                    )
                }
            }
            BleConstants.Characteristics.Device.MANUFACTURER -> {
                log("rx - Manufacturer: ${bytes.toUtf8()}")
            }
            BleConstants.Characteristics.Device.APPEARANCE -> {
                log("rx - Appearance: ${bytes.toHexString()}")
            }
            BleConstants.Characteristics.Device.CONNECTION_PARAMETERS -> {
                log("rx - Connection parameters: ${bytes.toHexString()}")
            }
            BleConstants.Characteristics.Device.SERIAL_NUMBER -> {
                log("rx - Serial number: ${bytes.toUtf8().censor(enabled = censorLog)}")
            }
            BleConstants.Characteristics.Device.FIRMWARE_VERSION -> {
                log("rx - Firmware version: ${bytes.toUtf8()}")
                _state.update {
                    it.copy(
                        firmwareVersion = bytes.toUtf8()
                    )
                }
            }
            BleConstants.Characteristics.Battery.HEALTH_STATUS -> {
                log("rx - Battery health status: ${bytes.toHexString()}")
            }
            BleConstants.Characteristics.Battery.HEALTH_INFO -> {
                log("rx - Battery health info: ${bytes.toHexString()}")
            }
            BleConstants.Characteristics.Battery.INFO -> {
                log("rx - Battery info: ${bytes.toHexString()} ${bytes.toUtf8()}")
                _state.update {
                    it.copy(
                        batteryState = BatteryState(
                            healthStatus = it.batteryState?.healthStatus,
                            healthInfo = it.batteryState?.healthInfo,
                            info = it.batteryState?.info,
                            level = it.batteryState?.level
                        )
                    )
                }
            }
            BleConstants.Characteristics.Battery.LEVEL -> {
                log("rx - Battery level: ${bytes.first().toInt()}%")
                _state.update {
                    it.copy(
                        batteryState = BatteryState(
                            healthStatus = it.batteryState?.healthStatus,
                            healthInfo = it.batteryState?.healthInfo,
                            info = it.batteryState?.info,
                            level = bytes.first().toInt()
                        )
                    )
                }
            }

            // Shimano
            BleConstants.Characteristics.Shimano.AUTH_NONCE -> {
                log("rx - Shimano auth nonce: ${bytes.toHexString()}")
                if (!writeEnabled) return
                authNonce = bytes
                lifecycleScope.launch {
                    val passkey = passkeyDao.getPasskey(_state.value.deviceAddress)
                    if (passkey != null) {
                        Timber.d("Using passkey: $passkey")
                        writeCharacteristic(
                            BleConstants.Characteristics.Shimano.AUTH_CONTROL,
                            BikeUtil.Shimano.getPasskeyAuthPayload(passkey, bytes)
                        )
                        consumeNextOp()
                    } else {
                        log("fatal - missing passkey")
                    }
                }
            }
            BleConstants.Characteristics.Shimano.AUTH_CONTROL -> {
                log("rx - Shimano auth control: ${bytes.toHexString()}")
                if (bytes.contentEquals(BleConstants.Characteristics.Shimano.USER_AUTH_ACCEPTED_PAYLOAD)) {
                    writeCharacteristic(
                        BleConstants.Characteristics.Shimano.AUTH_CONTROL,
                        BikeUtil.Shimano.getAppAuthPayload(authNonce)
                    )
                    consumeNextOp()
                }
                if (bytes.contentEquals(BleConstants.Characteristics.Shimano.APP_AUTH_ACCEPTED_PAYLOAD)) {
                    log("Successfully authenticated")
                    writeCharacteristic(
                        BleConstants.Characteristics.Shimano.FEATURE,
                        BleConstants.Characteristics.Shimano.WRITE_FLAG
                    ) // causes real time updates to stop
                    opQueue.addAll(
                        bikeCharacteristics.filter { it.isReadable() }.filterNot {
                            it.uuid == BleConstants.Characteristics.Shimano.AUTH_NONCE
                        }.map { ReadCharacteristic(it) }
                    )
                    opQueue.addAll(
                        bikeCharacteristics.filter { it.isNotifiable() || it.isIndicatable() }.filterNot {
                            it.uuid == BleConstants.Characteristics.Shimano.AUTH_CONTROL
                        }.map { EnableNotification(it) }
                    )
                    txTriggers(enqueueOnly = true) // TODO add option for this
                    consumeNextOp()
                }
                if (bytes.contentEquals(BleConstants.Characteristics.Shimano.USER_AUTH_FAILED_PAYLOAD)) {
                    log("Wrong passkey")
                    lifecycleScope.launch {
                        _events.emit(GattServiceEventAction(GATT_INCORRECT_PASSKEY))
                    }
                }
            }
            BleConstants.Characteristics.Shimano.STATE -> {
                when (bytes.size) {
                    10 -> {
                        val uBytes = bytes.toUByteArray()
                        _state.update {
                            it.copy(
                                shimanoState = ShimanoState(
                                    mode = BikeUtil.Shimano.getMode(uBytes),
                                    speed = BikeUtil.Shimano.getSpeed(uBytes),
                                    cadence = bytes[5].toInt(),
                                    gear = it.shimanoState?.gear,
                                    totalDistance = it.shimanoState?.totalDistance,
                                    averageSpeed = it.shimanoState?.averageSpeed,
                                    maxSpeed = it.shimanoState?.maxSpeed
                                )
                            )
                        }
                    }
                    17 -> {
                        _state.update {
                            it.copy(
                                shimanoState = ShimanoState(
                                    mode = it.shimanoState?.mode ?: BleConstants.Characteristics.Shimano.Mode.Unknown,
                                    cadence = it.shimanoState?.cadence ?: 0,
                                    speed = it.shimanoState?.speed ?: 0f,
                                    gear = bytes[5].toInt(),
                                    totalDistance = it.shimanoState?.totalDistance,
                                    averageSpeed = it.shimanoState?.averageSpeed,
                                    maxSpeed = it.shimanoState?.maxSpeed
                                )
                            )
                        }
                    }
                    19 -> {
                        val uBytes = bytes.toUByteArray()
                        _state.update {
                            it.copy(
                                shimanoState = ShimanoState(
                                    mode = it.shimanoState?.mode ?: BleConstants.Characteristics.Shimano.Mode.Unknown,
                                    cadence = it.shimanoState?.cadence ?: 0,
                                    speed = it.shimanoState?.speed ?: 0f,
                                    gear = it.shimanoState?.gear,
                                    totalDistance = BikeUtil.Shimano.getTotalDistance(uBytes),
                                    averageSpeed = BikeUtil.Shimano.getAverageSpeed(uBytes),
                                    maxSpeed = BikeUtil.Shimano.getMaxSpeed(uBytes)
                                )
                            )
                        }
                    }
                }
            }
            BleConstants.Characteristics.Shimano.PASSKEY -> {
                val mac = bluetoothGatt?.device?.address
                val passkey = bytes.toUtf8()
                if (mac != null && passkey.length == 6) {
                    log("rx - Passkey: ${passkey.censor(enabled = censorLog)}")
                    lifecycleScope.launch {
                        passkeyDao.insertPasskey(
                            Passkey(
                                mac = mac,
                                passkey = passkey
                            )
                        )
                    }
                }
            }
            BleConstants.Characteristics.Shimano.RESPONSE -> {
                log("rx - Shimano response ${bytes.toHexString()}")
            }
            BleConstants.Characteristics.Shimano.SETTINGS_RESPONSE -> {
                log("rx - Shimano settings ${bytes.toHexString()}")
                // dispatch to UI to render settings updates
                lifecycleScope.launch {
                    _events.emit(ShimanoSettingsNotification(GATT_NOTIFICATION, bytes))
                }
            }

            // SRAM
            BleConstants.Characteristics.SRAM.DERAILLEUR_STATE -> {
                log("rx - SRAM AXS Derailleur State ${bytes.toHexString()}")
                val uBytes = bytes.toUByteArray()
                _state.update {
                    it.copy(
                        sramAxsState = SramAxsState(
                            gear = BikeUtil.SRAM.getGear(uBytes),
                            microAdjust = BikeUtil.SRAM.getMicroAdjust(uBytes)
                        )
                    )
                }
            }
            BleConstants.Characteristics.SRAM.DERAILLEUR_ACTION,
            BleConstants.Characteristics.SRAM.POD_ACTION -> {
                log("rx - SRAM AXS Action ${bytes.toHexString()}")
                if (bytes.contentEquals(BleConstants.Characteristics.SRAM.DERAILLEUR_NOTIFICATION)) {
                    opQueue.add(ReadCharacteristic(bikeCharacteristics.findByUuid(BleConstants.Characteristics.SRAM.DERAILLEUR_STATE)))
                    consumeNextOp()
                }
            }
            else -> {
                log("rx - ${characteristic.uuid}: ${bytes.toHexString()}")
            }
        }
    }

    private fun scrape() {
        opQueue.addAll(bikeCharacteristics.filter { c -> c.isReadable() }.map { c -> ReadCharacteristic(c) })
        val n = bikeCharacteristics.filter { c -> c.isNotifiable() }
        val i = bikeCharacteristics.filter { c -> c.isIndicatable() }
        log("Notifiable UUIDs ${n.size}: ${n.map { u -> u.uuid }}")
        log("Indicatable UUIDs ${i.size}: ${i.map { u -> u.uuid }}")
        opQueue.addAll(n.plus(i).map { c -> EnableNotification(c) })
    }

    // Some bikes require certain opcode order e.g. Shimano has auth handshake
    // Fallback to scraping all values
    private fun entryPoint() {
        if (bluetoothGatt == null) return
        lifecycleScope.launch {
            when (_state.value.manufacturer) {
                BleConstants.Manufacturers.SHIMANO -> {
                    if (writeEnabled) {
                        val passkey = passkeyDao.getPasskey(_state.value.deviceAddress)
                        if (passkey != null) {
                            log("Write mode enabled - starting Shimano authentication handshake...")
                            opQueue.add(EnableNotification(bikeCharacteristics.findByUuid(BleConstants.Characteristics.Shimano.AUTH_CONTROL)))
                            opQueue.add(ReadCharacteristic(bikeCharacteristics.findByUuid(BleConstants.Characteristics.Shimano.AUTH_NONCE)))
                        } else {
                            log("Awaiting user to input passkey...")
                            // prompt user to enter passkey, then recall
                            _events.emit(GattServiceEventAction(GATT_REQUEST_PASSKEY))
                        }
                    } else {
                        scrape()
                    }
                }
                null -> {
                    log("Fatal - unknown manufacturer")
                }
                else -> {
                    log("No known op order - scraping everything")
                    scrape()
                }
            }
            consumeNextOp()
        }
    }

    @OptIn(ExperimentalStdlibApi::class)
    private val gattCallback: BluetoothGattCallback = object : BluetoothGattCallback() {
        override fun onConnectionStateChange(gatt: BluetoothGatt, status: Int, newState: Int) {
            val address = gatt.device.address
            if (status == BluetoothGatt.GATT_SUCCESS) {
                if (newState == BluetoothProfile.STATE_CONNECTED) {
                    log("Connected to ${address.censor(enabled = censorLog)}\nEnumerating device...")
                    this@BluetoothGattService.bluetoothGatt = gatt
                    _state.update {
                        it.copy(
                            deviceAddress = gatt.device.address
                        )
                    }
                    lifecycleScope.launch {
                        _events.emit(GattServiceEventAction(GATT_CONNECTED))
                    }
//                    gatt.requestMtu(512)
                    gatt.discoverServices()
                } else if (newState == BluetoothProfile.STATE_DISCONNECTED) {
                    log("Disconnected from ${address.censor(enabled = censorLog)}")
                    lifecycleScope.launch {
                        _events.emit(GattServiceEventAction(GATT_DISCONNECTED))
                    }
                }
            } else {
                log("Error connecting to ${address.censor(enabled = censorLog)} ${status.toShortHex()}")
                disconnectFromDevice()
            }
        }

        override fun onServiceChanged(gatt: BluetoothGatt) {
            log("Service changed")
            bluetoothGatt = gatt
            gatt.discoverServices()
        }

        override fun onServicesDiscovered(gatt: BluetoothGatt, status: Int) {
            log("Device enumeration finished")
            if (Preferences.getBoolean(Preferences.GATT_TABLE)) {
                log(gatt.getGattTable() ?: "Failed to produce GATT table")
            }
            val characteristics = gatt.getCharacteristics()
            bikeCharacteristics = characteristics
            val writableCharacteristics = characteristics.filter { c -> c.isWritable() || c.isWritableWithoutResponse() }
            _state.update {
                it.copy(
                    writableCharacteristics = writableCharacteristics,
                    writableRequiresTrigger = writableCharacteristics.any { c -> BleConstants.TRIGGERS.contains(c.uuid) }
                )
            }
            entryPoint()
        }

        override fun onCharacteristicRead(
            gatt: BluetoothGatt,
            characteristic: BluetoothGattCharacteristic,
            bytes: ByteArray,
            status: Int
        ) {
            stateReducer(characteristic, bytes)
            consumeNextOp()
        }

        override fun onCharacteristicChanged(
            gatt: BluetoothGatt,
            characteristic: BluetoothGattCharacteristic,
            bytes: ByteArray
        ) {
            stateReducer(characteristic, bytes)
        }

        override fun onCharacteristicWrite(
            gatt: BluetoothGatt?,
            characteristic: BluetoothGattCharacteristic?,
            status: Int
        ) {
            when (status) {
                BluetoothGatt.GATT_SUCCESS -> {
                    log("tx - ACK")
                }
                BluetoothGatt.GATT_INVALID_ATTRIBUTE_LENGTH -> {
                    log("tx - failed as packet size > mtu")
                }
                BluetoothGatt.GATT_WRITE_NOT_PERMITTED -> {
                    log("tx - write not permitted")
                }
                0x80 -> {
                    log("tx - failed (reason: no resources)")
                }
                0x85 -> {
                    log("tx - failed (reason: generic)")
                }
                else -> {
                    log("tx - failed (reason: $status (0x${status.toHexString()}))")
                }
            }
            consumeNextOp()
        }

        override fun onDescriptorWrite(
            gatt: BluetoothGatt?,
            descriptor: BluetoothGattDescriptor?,
            status: Int
        ) {
            log("wrote descriptor - ${if (status == 0) "ACK" else "failed"}")
            consumeNextOp()
        }

        override fun onReadRemoteRssi(gatt: BluetoothGatt?, rssi: Int, status: Int) {
            _state.update {
                it.copy(rssi = rssi)
            }
        }

        override fun onMtuChanged(gatt: BluetoothGatt, mtu: Int, status: Int) {
            log("MTU: $mtu")
        }
    }

    private val scanCallback: ScanCallback = object : ScanCallback() {
        override fun onScanResult(callbackType: Int, result: ScanResult) {
            super.onScanResult(callbackType, result)
            if (autoConnect && result.device.address == lastMac) {
                connectToDevice(result)
            }
            if (BikeUtil.isBike(result.getManufacturerId() ?: -1)) {
                val idx = _state.value.bikesFound.indexOfFirst { b -> b.device.address == result.device.address }
                _state.update {
                    if (idx != -1) {
                        it.copy(
                            bikesFound = it.bikesFound.updateEntry(result)
                        )
                    } else {
                        log("Found bike ${result.scanRecord?.deviceName}")
                        it.copy(
                            bikesFound = it.bikesFound.plus(result)
                        )
                    }
                }
            }
        }

        override fun onScanFailed(errorCode: Int) {
            super.onScanFailed(errorCode)
            log("Scan failed: $errorCode")
        }
    }

    private fun bikeHasCharacteristic(c: BluetoothGattCharacteristic): Boolean {
        return if (bikeCharacteristics.find { it == c } != null) {
            true
        } else {
            log("Bike doesn't support ${c.getShortUuid()}")
            false
        }
    }

    private fun log(message: String) {
        if (loggingEnabled) {
            _logEntries.update {
                it.toMutableList().apply {
                    add(message)
                }
            }
        }
        Timber.d(message)
    }

    inner class GattBinder: Binder() {
        val state: StateFlow<GattServiceState>
            get() = _state.asStateFlow()
        val events: SharedFlow<GattServiceEvent>
            get() = _events
        val logEntries: StateFlow<List<String>>
            get() = _logEntries.asStateFlow()
        fun startScan() = this@BluetoothGattService.startScan()
        fun stopScan() = this@BluetoothGattService.stopScan()
        fun connectToDevice(
            scanResult: ScanResult
        ) = this@BluetoothGattService.connectToDevice(scanResult)
        fun disconnectFromDevice() = this@BluetoothGattService.disconnectFromDevice()
        fun isConnected() = bluetoothGatt != null
        fun txTriggers() = this@BluetoothGattService.txTriggers()
        fun writeCharacteristic(
            uuid: UUID,
            payload: ByteArray
        ) = this@BluetoothGattService.writeCharacteristic(uuid, payload)
        fun clearLog() = _logEntries.update { emptyList() }
        fun entryPoint() = this@BluetoothGattService.entryPoint()
    }

    companion object {
        const val GATT_CONNECTED = "dev.bg.bikebridge.gatt.CONNECTED"
        const val GATT_DISCONNECTED = "dev.bg.bikebridge.gatt.DISCONNECTED"
        const val GATT_NOTIFICATION = "dev.bg.bikebridge.gatt.NOTIFICATION"
        const val GATT_REQUEST_PASSKEY = "dev.bg.bikebridge.gatt.REQUEST_PASSKEY"
        const val GATT_INCORRECT_PASSKEY = "dev.bg.bikebridge.gatt.INCORRECT_PASSKEY"
        const val GATT_ENQUEUED_TRIGGERS = "dev.bg.bikebridge.gatt.ENQUEUED_TRIGGERS"
    }

}
