package dev.bg.bikebridge.ble

import androidx.annotation.StringRes
import dev.bg.bikebridge.R
import dev.bg.bikebridge.util.byteArrayOfInts
import java.util.UUID

// Payloads are little endian
sealed class Payload(
    @StringRes open val res: Int,
    open val payload: ByteArray?
)

object BleConstants {
    // Assigned Numbers from Bluetooth
    object Manufacturers {
        const val BOSCH = 0x02A6
        const val BOSCH_2 = 0xFE02
        const val BROSE = 0x0C45
        const val DJI = 0x08AA
        const val FAZUA = 0x0B06
        const val SHIMANO = 0x044A
        const val SRAM = 0x0933
        const val TREK = 0x0B86
        const val YAMAHA = 0x0CA1
    }

    object Characteristics {
        object Device {
            val DEVICE_NAME = 0x2A00.getBluetoothUUID()
            val APPEARANCE = 0x2A01.getBluetoothUUID()
            val CONNECTION_PARAMETERS = 0x2A04.getBluetoothUUID()
            val SERIAL_NUMBER = 0x2A25.getBluetoothUUID()
            val FIRMWARE_VERSION = 0x2A26.getBluetoothUUID()
            val MANUFACTURER = 0x2A29.getBluetoothUUID()
        }

        object Battery {
            val HEALTH_STATUS = 0x2BEA.getBluetoothUUID()
            val HEALTH_INFO = 0x2BEB.getBluetoothUUID()
            val INFO = 0x2BEC.getBluetoothUUID()
            val LEVEL = 0x2A19.getBluetoothUUID()
        }

        object Shimano {
            val STATE = 0x2AC1.getShimanoBluetoothUUID()
            val SETTINGS_COMMAND = 0x2AFA.getShimanoBluetoothUUID()
            val AUTH_CONTROL = 0x2AF3.getShimanoBluetoothUUID()
            val AUTH_NONCE = 0x2AF4.getShimanoBluetoothUUID()
            val PASSKEY = 0x2AF8.getShimanoBluetoothUUID()
            val SETTINGS_RESPONSE = 0x2AF9.getShimanoBluetoothUUID()
            val FEATURE = 0x2AFF.getShimanoBluetoothUUID() // [0xff, 0x00] enables write mode & disables status updates
            val RESPONSE = 0x2AFD.getShimanoBluetoothUUID()

            val WRITE_FLAG = byteArrayOfInts(0xFF, 0x00)
            val USER_AUTH_ACCEPTED_PAYLOAD = byteArrayOfInts(0x10, 0x01, 0x01)
            val USER_AUTH_FAILED_PAYLOAD = byteArrayOfInts(0x10, 0x01, 0x02)
            val APP_AUTH_ACCEPTED_PAYLOAD = byteArrayOfInts(0x10, 0x02, 0x01)
            const val DEFAULT_PASSKEY = "000000"

            private val DISPLAY_LANGUAGE_BASE_PAYLOAD = byteArrayOfInts(0x00, 0x13, 0x05, 0x20)

            sealed class Mode(
                @StringRes override val res: Int,
                override val payload: ByteArray?
            ): Payload(res, payload) {
                data object Off: Mode(R.string.off_adj, null)
                data object Eco: Mode(R.string.mode_eco, null)
                data object Trail: Mode(R.string.mode_trail, null)
                data object Boost: Mode(R.string.mode_boost, null)
                data object Walk: Mode(R.string.mode_walk, null)
                data object Unknown: Mode(R.string.unknown, null)
            }

            sealed class Beep(
                @StringRes override val res: Int,
                override val payload: ByteArray?
            ): Payload(res, payload) {
                data object Off: Beep(R.string.off_adj, byteArrayOfInts(0x00, 0x13, 0x23, 0x00, 0x00))
                data object On: Beep(R.string.on_adj, byteArrayOfInts(0x00, 0x13, 0x23, 0x00, 0x01))
                data object Unknown: Beep(R.string.unknown, null)
            }

            sealed class DisplayColor(
                @StringRes override val res: Int,
                override val payload: ByteArray?
            ): Payload(res, payload) {
                data object Black: DisplayColor(R.string.black, byteArrayOfInts(0x00, 0x13, 0x05, 0x60, 0x00))
                data object White: DisplayColor(R.string.white, byteArrayOfInts(0x00, 0x13, 0x05, 0x60, 0x01))
                data object Unknown: DisplayColor(R.string.unknown, null)
            }

            sealed class DisplayLanguage(
                @StringRes override val res: Int,
                override val payload: ByteArray?
            ): Payload(res, payload) {
                data object English: DisplayLanguage(R.string.english, DISPLAY_LANGUAGE_BASE_PAYLOAD + byteArrayOfInts(0x00))
                data object French: DisplayLanguage(R.string.french, DISPLAY_LANGUAGE_BASE_PAYLOAD + byteArrayOfInts(0x01))
                data object German: DisplayLanguage(R.string.german, DISPLAY_LANGUAGE_BASE_PAYLOAD + byteArrayOfInts(0x02))
                data object Dutch: DisplayLanguage(R.string.dutch, DISPLAY_LANGUAGE_BASE_PAYLOAD + byteArrayOfInts(0x03))
                data object Italian: DisplayLanguage(R.string.italian, DISPLAY_LANGUAGE_BASE_PAYLOAD + byteArrayOfInts(0x04))
                data object Spanish: DisplayLanguage(R.string.spanish, DISPLAY_LANGUAGE_BASE_PAYLOAD + byteArrayOfInts(0x05))
                data object Unknown: DisplayLanguage(R.string.unknown, null)
            }

            sealed class DisplayUnits(
                @StringRes override val res: Int,
                override val payload: ByteArray?
            ): Payload(res, payload) {
                data object Kmh: DisplayUnits(R.string.kmh, byteArrayOfInts(0x00, 0x13, 0x05, 0x18, 0x00))
                data object Mph: DisplayUnits(R.string.mph, byteArrayOfInts(0x00, 0x13, 0x05, 0x18, 0x01))
                data object Unknown: DisplayUnits(R.string.unknown, null)
            }

            object Write {
                object Settings {
                    // beep
                    val BEEP_FILTER = byteArrayOfInts(0x33, 0x23, 0x02)
                    val BEEP_TRIGGER_FILTER = byteArrayOfInts(0x33, 0x23, 0x06)
                    val BEEP_TRIGGER_PAYLOAD = byteArrayOfInts(0x00, 0x13, 0x23, 0x04, 0x00)

                    // display color
                    val DISPLAY_COLOR_FILTER = byteArrayOfInts(0x33, 0x05, 0x62)
                    val DISPLAY_COLOR_TRIGGER_FILTER = byteArrayOfInts(0x33, 0x05, 0x66)
                    val DISPLAY_COLOR_TRIGGER_PAYLOAD = byteArrayOfInts(0x00, 0x13, 0x05, 0x64, 0x00)

                    // display units (has no trigger)
                    val DISPLAY_UNIT_FILTER = byteArrayOfInts(0x33, 0x05, 0x1A)

                    // display language
                    val DISPLAY_LANGUAGE_FILTER = byteArrayOfInts(0x33, 0x05, 0x22)
                    val DISPLAY_LANGUAGE_TRIGGER_FILTER = byteArrayOfInts(0x33, 0x05, 0x26)
                    val DISPLAY_LANGUAGE_TRIGGER_PAYLOAD = byteArrayOfInts(0x00, 0x13, 0x05, 0x24, 0x00)

                    // SeriesNo

                    // !!!
                    // DUUnitDataLink

                    // enums: BackLightMode, BuzzerMode, DisplayFontColor, DisplayLanguage, DistanceUnit, AutoGearChangeMode, DccPrmProfileAssistMode, DccPrmProfileNo, DestinationType
                }
            }
        }

        object SRAM {
            val DERAILLEUR_STATE = 0xD9050009.getSramBluetoothUUID()
            val DERAILLEUR_ACTION = 0xD905000B.getSramBluetoothUUID()
            val POD_ACTION = 0xD9050054.getSramBluetoothUUID()

            val DERAILLEUR_NOTIFICATION = byteArrayOfInts(0xFF)
        }
    }

    object Descriptors {
        val CCCD = 0x2902.getBluetoothUUID()
    }

    val TRIGGER_PAYLOAD_MAP = mapOf(
        Characteristics.Shimano.SETTINGS_COMMAND to listOf(
            Characteristics.Shimano.Write.Settings.BEEP_TRIGGER_PAYLOAD,
            Characteristics.Shimano.Write.Settings.DISPLAY_COLOR_TRIGGER_PAYLOAD,
            Characteristics.Shimano.Write.Settings.DISPLAY_LANGUAGE_TRIGGER_PAYLOAD
        )
    )
    val TRIGGERS = TRIGGER_PAYLOAD_MAP.keys
}

private fun Int.getBluetoothUUID(): UUID {
    return UUID.fromString("${"%08X".format(this)}-0000-1000-8000-00805f9b34fb")
}

private fun Int.getShimanoBluetoothUUID(): UUID {
    return UUID.fromString("${"%08X".format(this)}-5348-494d-414e-4f5f424c4500")
}

private fun Long.getSramBluetoothUUID(): UUID {
    return UUID.fromString("${"%08X".format(this)}-90aa-4c7c-b036-1e01fb8eb7ee")
}
