package dev.bg.bikebridge

import android.bluetooth.BluetoothAdapter
import android.content.BroadcastReceiver
import android.content.ComponentName
import android.content.Context
import android.content.Intent
import android.content.IntentFilter
import android.content.ServiceConnection
import android.location.LocationManager
import android.os.Build
import android.os.Bundle
import android.os.IBinder
import androidx.activity.ComponentActivity
import androidx.activity.compose.setContent
import androidx.compose.runtime.CompositionLocalProvider
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.setValue
import androidx.compose.runtime.staticCompositionLocalOf
import androidx.core.content.ContextCompat
import androidx.navigation.NavHostController
import androidx.navigation.compose.rememberNavController
import dagger.hilt.android.AndroidEntryPoint
import dev.bg.bikebridge.db.dao.PasskeyDao
import dev.bg.bikebridge.service.BluetoothGattService
import dev.bg.bikebridge.ui.Navigation
import dev.bg.bikebridge.ui.theme.BikeBridgeTheme
import dev.bg.bikebridge.util.ktx.intent
import dev.bg.bikebridge.util.ktx.isBluetoothEnabled
import dev.bg.bikebridge.util.ktx.isLocationEnabled
import timber.log.Timber
import javax.inject.Inject

val LocalNavController = staticCompositionLocalOf<NavHostController> { error("No navcontroller") }
val LocalGattBinder = staticCompositionLocalOf<BluetoothGattService.GattBinder?> { error("No binder") }
val LocalPasskeyDao = staticCompositionLocalOf<PasskeyDao> { error("No passkey dao") }

@AndroidEntryPoint
class MainActivity: ComponentActivity() {

    @Inject
    lateinit var passkeyDao: PasskeyDao

    private var isBluetoothEnabled by mutableStateOf(false)
    private var isLocationEnabled by mutableStateOf(false)
    private var binder by mutableStateOf<BluetoothGattService.GattBinder?>(null)
    private val serviceConnection = object : ServiceConnection {
        override fun onServiceConnected(name: ComponentName?, service: IBinder?) {
            if (service is BluetoothGattService.GattBinder) {
                Timber.d("Bound service")
                binder = service
            }
        }

        override fun onServiceDisconnected(name: ComponentName?) {
            binder = null
        }
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)

        bindService(intent<BluetoothGattService>(), serviceConnection, Context.BIND_AUTO_CREATE)

        isBluetoothEnabled = this.isBluetoothEnabled()
        isLocationEnabled = this.isLocationEnabled()

        ContextCompat.registerReceiver(
            this,
            bluetoothReceiver,
            IntentFilter(BluetoothAdapter.ACTION_STATE_CHANGED),
            ContextCompat.RECEIVER_NOT_EXPORTED
        )
        ContextCompat.registerReceiver(
            this,
            locationStateReceiver,
            IntentFilter(LocationManager.MODE_CHANGED_ACTION),
            ContextCompat.RECEIVER_NOT_EXPORTED
        )

        setContent {
            BikeBridgeTheme {
                val navController = rememberNavController()

                CompositionLocalProvider(
                    LocalNavController provides navController,
                    LocalGattBinder provides binder,
                    LocalPasskeyDao provides passkeyDao
                ) {
                    Navigation(
                        isBluetoothEnabled,
                        isLocationEnabled
                    )
                }
            }
        }
    }

    override fun onResume() {
        super.onResume()

        isBluetoothEnabled = this.isBluetoothEnabled()
        isLocationEnabled = this.isLocationEnabled()
    }

    override fun onDestroy() {
        super.onDestroy()

        binder?.disconnectFromDevice()

        unbindService(serviceConnection)
        unregisterReceiver(bluetoothReceiver)
        unregisterReceiver(locationStateReceiver)
    }

    private val bluetoothReceiver = object : BroadcastReceiver() {
        override fun onReceive(context: Context, intent: Intent) {
            if (intent.action == BluetoothAdapter.ACTION_STATE_CHANGED) {
                isBluetoothEnabled = intent.getIntExtra(
                    BluetoothAdapter.EXTRA_STATE,
                    BluetoothAdapter.ERROR
                ) == BluetoothAdapter.STATE_ON
            }
        }
    }

    private val locationStateReceiver = object : BroadcastReceiver() {
        override fun onReceive(context: Context, intent: Intent) {
            if (intent.action == LocationManager.MODE_CHANGED_ACTION) {
                isLocationEnabled = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.R) {
                    intent.getBooleanExtra(LocationManager.EXTRA_LOCATION_ENABLED, false)
                } else {
                    isLocationEnabled()
                }
            }
        }
    }

}
