var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
import * as fs from "fs/promises";
import path from "path";
import fetch from "node-fetch";
import { logError } from "./log.js";
export class LocalFs {
    constructor(baseDir) {
        this.baseDir = baseDir;
    }
    readFile(filePath) {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                return yield fs.readFile(path.join(this.baseDir, filePath), "utf8");
            }
            catch (_a) { }
        });
    }
    readdir() {
        return __awaiter(this, void 0, void 0, function* () {
            return yield fs.readdir(this.baseDir);
        });
    }
}
export function resolveRemoteFs(repoUrl) {
    if (repoUrl.startsWith("https://github.com/")) {
        return new GitHubFs(repoUrl);
    }
    if (repoUrl.startsWith("https://gitlab.com/")) {
        return new GitLabFs(repoUrl);
    }
    logError(`unsupported repository url \"${repoUrl}\"`);
    return new NullFs();
}
class NullFs {
    readFile(_path) {
        return __awaiter(this, void 0, void 0, function* () {
            return undefined;
        });
    }
    readdir() {
        return __awaiter(this, void 0, void 0, function* () {
            return [];
        });
    }
}
class GitHubFs {
    constructor(url) {
        this.readFilePrefix =
            url.replace("github.com", "raw.githubusercontent.com") + "/HEAD/";
        this.readdirUrl =
            url.replace("https://github.com/", "https://api.github.com/repos/") +
                "/contents";
    }
    readFile(path) {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                const response = yield fetch(this.readFilePrefix + path);
                if (response.status == 200) {
                    return response.text();
                }
            }
            catch (_a) { }
        });
    }
    readdir() {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                const response = yield fetch(this.readdirUrl);
                if (!response.ok) {
                    logError(`"${this.readdirUrl}" responded with ${response.status}:\n` +
                        (yield response.text()));
                    return [];
                }
                const contents = (yield response.json());
                return contents.map((file) => file.name);
            }
            catch (_a) {
                return [];
            }
        });
    }
}
class GitLabFs {
    constructor(url) {
        this.readFilePrefix = url + "/-/raw/master/";
        this.readdirUrl =
            "https://gitlab.com/api/v4/projects/" +
                encodeURIComponent(url.slice("https://gitlab.com/".length)) +
                "/repository/tree?per_page=100";
    }
    readFile(path) {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                const response = yield fetch(this.readFilePrefix + path);
                if (response.status == 200) {
                    return response.text();
                }
            }
            catch (_a) { }
        });
    }
    readdir() {
        return __awaiter(this, void 0, void 0, function* () {
            const results = [];
            let nextUrl = this.readdirUrl;
            while (nextUrl) {
                try {
                    const response = yield fetch(nextUrl);
                    if (!response.ok) {
                        logError(`"${nextUrl}" responded with ${response.status}:\n` +
                            (yield response.text()));
                        break;
                    }
                    const linkHeader = response.headers.get("link");
                    if (linkHeader) {
                        // find the url relevant to the next page
                        nextUrl = linkHeader
                            .split(", ")
                            .find((link) => link.endsWith('rel="next"'));
                        // get the link within <>
                        nextUrl = nextUrl === null || nextUrl === void 0 ? void 0 : nextUrl.slice(1, nextUrl.indexOf(">"));
                    }
                    else {
                        nextUrl = undefined;
                    }
                    const contents = (yield response.json());
                    for (const content of contents) {
                        results.push(content.name);
                    }
                }
                catch (_a) {
                    break;
                }
            }
            return results;
        });
    }
}
