var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
import * as fs from "fs/promises";
import resolveExpression, { mergeExpressions } from "./resolve-expression.js";
import { ripMarkdownLicense } from "./rip-markdown-license.js";
import spdxCorrect from "spdx-correct";
import { LocalFs, resolveRemoteFs } from "./fs.js";
import normalizePackageRepo from "./normalize-package-repo.js";
import resolveMetaLicenseExpression from "./normalize-package-license.js";
import { cacheResult, licenseFromCache } from "./cache.js";
import loadForcedLicenses from "./load-forced-licenses.js";
export function ripOne(packagePath, options) {
    return __awaiter(this, void 0, void 0, function* () {
        var _a;
        const packageJSON = yield tryDisk(packagePath + `/package.json`);
        if (!packageJSON) {
            // not a package
            return;
        }
        const packageMeta = JSON.parse(packageJSON);
        const override = (_a = options === null || options === void 0 ? void 0 : options.overrides) === null || _a === void 0 ? void 0 : _a[packageMeta.name];
        let licenses = [];
        if (override && override.licenses) {
            licenses = yield loadForcedLicenses(override.licenses);
        }
        if (licenses.length == 0) {
            licenses = yield findLicenseText(packagePath, packageMeta, options);
        }
        let licenseExpression = (override === null || override === void 0 ? void 0 : override.licenseExpression) || resolveMetaLicenseExpression(packageMeta);
        if (licenseExpression) {
            licenseExpression =
                spdxCorrect(licenseExpression, { upgrade: false }) || licenseExpression;
        }
        else if (licenses.length > 0) {
            licenseExpression = mergeExpressions(licenses) + "*";
        }
        const output = {
            name: packageMeta.name,
            version: packageMeta.version,
            path: packagePath,
            licenseExpression: licenseExpression || "UNKNOWN",
            licenses,
        };
        if (!options) {
            return output;
        }
        if (options.includeHomepage) {
            output.homepage = packageMeta.homepage;
        }
        if (options.includeRepository) {
            output.repository = normalizePackageRepo(packageMeta);
        }
        if (packageMeta.funding && options.includeFunding) {
            const funding = Array.isArray(packageMeta.funding)
                ? packageMeta.funding
                : [packageMeta.funding];
            output.funding = funding.map((info) => typeof info == "string" ? info : info.url);
        }
        if (packageMeta.description && options.includeDescription) {
            output.description = packageMeta.description;
        }
        return output;
    });
}
function findLicenseText(baseDir, packageMeta, options) {
    return __awaiter(this, void 0, void 0, function* () {
        // try grabbing the license from local files
        const localResult = yield licenseFromFolder(new LocalFs(baseDir));
        if (localResult.some((license) => license.source != "readme" || license.expression != "UNKNOWN")) {
            // stop early if we have a valid result
            return localResult;
        }
        // try grabbing the license from the repo
        const repoUrl = normalizePackageRepo(packageMeta);
        if (repoUrl) {
            const encodedRepoUrl = encodeURIComponent(repoUrl);
            const remoteFs = resolveRemoteFs(repoUrl);
            const cachedResult = yield licenseFromCache(encodedRepoUrl, options);
            if (cachedResult) {
                return cachedResult;
            }
            const remoteResult = yield licenseFromFolder(remoteFs);
            if (remoteResult.length > 0) {
                cacheResult(encodedRepoUrl, remoteResult, options);
                return remoteResult;
            }
            if (localResult.length > 0) {
                // cache our local result to reduce API usage
                cacheResult(encodedRepoUrl, localResult, options);
            }
        }
        return localResult;
    });
}
function licenseFromFolder(fs) {
    return __awaiter(this, void 0, void 0, function* () {
        let noticeList = [];
        let licenseList = [];
        let readmeLicense;
        for (const entry of yield fs.readdir()) {
            const lowercaseName = entry.toLowerCase();
            // test as apache notice file
            if (lowercaseName.includes("notice")) {
                const text = yield fs.readFile(entry);
                noticeList.push(text);
                continue;
            }
            // test as license file
            const isLicense = lowercaseName.includes("license") ||
                lowercaseName.includes("licence") ||
                lowercaseName.includes("copying");
            if (isLicense) {
                // append the license file
                const text = yield fs.readFile(entry);
                licenseList.push(text);
                continue;
            }
            // test as readme file
            if (lowercaseName.startsWith("readme")) {
                // overwrite the readme license text
                const text = yield fs.readFile(entry);
                readmeLicense = ripMarkdownLicense(text);
            }
        }
        const resolved = [];
        for (const text of noticeList) {
            resolved.push({
                source: "notice",
                text: text,
            });
        }
        for (const text of licenseList) {
            resolved.push({
                expression: resolveExpression(text),
                source: "license",
                text: text,
            });
        }
        if (readmeLicense) {
            resolved.push({
                expression: resolveExpression(readmeLicense),
                source: "readme",
                text: readmeLicense,
            });
        }
        return resolved;
    });
}
function tryDisk(path) {
    return __awaiter(this, void 0, void 0, function* () {
        try {
            return yield fs.readFile(path, "utf8");
        }
        catch (_a) { }
    });
}
