var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
import * as fs from "node:fs/promises";
import child_process from "node:child_process";
import { promisify } from "node:util";
import path from "path";
import { ripOne, } from "./rip-license.js";
import resolveExpression, { mergeExpressions } from "./resolve-expression.js";
import { getDefaultCacheFolder } from "./cache.js";
import { logError } from "./log.js";
import loadForcedLicenses from "./load-forced-licenses.js";
const exec = promisify(child_process.exec);
export { ripOne, resolveExpression as resolveLicenseExpression, getDefaultCacheFolder, };
export function ripAll(projectRoot, options) {
    return __awaiter(this, void 0, void 0, function* () {
        const resolved = [];
        const errors = {
            missingLicenseText: [],
            invalidLicense: [],
        };
        if (!options) {
            options = {};
        }
        else {
            options = Object.assign({}, options);
        }
        if (!options.cacheFolder) {
            options.cacheFolder = getDefaultCacheFolder(projectRoot);
        }
        const resolvedMap = {};
        for (const packagePath of yield packageFolders(projectRoot, options)) {
            const data = yield ripOne(packagePath, options);
            if (!data) {
                continue;
            }
            if (!isNameAccepted(data.name, options)) {
                continue;
            }
            if (data.licenseExpression.includes("UNKNOWN")) {
                errors.invalidLicense.push(data.name);
            }
            if (data.licenses.length == 0) {
                errors.missingLicenseText.push(data.name);
            }
            // reducing duplicates by only storing the latest version when licenses are exactly the same
            let existing = resolvedMap[data.name];
            if (!existing) {
                existing = [];
                resolvedMap[data.name] = existing;
            }
            const version = data.version.split(".").map((v) => parseInt(v));
            const match = existing.find((oldData) => 
            // same license count
            data.licenses.length == oldData.data.licenses.length &&
                // every license has an identical match
                data.licenses.every((license) => oldData.data.licenses.some((oldLicense) => oldLicense.text == license.text)));
            if (!match) {
                // add a new entry
                existing.push({ data, version });
                resolved.push(data);
            }
            else if (isVersionNewer(version, match.version)) {
                // overwrite existing data
                Object.assign(match.data, data);
                match.version = version;
            }
        }
        if (options.append) {
            for (const template of options.append) {
                const forcedPackage = {
                    name: template.name,
                    version: template.version || "",
                    path: template.path || "",
                    licenseExpression: template.licenseExpression,
                    licenses: yield loadForcedLicenses(template.licenses),
                    homepage: template.homepage,
                    repository: template.repository,
                    funding: template.funding,
                    description: template.description,
                };
                if (!forcedPackage.licenseExpression) {
                    forcedPackage.licenseExpression = mergeExpressions(forcedPackage.licenses);
                }
                resolved.push(forcedPackage);
            }
        }
        return { resolved, errors };
    });
}
function packageFolders(projectRoot, options) {
    return __awaiter(this, void 0, void 0, function* () {
        try {
            const lockPath = path.join(projectRoot, "package-lock.json");
            const npmLockJson = yield fs.readFile(lockPath, "utf8");
            return yield packageFoldersNpm(projectRoot, npmLockJson, options);
        }
        catch (_a) {
            // file not found, not using npm
        }
        try {
            return yield packageFoldersPnpm(projectRoot, options);
        }
        catch (_b) {
            // command failed, not using pnpm
        }
        if (!options.includeDev) {
            return yield packageFoldersFallbackNoDev(projectRoot, options);
        }
        return yield packageFoldersFallback(projectRoot, options);
    });
}
function packageFoldersNpm(projectRoot, npmLockJson, options) {
    return __awaiter(this, void 0, void 0, function* () {
        let npmLock;
        try {
            npmLock = JSON.parse(npmLockJson);
        }
        catch (_a) {
            logError("failed to parse npm lock file");
            return [];
        }
        const folders = [];
        for (const packagePath in npmLock.packages) {
            if (packagePath == "") {
                continue;
            }
            const packageData = npmLock.packages[packagePath];
            if (!options.includeDev && packageData.dev) {
                continue;
            }
            const name = packageData.name ||
                packagePath.slice(packagePath.lastIndexOf("node_modules") + 13);
            if (!isNameAccepted(name, options)) {
                continue;
            }
            folders.push(path.join(projectRoot, packagePath));
        }
        return folders;
    });
}
function packageFoldersPnpm(projectFolder, options) {
    return __awaiter(this, void 0, void 0, function* () {
        const filterFlag = options.includeDev ? "" : "--prod";
        const command = `pnpm licenses ls ${filterFlag} --json`;
        const { stdout, stderr } = yield exec(command, { cwd: projectFolder });
        if (stderr) {
            throw stderr;
        }
        const output = JSON.parse(stdout);
        const folders = [];
        for (const projects of Object.values(output)) {
            for (const project of projects) {
                if (!isNameAccepted(project.name, options)) {
                    continue;
                }
                folders.push(...project.paths);
            }
        }
        return folders;
    });
}
const readdirOptions = {
    withFileTypes: true,
};
function packageFoldersFallbackNoDev(projectRoot, options) {
    return __awaiter(this, void 0, void 0, function* () {
        // package.json guided search
        const modulesRoot = path.join(projectRoot, "node_modules");
        const folders = [];
        const needsSearch = [projectRoot];
        while (needsSearch.length > 0) {
            const searchFolder = needsSearch.pop();
            let packageMeta;
            try {
                const packageMetaPath = path.join(searchFolder, "package.json");
                const packageMetaJson = yield fs.readFile(packageMetaPath, "utf8");
                packageMeta = JSON.parse(packageMetaJson);
            }
            catch (_a) {
                continue;
            }
            let storedPackages = [];
            const modulesPath = path.join(searchFolder, "node_modules");
            try {
                for (const entry of yield fs.readdir(modulesPath, readdirOptions)) {
                    if (!entry.isDirectory() || entry.name.startsWith(".")) {
                        continue;
                    }
                    if (!entry.name.startsWith("@")) {
                        // not scoped: lodash
                        storedPackages.push(entry.name);
                        continue;
                    }
                    // scoped: @ava/typescript
                    const entryPath = path.join(modulesPath, entry.name);
                    const scope = entry.name;
                    for (const entry of yield fs.readdir(entryPath, readdirOptions)) {
                        if (entry.isDirectory() && !entry.name.startsWith(".")) {
                            storedPackages.push(`${scope}/${entry.name}`);
                        }
                    }
                }
            }
            catch (_b) { }
            const dependencyNames = Object.keys(packageMeta.dependencies || {})
                // npm installs all optionals by default
                .concat(Object.keys(packageMeta.optionalDependencies || {}))
                // these dependencies are required
                // if it's defined only by devDependency it's a bug, but we should still track it
                .concat(Object.keys(packageMeta.peerDependencies || {}))
                // everything in a dependency's node_modules has been installed for something in this package, assume it's necessary
                // necessary as it seems child dependencies may install their dependencies in parent node_modules to avoid deep trees
                .concat(modulesRoot != modulesPath ? storedPackages : []);
            for (const packageName of dependencyNames) {
                if (!isNameAccepted(packageName, options)) {
                    continue;
                }
                const packagePath = path.join(storedPackages.includes(packageName) ? modulesPath : modulesRoot, packageName);
                if (!folders.includes(packagePath)) {
                    // hasn't already been added
                    folders.push(packagePath);
                    needsSearch.push(packagePath);
                }
            }
        }
        return folders;
    });
}
function packageFoldersFallback(projectRoot, options) {
    return __awaiter(this, void 0, void 0, function* () {
        // readdir based search
        const folders = [];
        const needsSearch = [path.join(projectRoot, "node_modules")];
        while (needsSearch.length > 0) {
            const searchFolder = needsSearch.pop();
            let entries;
            try {
                entries = yield fs.readdir(searchFolder, readdirOptions);
            }
            catch (_a) {
                continue;
            }
            for (const entry of entries) {
                if (!entry.isDirectory() || entry.name.startsWith(".")) {
                    continue;
                }
                const entryPath = path.join(searchFolder, entry.name);
                if (entry.name.startsWith("@")) {
                    needsSearch.push(entryPath);
                    continue;
                }
                needsSearch.push(path.join(entryPath, "node_modules"));
                const name = entryPath.slice(entryPath.lastIndexOf("/node_modules/") + 14);
                if (isNameAccepted(name, options)) {
                    folders.push(entryPath);
                }
            }
        }
        return folders;
    });
}
function isVersionNewer(sample, against) {
    for (let i = 0; i < sample.length && i < against.length; i++) {
        const a = sample[i];
        const b = against[i];
        if (a > b) {
            // newer
            return true;
        }
        if (a < b) {
            // older
            return false;
        }
    }
    // same version?
    return false;
}
function isNameAccepted(name, options) {
    var _a;
    if ((_a = options.exclude) === null || _a === void 0 ? void 0 : _a.includes(name)) {
        return false;
    }
    if (!options.include) {
        return true;
    }
    return options.include.includes(name);
}
export function compress(resolvedPackages) {
    const output = {
        packages: [],
        licenseText: [],
    };
    const reverseLookup = {};
    for (const result of resolvedPackages) {
        const licenses = [];
        for (let i = 0; i < result.licenses.length; i++) {
            const license = result.licenses[i];
            let key = reverseLookup[license.text];
            if (key == undefined) {
                key = output.licenseText.length;
                reverseLookup[license.text] = key;
                output.licenseText.push(license.text);
            }
            licenses.push(Object.assign(Object.assign({}, license), { text: key }));
        }
        const packageCompressed = Object.assign(Object.assign({}, result), { licenses });
        output.packages.push(packageCompressed);
    }
    return output;
}
