/*
 *
 * Copyright (C) 2013 Tobias Schoene www.yaacc.de
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package de.yaacc.upnp.server;

import android.annotation.SuppressLint;
import android.content.ContentUris;
import android.content.Context;
import android.content.SharedPreferences;
import android.database.Cursor;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.drawable.BitmapDrawable;
import android.graphics.drawable.Drawable;
import android.net.Uri;
import android.os.Build;
import android.provider.MediaStore;
import android.util.Log;
import android.util.Size;

import androidx.core.content.res.ResourcesCompat;
import androidx.preference.PreferenceManager;

import org.apache.hc.core5.http.ContentType;
import org.apache.hc.core5.http.EntityDetails;
import org.apache.hc.core5.http.HttpException;
import org.apache.hc.core5.http.HttpHeaders;
import org.apache.hc.core5.http.HttpRequest;
import org.apache.hc.core5.http.HttpStatus;
import org.apache.hc.core5.http.Message;
import org.apache.hc.core5.http.MethodNotSupportedException;
import org.apache.hc.core5.http.nio.AsyncEntityProducer;
import org.apache.hc.core5.http.nio.AsyncRequestConsumer;
import org.apache.hc.core5.http.nio.AsyncServerRequestHandler;
import org.apache.hc.core5.http.nio.StreamChannel;
import org.apache.hc.core5.http.nio.entity.AbstractBinAsyncEntityProducer;
import org.apache.hc.core5.http.nio.entity.AsyncEntityProducers;
import org.apache.hc.core5.http.nio.entity.BasicAsyncEntityConsumer;
import org.apache.hc.core5.http.nio.support.AsyncResponseBuilder;
import org.apache.hc.core5.http.nio.support.BasicRequestConsumer;
import org.apache.hc.core5.http.protocol.HttpContext;
import org.seamless.util.MimeType;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.RandomAccessFile;
import java.net.URL;
import java.net.URLConnection;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Locale;

import de.yaacc.R;
import de.yaacc.upnp.server.contentdirectory.MediaPathFilter;
import de.yaacc.util.HttpRange;

/**
 * A http service to retrieve media content by an id.
 *
 * @author Tobias Schoene (tobexyz)
 */
public class YaaccUpnpServerServiceHttpHandler implements AsyncServerRequestHandler<Message<HttpRequest, byte[]>> {

    private final Context context;

    public YaaccUpnpServerServiceHttpHandler(Context context) {
        this.context = context;

    }


    @Override
    public AsyncRequestConsumer<Message<HttpRequest, byte[]>> prepare(HttpRequest request, EntityDetails entityDetails, HttpContext context) {
        return new BasicRequestConsumer<>(entityDetails != null ? new BasicAsyncEntityConsumer() : null);
    }

    @Override
    public void handle(final Message<HttpRequest, byte[]> request,
                       final ResponseTrigger responseTrigger,
                       final HttpContext context) throws HttpException, IOException {

        Log.d(getClass().getName(), "Processing HTTP request: "
                + request.getHead().getRequestUri());
        final AsyncResponseBuilder responseBuilder = AsyncResponseBuilder.create(HttpStatus.SC_OK);
        // Extract what we need from the HTTP httpRequest
        String requestMethod = request.getHead().getMethod()
                .toUpperCase(Locale.ENGLISH);

        // Only accept HTTP-GET
        if (!requestMethod.equals("GET") && !requestMethod.equals("HEAD")) {
            Log.d(getClass().getName(),
                    "HTTP request isn't GET or HEAD stop! Method was: "
                            + requestMethod);
            throw new MethodNotSupportedException(requestMethod
                    + " method not supported");
        }

        Uri requestUri = Uri.parse(request.getHead().getRequestUri());
        List<String> pathSegments = requestUri.getPathSegments();
        if (pathSegments.size() == 1 && "health".equals(pathSegments.get(0))) {
            responseBuilder.setStatus(HttpStatus.SC_OK);
            responseBuilder.setEntity(AsyncEntityProducers.create("<html><body>I am alive</body></html>", ContentType.TEXT_HTML));
            responseTrigger.submitResponse(responseBuilder.build(), context);
            return;
        }
        if (pathSegments.size() < 2 || pathSegments.size() > 3) {
            createForbiddenResponse(responseTrigger, context, responseBuilder);
            return;
        }


        String type = pathSegments.get(0);
        String albumId = "";
        String thumbId = "";
        String contentId = "";
        if ("album".equals(type)) {
            albumId = pathSegments.get(1);
            try {
                Long.parseLong(albumId);
            } catch (NumberFormatException nex) {
                createForbiddenResponse(responseTrigger, context, responseBuilder);
                return;
            }
        } else if ("thumb".equals(type)) {
            thumbId = pathSegments.get(1);
            try {
                Long.parseLong(thumbId);
            } catch (NumberFormatException nex) {
                createForbiddenResponse(responseTrigger, context, responseBuilder);
                return;
            }
        } else if ("res".equals(type)) {
            contentId = pathSegments.get(1);
            try {
                Long.parseLong(contentId);
            } catch (NumberFormatException nex) {
                createForbiddenResponse(responseTrigger, context, responseBuilder);
                return;
            }
        }
        Arrays.stream(request.getHead().getHeaders()).forEach(it -> Log.d(getClass().getName(), "HEADER " + it.getName() + ": " + it.getValue()));
        List<HttpRange> ranges = new ArrayList<>();
        if (request.getHead().getHeader(HttpHeaders.RANGE) != null) {
            ranges = HttpRange.parseRangeHeader(request.getHead().getHeader(HttpHeaders.RANGE).getValue().toString());
        }
        ContentHolder contentHolder = null;
        if (!contentId.isEmpty()) {
            contentHolder = lookupContent(contentId, ranges);
        } else if (!albumId.isEmpty()) {
            contentHolder = lookupAlbumArt(albumId, ranges);
        } else if (!thumbId.isEmpty()) {
            contentHolder = lookupThumbnail(thumbId, ranges);
        } else if (YaaccUpnpServerService.PROXY_PATH.equals(type)) {
            contentHolder = lookupProxyContent(pathSegments.get(1), ranges);
        }
        if (contentHolder == null) {
            // tricky but works
            Log.d(getClass().getName(), "Resource with id " + contentId
                    + albumId + thumbId + pathSegments.get(1) + " not found");
            responseBuilder.setStatus(HttpStatus.SC_NOT_FOUND);
            String response =
                    "<html><body>Resource with id " + contentId + albumId
                            + thumbId + pathSegments.get(1) + " not found</body></html>";
            responseBuilder.setEntity(AsyncEntityProducers.create(response, ContentType.TEXT_HTML));
        } else {

            responseBuilder.setStatus(HttpStatus.SC_OK);
            responseBuilder.setEntity(contentHolder.getEntityProducer());
        }
        responseBuilder.setHeader(HttpHeaders.ACCEPT_RANGES, "none");
        responseTrigger.submitResponse(responseBuilder.build(), context);
        Log.d(getClass().getName(), "end doService: ");
    }

    private void createForbiddenResponse(ResponseTrigger responseTrigger, HttpContext context, AsyncResponseBuilder responseBuilder) throws HttpException, IOException {
        responseBuilder.setStatus(HttpStatus.SC_FORBIDDEN);
        responseBuilder.setEntity(AsyncEntityProducers.create("<html><body>Access denied</body></html>", ContentType.TEXT_HTML));
        responseTrigger.submitResponse(responseBuilder.build(), context);
        Log.d(getClass().getName(), "end doService: Access denied");
    }

    private Context getContext() {
        return context;
    }

    /**
     * Lookup content in the mediastore
     *
     * @param contentId the id of the content
     * @return the content description
     */
    private ContentHolder lookupContent(String contentId, List<HttpRange> ranges) {
        ContentHolder result = null;
        SharedPreferences preferences = PreferenceManager.getDefaultSharedPreferences(getContext());
        if (!preferences.getBoolean(getContext().getString(R.string.settings_local_server_chkbx), false)) {
            return null;
        }

        if (contentId == null) {
            return null;
        }
        Log.d(getClass().getName(), "System media store lookup: " + contentId);
        String[] projection = {MediaStore.Files.FileColumns._ID,
                MediaStore.Files.FileColumns.MIME_TYPE,
                MediaStore.Files.FileColumns.DATA};
        String selection = MediaStore.Files.FileColumns._ID + "=? and (" + MediaPathFilter.makeLikeClause(MediaStore.Files.FileColumns.DATA, MediaPathFilter.getMediaPathes(getContext()).size()) + ")";
        List<String> selectionArgsList = new ArrayList<>();
        selectionArgsList.add(contentId);
        selectionArgsList.addAll(MediaPathFilter.getMediaPathesForLikeClause(getContext()));
        String[] selectionArgs = selectionArgsList.toArray(new String[0]);
        try (Cursor mFilesCursor = getContext().getContentResolver().query(
                MediaStore.Files.getContentUri("external"), projection,
                selection, selectionArgs, null)) {

            if (mFilesCursor != null) {
                mFilesCursor.moveToFirst();
                while (!mFilesCursor.isAfterLast()) {
                    @SuppressLint("Range") String dataUri = mFilesCursor.getString(mFilesCursor
                            .getColumnIndex(MediaStore.Files.FileColumns.DATA));

                    @SuppressLint("Range") String mimeTypeStr = mFilesCursor
                            .getString(mFilesCursor
                                    .getColumnIndex(MediaStore.Files.FileColumns.MIME_TYPE));
                    MimeType mimeType = MimeType.valueOf("*/*");
                    if (mimeTypeStr != null) {
                        mimeType = MimeType.valueOf(mimeTypeStr);
                    }
                    Log.d(getClass().getName(), "Content found: " + mimeType
                            + " Uri: " + dataUri);
                    result = new ContentHolder(mimeType, dataUri, ranges);
                    mFilesCursor.moveToNext();
                }
            } else {
                Log.d(getClass().getName(), "System media store is empty.");
            }
        }

        return result;

    }

    /**
     * Lookup content in the mediastore
     *
     * @param albumId the id of the album
     * @return the content description
     */
    private ContentHolder lookupAlbumArt(String albumId, List<HttpRange> ranges) {

        ContentHolder result = new ContentHolder(MimeType.valueOf("image/png"),
                getDefaultIcon(), ranges);
        SharedPreferences preferences = PreferenceManager.getDefaultSharedPreferences(getContext());
        if (!preferences.getBoolean(getContext().getString(R.string.settings_local_server_chkbx), false)) {
            return result;
        }
        if (albumId == null) {
            return result;
        }
        if (Build.VERSION.SDK_INT <= Build.VERSION_CODES.P) {
            Log.d(getClass().getName(), "System media store lookup album: "
                    + albumId);
            String[] projection = {MediaStore.Audio.Albums._ID,
                    // FIXME what is the right mime type?
                    // MediaStore.Audio.Albums.MIME_TYPE,
                    MediaStore.Audio.Albums.ALBUM_ART};
            String selection = MediaStore.Audio.Albums._ID + "=?";
            String[] selectionArgs = {albumId};
            try (Cursor cursor = getContext().getContentResolver().query(
                    MediaStore.Audio.Albums.EXTERNAL_CONTENT_URI, projection,
                    selection, selectionArgs, null)) {

                if (cursor != null) {
                    cursor.moveToFirst();
                    while (!cursor.isAfterLast()) {
                        @SuppressLint("Range") String dataUri = cursor.getString(cursor
                                .getColumnIndex(MediaStore.Audio.Albums.ALBUM_ART));

                        // String mimeTypeStr = null;
                        // FIXME mime type resolving cursor
                        // .getString(cursor
                        // .getColumnIndex(MediaStore.Files.FileColumns.MIME_TYPE));

                        MimeType mimeType = MimeType.valueOf("image/png");
                        // if (mimeTypeStr != null) {
                        // mimeType = MimeType.valueOf(mimeTypeStr);
                        // }
                        if (dataUri != null) {
                            Log.d(getClass().getName(), "Content found: " + mimeType
                                    + " Uri: " + dataUri);
                            result = new ContentHolder(mimeType, dataUri, ranges);
                        } else {
                            Log.d(getClass().getName(), "Album art not found in media store. Fallback to default");
                            Bitmap bitmap = BitmapFactory.decodeResource(context.getResources(), R.drawable.yaacc192_32);

                            try {
                                File art = new File(context.getCacheDir(), "albumart" + albumId + ".jpg");
                                art.createNewFile();
                                FileOutputStream fos = new FileOutputStream(art);
                                bitmap.compress(Bitmap.CompressFormat.JPEG, 90, fos);
                                fos.flush();
                                fos.close();
                                result = new ContentHolder(mimeType, art.getAbsolutePath(), ranges);
                            } catch (IOException e) {
                                Log.e(getClass().getName(), "Error loading album art from file", e);
                            }
                        }
                        cursor.moveToNext();
                    }
                } else {
                    Log.d(getClass().getName(), "System media store is empty.");
                }
            }
        } else {
            Uri albumArtUri = ContentUris.withAppendedId(MediaStore.Audio.Albums.EXTERNAL_CONTENT_URI, Long.parseLong(albumId));
            MimeType mimeType = MimeType.valueOf("image/jpeg");
            Log.d(getClass().getName(), "Content found: " + mimeType
                    + " Uri: " + albumArtUri);
            Bitmap bitmap;
            try {
                bitmap = context.getContentResolver().loadThumbnail(albumArtUri, new Size(1024, 1024), null);
            } catch (IOException io) {
                Log.d(getClass().getName(), "Album art not found in media store. Fallback to default");
                bitmap = BitmapFactory.decodeResource(context.getResources(), R.drawable.yaacc192_32);
            }
            try {
                File art = new File(context.getCacheDir(), "albumart" + albumId + ".jpg");
                art.createNewFile();
                FileOutputStream fos = new FileOutputStream(art);
                bitmap.compress(Bitmap.CompressFormat.JPEG, 90, fos);
                fos.flush();
                fos.close();
                result = new ContentHolder(mimeType, art.getAbsolutePath(), ranges);
            } catch (IOException e) {
                Log.e(getClass().getName(), "Error loading album art from file", e);
            }

        }
        return result;
    }

    /**
     * Lookup a thumbnail content in the mediastore
     *
     * @param idStr the id of the thumbnail
     * @return the content description
     */
    private ContentHolder lookupThumbnail(String idStr, List<HttpRange> ranges) {

        ContentHolder result = new ContentHolder(MimeType.valueOf("image/png"),
                getDefaultIcon(), ranges);
        SharedPreferences preferences = PreferenceManager.getDefaultSharedPreferences(getContext());
        if (!preferences.getBoolean(getContext().getString(R.string.settings_local_server_chkbx), false)) {
            return result;
        }
        if (idStr == null) {
            return result;
        }
        long id;
        try {
            id = Long.parseLong(idStr);
        } catch (NumberFormatException nfe) {
            Log.d(getClass().getName(), "ParsingError of id: " + idStr, nfe);
            return result;
        }

        Log.d(getClass().getName(), "System media store lookup thumbnail: "
                + idStr);
        Bitmap bitmap = MediaStore.Images.Thumbnails.getThumbnail(getContext()
                        .getContentResolver(), id,
                MediaStore.Images.Thumbnails.MINI_KIND, null);
        if (bitmap != null) {
            ByteArrayOutputStream stream = new ByteArrayOutputStream();
            bitmap.compress(Bitmap.CompressFormat.PNG, 100, stream);
            byte[] byteArray = stream.toByteArray();

            MimeType mimeType = MimeType.valueOf("image/png");

            result = new ContentHolder(mimeType, byteArray, ranges);

        } else {
            Log.d(getClass().getName(), "System media store is empty.");
        }
        return result;
    }

    private ContentHolder lookupProxyContent(String contentKey, List<HttpRange> ranges) {

        String targetUri = PreferenceManager.getDefaultSharedPreferences(getContext()).getString(YaaccUpnpServerService.PROXY_LINK_KEY_PREFIX + contentKey, null);
        if (targetUri == null) {
            return null;
        }
        String targetMimetype = PreferenceManager.getDefaultSharedPreferences(getContext()).getString(YaaccUpnpServerService.PROXY_LINK_MIME_TYPE_KEY_PREFIX + contentKey, null);
        MimeType mimeType = MimeType.valueOf("*/*");
        if (targetMimetype != null) {
            mimeType = MimeType.valueOf(targetMimetype);
        }
        return new ContentHolder(mimeType, targetUri, ranges);
    }

    private byte[] getDefaultIcon() {
        Drawable drawable = ResourcesCompat.getDrawable(getContext().getResources(),
                R.drawable.yaacc192_32, getContext().getTheme());
        byte[] result = null;
        if (drawable != null) {
            Bitmap bitmap = ((BitmapDrawable) drawable).getBitmap();
            ByteArrayOutputStream stream = new ByteArrayOutputStream();
            bitmap.compress(Bitmap.CompressFormat.PNG, 100, stream);
            result = stream.toByteArray();
        }
        return result;
    }


    /**
     * ValueHolder for media content.
     */
    static class ContentHolder {
        private final MimeType mimeType;
        private String uri;
        private byte[] content;

        private List<HttpRange> ranges;

        public ContentHolder(MimeType mimeType, String uri, List<HttpRange> ranges) {
            this.uri = uri;
            this.mimeType = mimeType;
            this.ranges = ranges;

        }

        public ContentHolder(MimeType mimeType, byte[] content, List<HttpRange> ranges) {
            this.content = content;
            this.mimeType = mimeType;
            this.ranges = ranges;

        }

        /**
         * @return the uri
         */
        public String getUri() {
            return uri;
        }

        /**
         * @return the mimeType
         */
        public MimeType getMimeType() {
            return mimeType;
        }

        private byte[] readRangeFormFile(File file, List<HttpRange> ranges) throws IOException {


            try (RandomAccessFile raf = new RandomAccessFile(file, "r")) {
                long fileSize = raf.length();
                long startPosition;
                long rangeLength;
                if (ranges.size() > 1) {
                    Log.d(getClass().getName(), "More than on ranges requested. Currently only one range is supported. Responding with the first range");
                }
                if (ranges.isEmpty()) {
                    startPosition = 0;
                    rangeLength = fileSize;
                } else {
                    HttpRange range = ranges.get(0);
                    startPosition = range.getStart() == null ? 0 : range.getStart();
                    if (range.getEnd() == null || range.getEnd() == 0) {
                        rangeLength = fileSize;
                    } else {
                        rangeLength = range.getEnd() - startPosition;
                    }
                    if (range.getSuffixLength() != null && range.getSuffixLength() > 0) {
                        startPosition = fileSize - range.getSuffixLength();
                        rangeLength = range.getSuffixLength();
                    }
                }

                // Read a range of bytes (e.g., bytes 100 to 200)
                if (startPosition < 0 || startPosition + rangeLength > fileSize) {
                    Log.d(getClass().getName(), "Invalid range startPosition: " + startPosition + " rangeLength: " + rangeLength + " fileSize: " + fileSize);
                    rangeLength = fileSize - startPosition;
                    Log.d(getClass().getName(), "Adjusted range startPosition: " + startPosition + " rangeLength: " + rangeLength + " fileSize: " + fileSize);
                }

                raf.seek(startPosition); // Move to the starting position
                byte[] buffer = new byte[(int) rangeLength]; // Create a buffer
                raf.read(buffer);
                return buffer;
            }

        }

        public AsyncEntityProducer getEntityProducer() throws IOException {
            AsyncEntityProducer result = null;
            if (getUri() != null && !getUri().isEmpty()) {
                File file = new File(getUri());
                if (file.exists()) {
                    if (ranges.isEmpty()) {
                        result = AsyncEntityProducers.create(file, ContentType.parse(getMimeType().toString()));
                        Log.d(getClass().getName(), "Return without range request file-Uri: " + getUri()
                                + " Mimetype: " + getMimeType());
                    } else {
                        result = AsyncEntityProducers.create(readRangeFormFile(file, ranges), ContentType.parse(getMimeType().toString()));
                    }
                } else {
                    //file not found maybe external url
                    result = new AbstractBinAsyncEntityProducer(0, ContentType.parse(getMimeType().toString())) {
                        private InputStream input;
                        private long length = -1;

                        AbstractBinAsyncEntityProducer init() {
                            try {
                                if (input == null) {
                                    //https://www.experts-exchange.com/questions/10171110/Reading-a-part-of-a-file-using-URLConnection.html
                                    URLConnection con = new URL(getUri()).openConnection();
                                    con.setRequestProperty("Range", HttpRange.toHeaderString(ranges));
                                    input = con.getInputStream();
                                    length = con.getContentLength();
                                }
                            } catch (IOException e) {
                                Log.e(getClass().getName(), "Error opening external content", e);
                            }
                            return this;
                        }

                        @Override
                        public long getContentLength() {
                            return length;
                        }

                        @Override
                        protected int availableData() {
                            return Integer.MAX_VALUE;
                        }

                        @Override
                        protected void produceData(final StreamChannel<ByteBuffer> channel) throws IOException {
                            try {
                                if (input == null) {
                                    //retry opening external content if it hasn't been opened yet
                                    URLConnection con = new URL(getUri()).openConnection();
                                    input = con.getInputStream();
                                    length = con.getContentLength();
                                }
                                byte[] tempBuffer = new byte[1024];
                                int bytesRead;
                                if (-1 != (bytesRead = input.read(tempBuffer))) {
                                    channel.write(ByteBuffer.wrap(tempBuffer, 0, bytesRead));
                                }
                                if (bytesRead == -1) {
                                    channel.endStream();
                                }

                            } catch (IOException e) {
                                Log.e(getClass().getName(), "Error reading external content", e);
                                throw e;
                            }
                        }


                        @Override
                        public boolean isRepeatable() {
                            return true;
                        }

                        @Override
                        public void failed(final Exception cause) {
                        }

                    }.init();

                    Log.d(getClass().getName(), "Return external-Uri: " + getUri()
                            + "Mimetype: " + getMimeType());
                }
            } else if (content != null) {
                result = AsyncEntityProducers.create(content, ContentType.parse(getMimeType().toString()));
            }
            if (result == null) {
                Log.d(getClass().getName(), "Resource is null");
                return AsyncEntityProducers.create("<html><body><h1>Resource not found</h1></body></html>", ContentType.TEXT_HTML);
            }
            return result;

        }
    }
}
