package de.westnordost.streetcomplete.osm.street_parking

import de.westnordost.streetcomplete.osm.Sides
import de.westnordost.streetcomplete.osm.expandSidesTags
import de.westnordost.streetcomplete.osm.street_parking.ParkingOrientation.*
import de.westnordost.streetcomplete.osm.street_parking.ParkingPosition.*

fun parseStreetParkingSides(tags: Map<String, String>): Sides<StreetParking>? {
    // expand tags first so we do not need to deal with :both and naked tags
    val expandedTags = expandRelevantSidesTags(tags)
    // first try to parse new schema
    var left = parseParkingForSide(expandedTags, "left")
    var right = parseParkingForSide(expandedTags, "right")
    if (left == null && right == null) {
        // otherwise parse the old schema
        left = parseParkingForSideOldSchema(expandedTags, "left")
        right = parseParkingForSideOldSchema(expandedTags, "right")
    }

    if (left == null && right == null) return null

    return Sides(left, right)
}

/** Parsing new schema:
 *  https://wiki.openstreetmap.org/wiki/Street_parking */
private fun parseParkingForSide(tags: Map<String, String>, side: String): StreetParking? {
    val isStaggered = tags["parking:$side:staggered"] == "yes"
    val markings = tags["parking:$side:markings"]
    val hasMarkings = markings != "no" && markings != null

    val position = when (tags["parking:$side"]) {
        "lane" -> when {
            isStaggered && hasMarkings -> PAINTED_AREA_ONLY
            isStaggered -> STAGGERED_ON_STREET
            else -> ON_STREET
        }
        "half_on_kerb" -> if (isStaggered) STAGGERED_HALF_ON_STREET else HALF_ON_STREET
        "on_kerb" -> OFF_STREET
        // interpret "shoulder" as "off street", i.e. the same as "on_kerb". Reasons given in
        // https://wiki.openstreetmap.org/wiki/Talk:Street_parking#Suggestion_to_remove_parking:side=shoulder
        "shoulder" -> OFF_STREET
        "street_side" -> STREET_SIDE
        "no" -> return StreetParking.None
        "separate" -> return StreetParking.Separate
        "yes" -> return StreetParking.Incomplete
        null -> null
        else -> return StreetParking.Unknown
    }

    val orientation = when (tags["parking:$side:orientation"]) {
        "parallel" -> PARALLEL
        "diagonal" -> DIAGONAL
        "perpendicular" -> PERPENDICULAR
        null -> null
        else -> return StreetParking.Unknown
    }

    if (position == null && orientation == null) return null
    if (position == null || orientation == null) return StreetParking.Incomplete

    return StreetParking.PositionAndOrientation(orientation, position)
}

/** Parsing old parking schema:
 *  https://wiki.openstreetmap.org/wiki/Key:parking:lane */
private fun parseParkingForSideOldSchema(tags: Map<String, String>, side: String): StreetParking? {
    val parkingValue = tags["parking:lane:$side"] ?: return null

    when (parkingValue) {
        "no_parking", "no_standing", "no_stopping", "no" ->
            return StreetParking.None
        "yes" ->
            return StreetParking.Incomplete
        "separate" ->
            return StreetParking.Separate
        else -> {
            // regard parking:lanes:*=marked as incomplete (because position is missing implicitly)
            val parkingOrientation = when (parkingValue) {
                "parallel" -> PARALLEL
                "diagonal" -> DIAGONAL
                "perpendicular" -> PERPENDICULAR
                else -> null
            } ?: return if (parkingValue == "marked") StreetParking.Incomplete else StreetParking.Unknown

            val parkingPositionValue = tags["parking:lane:$side:$parkingValue"]
                // parking position is mandatory to be regarded as complete
                ?: return StreetParking.Incomplete

            val parkingPosition = when (parkingPositionValue) {
                "on_street" -> ON_STREET
                "half_on_kerb" -> HALF_ON_STREET
                "on_kerb", "shoulder" -> OFF_STREET
                "painted_area_only" -> PAINTED_AREA_ONLY
                "lay_by", "street_side" -> STREET_SIDE

                else -> null
            } ?: return StreetParking.Unknown

            return StreetParking.PositionAndOrientation(parkingOrientation, parkingPosition)
        }
    }
}

private fun expandRelevantSidesTags(tags: Map<String, String>): Map<String, String> {
    val result = tags.toMutableMap()
    result.expandSidesTags("parking:lane", "", true)
    result.expandSidesTags("parking:lane", "parallel", true)
    result.expandSidesTags("parking:lane", "diagonal", true)
    result.expandSidesTags("parking:lane", "perpendicular", true)
    result.expandSidesTags("parking", "", false)
    result.expandSidesTags("parking", "orientation", false)
    result.expandSidesTags("parking", "staggered", false)
    result.expandSidesTags("parking", "markings", false)
    return result
}
