package de.westnordost.streetcomplete.quests.address

import de.westnordost.streetcomplete.data.meta.CountryInfo
import de.westnordost.streetcomplete.data.meta.IncompleteCountryInfo
import de.westnordost.streetcomplete.data.osm.edits.update_tags.StringMapEntryAdd
import de.westnordost.streetcomplete.data.osm.edits.update_tags.StringMapEntryModify
import de.westnordost.streetcomplete.data.osm.geometry.ElementPointGeometry
import de.westnordost.streetcomplete.data.osm.geometry.ElementPolygonsGeometry
import de.westnordost.streetcomplete.data.osm.mapdata.ElementType
import de.westnordost.streetcomplete.data.osm.mapdata.Node
import de.westnordost.streetcomplete.data.osm.mapdata.Way
import de.westnordost.streetcomplete.osm.address.HouseNumber
import de.westnordost.streetcomplete.quests.answerApplied
import de.westnordost.streetcomplete.quests.answerAppliedTo
import de.westnordost.streetcomplete.quests.createMapData
import de.westnordost.streetcomplete.testutils.member
import de.westnordost.streetcomplete.testutils.node
import de.westnordost.streetcomplete.testutils.p
import de.westnordost.streetcomplete.testutils.rel
import de.westnordost.streetcomplete.testutils.way
import kotlin.test.Test
import kotlin.test.assertEquals
import kotlin.test.assertNull

class AddHousenumberTest {

    private val questType = AddHousenumber({ CountryInfo(listOf(IncompleteCountryInfo("DE"))) })

    @Test fun `does not create quest for generic building`() {
        val building = way(1L, NODES1, mapOf("building" to "yes"))
        val mapData = createMapData(mapOf(building to POSITIONS1))
        assertEquals(0, questType.getApplicableElements(mapData).toList().size)
        assertEquals(false, questType.isApplicableTo(building))
    }

    @Test fun `does not create quest for building with address`() {
        val building = way(1L, NODES1, mapOf(
            "building" to "detached",
            "addr:housenumber" to "123"
        ))
        val mapData = createMapData(mapOf(building to POSITIONS1))
        assertEquals(0, questType.getApplicableElements(mapData).toList().size)
        assertEquals(false, questType.isApplicableTo(building))
    }

    @Test fun `does create quest for building without address`() {
        val building = way(1L, NODES1, mapOf(
            "building" to "detached"
        ))
        val mapData = createMapData(mapOf(building to POSITIONS1))
        assertEquals(1, questType.getApplicableElements(mapData).toList().size)
        assertNull(questType.isApplicableTo(building))
    }

    @Test fun `does not create quest for building with address node on outline`() {
        val building = way(1, NODES1, mapOf(
            "building" to "detached"
        ))
        val addr = node(2, P2, mapOf(
            "addr:housenumber" to "123"
        ))
        val mapData = createMapData(mapOf(
            building to POSITIONS1,
            addr to ElementPointGeometry(P2)
        ))
        assertEquals(0, questType.getApplicableElements(mapData).toList().size)
        assertNull(questType.isApplicableTo(building))
    }

    @Test fun `does not create quest for building that is part of a relation with an address`() {
        val building = way(1, NODES1, mapOf(
            "building" to "detached"
        ))
        val relationWithAddr = rel(
            members = listOf(member(ElementType.WAY, 1)),
            tags = mapOf("addr:housenumber" to "123")
        )

        val mapData = createMapData(mapOf(
            building to POSITIONS1,
            relationWithAddr to ElementPointGeometry(P2)
        ))
        assertEquals(0, questType.getApplicableElements(mapData).toList().size)
        assertNull(questType.isApplicableTo(building))
    }

    @Test fun `does not create quest for building that is inside an area with an address`() {
        val building = way(1L, NODES1, mapOf(
            "building" to "detached"
        ))
        val areaWithAddr = way(1L, NODES2, mapOf(
            "addr:housenumber" to "123",
            "amenity" to "school",
        ))
        val mapData = createMapData(mapOf(
            building to POSITIONS1,
            areaWithAddr to POSITIONS2,
        ))
        assertEquals(0, questType.getApplicableElements(mapData).toList().size)
        assertNull(questType.isApplicableTo(building))
    }

    @Test fun `does not create quest for building that is inside an area with an address on its outline`() {
        val mapData = createMapData(mapOf(
            Way(1L, NODES1, mapOf(
                "building" to "detached"
            ), 1) to POSITIONS1,
            Way(1L, NODES2, mapOf(
                "amenity" to "school"
            ), 1) to POSITIONS2,
            Node(6L, P6, mapOf(
                "addr:housenumber" to "123"
            ), 1) to ElementPointGeometry(P6)
        ))
        assertEquals(0, questType.getApplicableElements(mapData).toList().size)
    }

    // https://github.com/streetcomplete/StreetComplete/issues/6500#issuecomment-3353073272
    @Test fun `does create quest for building that is inside an area with an address on its outline that is however associated to another building`() {
        val houseWithAddressNode = Pair(
            Way(1L, listOf(1, 2, 3, 4, 1), mapOf("building" to "detached"), 1),
            ElementPolygonsGeometry(listOf(listOf(P1, P2, P3, P4, P1)), PC),
        )
        val areaWithAddressNode = Pair(
            Way(2L, listOf(1, 5, 6, 7, 8, 1), mapOf("landuse" to "residential"), 1),
            ElementPolygonsGeometry(listOf(listOf(P1, P5, P6, P7, P8, P1)), PC),
        )
        // address is part of both of the above
        val addressNode = Pair(
            Node(1L, P1, mapOf("addr:housenumber" to "123"), 1),
            ElementPointGeometry(P1)
        )

        val house = Pair(
            Way(3L, listOf(4, 3, 7, 8, 4), mapOf("building" to "detached"), 1),
            ElementPolygonsGeometry(listOf(listOf(P4, P3, P7, P8, P4)), PC),
        )

        val mapData = createMapData(mapOf(
            houseWithAddressNode,
            areaWithAddressNode,
            addressNode,
            house,
        ))
        assertEquals(house.first, questType.getApplicableElements(mapData).toList().single())
    }

    // this test is identical to the above, but buildings and other areas are treated differently,
    // so this should be tested both with buildings and other areas
    @Test fun `does create quest for building that is inside an area with an address on its outline that is however associated to another area`() {
        val smallerAreaWithAddressNode = Pair(
            Way(1L, listOf(1, 2, 3, 4, 1), mapOf("amenity" to "school"), 1),
            ElementPolygonsGeometry(listOf(listOf(P1, P2, P3, P4, P1)), PC),
        )
        val largerAreaWithAddressNode = Pair(
            Way(2L, listOf(1, 5, 6, 7, 8, 1), mapOf("landuse" to "residential"), 1),
            ElementPolygonsGeometry(listOf(listOf(P1, P5, P6, P7, P8, P1)), PC),
        )
        // address is part of both of the above
        val addressNode = Pair(
            Node(1L, P1, mapOf("addr:housenumber" to "123"), 1),
            ElementPointGeometry(P1)
        )

        val house = Pair(
            Way(3L, listOf(4, 3, 7, 8, 4), mapOf("building" to "detached"), 1),
            ElementPolygonsGeometry(listOf(listOf(P4, P3, P7, P8, P4)), PC),
        )

        val mapData = createMapData(mapOf(
            smallerAreaWithAddressNode,
            largerAreaWithAddressNode,
            addressNode,
            house,
        ))
        assertEquals(house.first, questType.getApplicableElements(mapData).toList().single())
    }

    @Test fun `does not create quest for building that contains an address node`() {
        val building = way(1L, NODES1, mapOf(
            "building" to "detached"
        ))
        val addr = node(1L, PC, mapOf(
            "addr:housenumber" to "123"
        ))
        val mapData = createMapData(mapOf(
            building to POSITIONS1,
            addr to ElementPointGeometry(PC),
        ))
        assertEquals(0, questType.getApplicableElements(mapData).toList().size)
        assertNull(questType.isApplicableTo(building))
    }

    @Test fun `does not create quest for building that intersects bounding box`() {
        val building = way(1L, NODES1, mapOf(
            "building" to "detached"
        ))
        val mapData = createMapData(mapOf(
            building to ElementPolygonsGeometry(listOf(listOf(P1, P2, PO, P4, P1)), PC)
        ))
        assertEquals(0, questType.getApplicableElements(mapData).toList().size)
        assertNull(questType.isApplicableTo(building))
    }

    @Test fun `apply house number answer`() {
        assertEquals(
            setOf(StringMapEntryAdd("addr:housenumber", "99b")),
            questType.answerApplied(AddressNumberAndName(HouseNumber("99b"), null))
        )
    }

    @Test fun `apply house name answer`() {
        assertEquals(
            setOf(StringMapEntryAdd("addr:housename", "La Escalera")),
            questType.answerApplied(AddressNumberAndName(null, "La Escalera"))
        )
    }

    @Test fun `apply no house number answer`() {
        assertEquals(
            setOf(StringMapEntryAdd("nohousenumber", "yes")),
            questType.answerApplied(AddressNumberAndName(null, null))
        )
    }

    @Test fun `apply wrong building type answer`() {
        assertEquals(
            setOf(StringMapEntryModify("building", "residential", "yes")),
            questType.answerAppliedTo(HouseNumberAnswer.WrongBuildingType, mapOf("building" to "residential"))
        )
    }
}

private val P1 = p(0.25, 0.25)
private val P2 = p(0.25, 0.75)
private val P3 = p(0.75, 0.75)
private val P4 = p(0.75, 0.25)

private val P5 = p(0.1, 0.1)
private val P6 = p(0.1, 0.9)
private val P7 = p(0.9, 0.9)
private val P8 = p(0.9, 0.1)

private val PO = p(1.5, 1.5)
private val PC = p(0.5, 0.5)

private val NODES1 = listOf<Long>(1, 2, 3, 4, 1)
private val NODES2 = listOf<Long>(5, 6, 7, 8, 5)

private val POSITIONS1 = ElementPolygonsGeometry(listOf(listOf(P1, P2, P3, P4, P1)), PC)
private val POSITIONS2 = ElementPolygonsGeometry(listOf(listOf(P5, P6, P7, P8, P5)), PC)
