package de.westnordost.streetcomplete.quests.width

import de.westnordost.streetcomplete.R
import de.westnordost.streetcomplete.data.elementfilter.toElementFilterExpression
import de.westnordost.streetcomplete.data.osm.geometry.ElementGeometry
import de.westnordost.streetcomplete.data.osm.mapdata.Element
import de.westnordost.streetcomplete.data.osm.mapdata.MapDataWithGeometry
import de.westnordost.streetcomplete.data.osm.mapdata.filter
import de.westnordost.streetcomplete.data.osm.osmquests.OsmElementQuestType
import de.westnordost.streetcomplete.data.quest.AndroidQuest
import de.westnordost.streetcomplete.data.user.achievements.EditTypeAchievement
import de.westnordost.streetcomplete.osm.ROADS_ASSUMED_TO_BE_PAVED
import de.westnordost.streetcomplete.osm.Tags
import de.westnordost.streetcomplete.osm.maxspeed.MAX_SPEED_TYPE_KEYS
import de.westnordost.streetcomplete.osm.surface.PAVED_SURFACES
import de.westnordost.streetcomplete.resources.Res
import de.westnordost.streetcomplete.resources.default_disabled_msg_difficult_and_time_consuming
import de.westnordost.streetcomplete.resources.default_disabled_msg_no_ar
import de.westnordost.streetcomplete.screens.measure.ArSupportChecker
import org.jetbrains.compose.resources.StringResource

class AddRoadWidth(
    private val checkArSupport: ArSupportChecker
) : OsmElementQuestType<WidthAnswer>, AndroidQuest {

    private val nodeFilter by lazy { """
       nodes with
         traffic_calming ~ ${ROAD_NARROWERS.joinToString("|")}
         and (!width or source:width ~ ".*estimat.*")
         and (!maxwidth or source:maxwidth ~ ".*estimat.*")
    """.toElementFilterExpression() }

    private val wayFilter by lazy { """
        ways with (
          (
            highway ~ trunk|primary|secondary|tertiary|unclassified|residential|busway
            and (lane_markings = no or lanes < 2)
          ) or (
            highway = residential
            and (
              maxspeed < 33
              or maxspeed = walk
              or ~"${MAX_SPEED_TYPE_KEYS.joinToString("|")}" ~ ".*:(zone)?:?([1-9]|[1-2][0-9]|30)"
            )
            and lane_markings != yes and (!lanes or lanes < 2)
          )
          or highway = living_street
          or highway = service and service = alley
        )
        and area != yes
        and (!width or source:width ~ ".*estimat.*")
        and (traffic_calming !~ ${ROAD_NARROWERS.joinToString("|")} or !maxwidth or source:maxwidth ~".*estimat.*")
        and (surface ~ ${PAVED_SURFACES.joinToString("|")} or highway ~ ${ROADS_ASSUMED_TO_BE_PAVED.joinToString("|")})
        and (access !~ private|no or (foot and foot !~ private|no))
        and foot != no
        and placement != transition
    """.toElementFilterExpression() }

    override val changesetComment = "Determine road widths"
    override val wikiLink = "Key:width"
    override val icon = R.drawable.quest_street_width
    override val achievements = listOf(EditTypeAchievement.CAR)
    override val defaultDisabledMessage: StringResource? get() =
        if (!checkArSupport()) Res.string.default_disabled_msg_no_ar
        else Res.string.default_disabled_msg_difficult_and_time_consuming

    override fun getTitle(tags: Map<String, String>) = R.string.quest_road_width_title

    override fun getApplicableElements(mapData: MapDataWithGeometry) =
        mapData.nodes.filter { nodeFilter.matches(it) } + mapData.ways.filter { wayFilter.matches(it) }

    override fun isApplicableTo(element: Element) =
        nodeFilter.matches(element) || wayFilter.matches(element)

    override fun getHighlightedElements(element: Element, getMapData: () -> MapDataWithGeometry) =
        getMapData().filter("nodes with traffic_calming ~ choker|chicane|island|choked_island|choked_table")

    override fun createForm() = AddWidthForm()

    override fun applyAnswerTo(answer: WidthAnswer, tags: Tags, geometry: ElementGeometry, timestampEdited: Long) {
        val key = if (tags["traffic_calming"] in ROAD_NARROWERS) "maxwidth" else "width"

        tags[key] = answer.width.toOsmValue()

        if (answer.isARMeasurement) {
            tags["source:$key"] = "ARCore"
        } else {
            tags.remove("source:$key")
        }

        // update width:carriageway if it is set
        if (key == "width" && tags.containsKey("width:carriageway")) {
            tags["width:carriageway"] = answer.width.toOsmValue()
        }
    }
}

private val ROAD_NARROWERS = setOf("choker", "chicane", "choked_table")
