package de.westnordost.streetcomplete.quests.road_name

import de.westnordost.streetcomplete.R
import de.westnordost.streetcomplete.data.osm.geometry.ElementGeometry
import de.westnordost.streetcomplete.data.osm.osmquests.OsmFilterQuestType
import de.westnordost.streetcomplete.data.quest.AllCountriesExcept
import de.westnordost.streetcomplete.data.quest.AndroidQuest
import de.westnordost.streetcomplete.data.user.achievements.EditTypeAchievement.CAR
import de.westnordost.streetcomplete.data.user.achievements.EditTypeAchievement.PEDESTRIAN
import de.westnordost.streetcomplete.data.user.achievements.EditTypeAchievement.POSTMAN
import de.westnordost.streetcomplete.osm.Tags
import de.westnordost.streetcomplete.osm.localized_name.LocalizedName
import de.westnordost.streetcomplete.osm.localized_name.applyTo

class AddRoadName : OsmFilterQuestType<RoadNameAnswer>(), AndroidQuest {

    override val elementFilter = """
        ways with
          highway ~ primary|secondary|tertiary|unclassified|residential|living_street|pedestrian|busway
          and !name and !name:left and !name:right
          and !ref
          and noname != yes
          and name:signed != no
          and !junction
          and area != yes
          and (
            access !~ private|no
            or foot and foot !~ private|no
          )
    """
    override val enabledInCountries = AllCountriesExcept("JP")
    override val changesetComment = "Determine road names and types"
    override val wikiLink = "Key:name"
    override val icon = R.drawable.quest_street_name2
    override val hasMarkersAtEnds = true
    override val achievements = listOf(CAR, PEDESTRIAN, POSTMAN)

    override val hint = R.string.quest_streetName_hint

    override fun getTitle(tags: Map<String, String>) = R.string.quest_streetName_title

    override fun createForm() = AddRoadNameForm()

    override fun applyAnswerTo(answer: RoadNameAnswer, tags: Tags, geometry: ElementGeometry, timestampEdited: Long) {
        when (answer) {
            RoadNameAnswer.NoName -> tags["noname"] = "yes"
            is RoadName -> {
                val singleName = answer.localizedNames.singleOrNull()
                if (singleName?.isRef() == true) {
                    tags["ref"] = singleName.name
                } else {
                    answer.localizedNames.applyTo(tags)
                }
            }
        }
    }
}

private fun LocalizedName.isRef() =
    languageTag.isEmpty() && name.matches("[A-Z]{0,3}[ -]?[0-9]{0,5}".toRegex())
