import com.esotericsoftware.yamlbeans.YamlConfig
import com.esotericsoftware.yamlbeans.YamlWriter
import de.westnordost.countryboundaries.CountryBoundaries
import kotlinx.io.asSource
import kotlinx.io.buffered
import org.gradle.api.DefaultTask
import org.gradle.api.tasks.Input
import org.gradle.api.tasks.TaskAction
import java.io.FileInputStream
import java.io.FileWriter
import java.io.StringWriter
import java.net.HttpURLConnection
import java.net.URL
import java.net.URLEncoder

/** Counts the occurrence of values for a given key for a certain tag combination by country and
 *  writes the result in a YML file.
 *
 *  So, much like for example taginfo's values page but sorted by country code plus only counting
 *  elements sufficing a certain tag combination.
 *  ( https://taginfo.openstreetmap.org/keys/operator#values ) */
open class QLeverCountValueByCountryTask : DefaultTask() {

    @get:Input lateinit var targetFile: String
    @get:Input lateinit var osmTag: String
    @get:Input lateinit var sparqlQueryPart: String
    @get:Input var minCount: Int = 1
    @get:Input var minPercent: Double = 0.0

    private val firstPointRegex = Regex("[A-Za-z(]*([-+\\d.]*) ([-+\\d.]*)")
    private val boundaries = CountryBoundaries.deserializeFrom(
        FileInputStream("${project.projectDir}/app/src/androidMain/assets/boundaries.ser").asSource().buffered()
    )

    @TaskAction fun run() {
        val query = """
            PREFIX geo: <http://www.opengis.net/ont/geosparql#>
            PREFIX osmkey: <https://www.openstreetmap.org/wiki/Key:>
            SELECT ?value ?geometry WHERE {
            ?osm $sparqlQueryPart
            osmkey:$osmTag ?value;
            geo:hasGeometry/geo:asWKT ?geometry.
            }
            """.trimIndent()

        // country code -> ( value -> count )
        val result: MutableMap<String, MutableMap<String, Int>> = mutableMapOf()

        val rows = queryQLeverTsv(query).mapNotNull { parseTsvRow(it) }

        for (row in rows) {
            row.countryCode?.let {
                result.getOrPut(it, { mutableMapOf() })
                    .compute(row.value) { _, u -> (u ?: 0) + 1 }
            }
        }

        val config = YamlConfig().apply {
            writeConfig.setWriteClassname(YamlConfig.WriteClassName.NEVER)
            writeConfig.isFlowStyle = true
            writeConfig.setEscapeUnicode(false)
        }

        val fileWriter = FileWriter(targetFile, false)
        fileWriter.write("# Do not edit manually, if anything is wrong here it almost certainly should be fixed in OSM map data.\n")
        fileWriter.write("# Data generated by counting number of OSM elements in the respective countries.\n\n")

        for (countryCode in result.keys.sorted()) {
            val valuesForCountry = result[countryCode]!!
            val entries = valuesForCountry.entries.sortedByDescending { it.key }.sortedByDescending { it.value }
            val totalCount = valuesForCountry.values.sum()
            var hasAddedCountry = false
            for ((value, count) in entries) {
                if (count < minCount) continue
                if (100 * (count.toDouble() / totalCount) < minPercent) continue

                if (!hasAddedCountry) {
                    fileWriter.write("$countryCode:\n")
                    hasAddedCountry = true
                }
                fileWriter.write("  - ${writeYaml(value, config)} # $count\n")
            }
        }
        fileWriter.close()
    }

    private val Row.countryCode: String? get() = boundaries.getIds(lon, lat).firstOrNull()

    private fun writeYaml(obj: String, config: YamlConfig): String {
        val str = StringWriter()
        val writer = YamlWriter(str, config)
        writer.write(obj)
        writer.close()
        return str.toString().removeSuffix("\n").removeSuffix("\r")
    }

    private fun queryQLeverTsv(query: String): List<String> {
        val url = URL("https://qlever.dev/api/osm-planet?query=${URLEncoder.encode(query, "UTF-8")}&action=tsv_export")
        val connection = url.openConnection() as HttpURLConnection
        try {
            connection.setRequestProperty("User-Agent", "StreetComplete")
            return connection.inputStream.bufferedReader().readLines()
        } finally {
            connection.disconnect()
        }
    }

    private fun parseTsvRow(row: String): Row? {
        val t = row.lastIndexOf('\t')
        if (t == -1) return null
        if (row.length < 3) return null
        val value = row.substring(1, t - 1) // value without "..."
        if (t + 1 >= row.length) return null
        val geometry = row.substring(t + 1 + 1) // geometry without starting "
        val matchResult = firstPointRegex.matchAt(geometry, 0) ?: return null
        val lon = matchResult.groupValues[1].toDoubleOrNull() ?: return null
        val lat = matchResult.groupValues[2].toDoubleOrNull() ?: return null
        return Row(value, lon, lat)
    }
}

private data class Row(val value: String, val lon: Double, val lat: Double)
