package de.westnordost.streetcomplete.screens.user.achievements

import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.PaddingValues
import androidx.compose.foundation.layout.WindowInsets
import androidx.compose.foundation.layout.WindowInsetsSides
import androidx.compose.foundation.layout.asPaddingValues
import androidx.compose.foundation.layout.consumeWindowInsets
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.only
import androidx.compose.foundation.layout.safeDrawing
import androidx.compose.runtime.Composable
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.derivedStateOf
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.unit.dp
import de.westnordost.streetcomplete.data.user.achievements.Achievement
import de.westnordost.streetcomplete.resources.Res
import de.westnordost.streetcomplete.resources.achievements_empty
import de.westnordost.streetcomplete.resources.stats_are_syncing
import de.westnordost.streetcomplete.ui.common.CenteredLargeTitleHint
import de.westnordost.streetcomplete.ui.ktx.plus
import org.jetbrains.compose.resources.stringResource

/** Shows the icons for all achieved achievements and opens a dialog to show the details on click. */
@Composable
fun AchievementsScreen(viewModel: AchievementsViewModel) {
    val achievements by viewModel.achievements.collectAsState()
    val hasAchievements by remember { derivedStateOf { achievements?.isNotEmpty() == true } }

    var showAchievement by remember { mutableStateOf<Pair<Achievement, Int>?>(null) }

    Box {
        achievements?.let {
            val insets = WindowInsets.safeDrawing.only(
                WindowInsetsSides.Horizontal + WindowInsetsSides.Bottom
            ).asPaddingValues()
            LazyAchievementsGrid(
                achievements = it,
                onClickAchievement = { achievement, level ->
                    showAchievement = achievement to level
                },
                modifier = Modifier.fillMaxSize().consumeWindowInsets(insets),
                contentPadding = insets + PaddingValues(16.dp)
            )
        }
        if (!hasAchievements) {
            val isSynchronizingStatistics by viewModel.isSynchronizingStatistics.collectAsState()
            CenteredLargeTitleHint(stringResource(
                if (isSynchronizingStatistics) Res.string.stats_are_syncing
                else Res.string.achievements_empty
            ))
        }
    }

    showAchievement?.let { (achievement, level) ->
        AchievementDialog(
            achievement, level,
            unlockedLinks = emptyList(),
            onDismissRequest = { showAchievement = null },
            isNew = false
        )
    }
}
