package de.westnordost.streetcomplete.screens.about.logs

import android.content.Context
import android.content.Intent
import androidx.compose.foundation.background
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.WindowInsets
import androidx.compose.foundation.layout.WindowInsetsSides
import androidx.compose.foundation.layout.asPaddingValues
import androidx.compose.foundation.layout.consumeWindowInsets
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.only
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.safeDrawing
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.itemsIndexed
import androidx.compose.foundation.lazy.rememberLazyListState
import androidx.compose.foundation.shape.CircleShape
import androidx.compose.material.AppBarDefaults
import androidx.compose.material.Divider
import androidx.compose.material.Icon
import androidx.compose.material.IconButton
import androidx.compose.material.MaterialTheme
import androidx.compose.material.Text
import androidx.compose.material.TopAppBar
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.unit.dp
import de.westnordost.streetcomplete.ApplicationConstants
import de.westnordost.streetcomplete.BuildConfig
import de.westnordost.streetcomplete.data.logs.format
import de.westnordost.streetcomplete.resources.Res
import de.westnordost.streetcomplete.resources.about_title_logs
import de.westnordost.streetcomplete.resources.action_filter
import de.westnordost.streetcomplete.resources.action_share
import de.westnordost.streetcomplete.resources.ic_filter_list_24
import de.westnordost.streetcomplete.resources.ic_share_24
import de.westnordost.streetcomplete.resources.no_search_results
import de.westnordost.streetcomplete.ui.common.BackIcon
import de.westnordost.streetcomplete.ui.common.CenteredLargeTitleHint
import de.westnordost.streetcomplete.ui.ktx.isScrolledToEnd
import de.westnordost.streetcomplete.util.ktx.now
import kotlinx.datetime.LocalDateTime
import org.jetbrains.compose.resources.painterResource
import org.jetbrains.compose.resources.stringResource

/** Shows the app logs */
@Composable
fun LogsScreen(
    viewModel: LogsViewModel,
    onClickBack: () -> Unit,
) {
    val logs by viewModel.logs.collectAsState()
    val filters by viewModel.filters.collectAsState()
    val filtersCount = remember(filters) { filters.count() }

    var showFiltersDialog by remember { mutableStateOf(false) }
    val listState = rememberLazyListState()

    LaunchedEffect(logs.size) {
        if (listState.isScrolledToEnd) listState.scrollToItem(logs.size)
    }

    Column(Modifier.fillMaxSize()) {
        TopAppBar(
            title = { Text(stringResource(Res.string.about_title_logs, logs.size)) },
            windowInsets = AppBarDefaults.topAppBarWindowInsets,
            navigationIcon = { IconButton(onClick = onClickBack) { BackIcon() } },
            actions = {
                IconButton(onClick = { showFiltersDialog = true }) {
                    Box {
                        Icon(
                            painter = painterResource(Res.drawable.ic_filter_list_24),
                            contentDescription = stringResource(Res.string.action_filter)
                        )
                        if (filtersCount > 0) {
                            FiltersCounter(filtersCount, Modifier.align(Alignment.TopEnd))
                        }
                    }
                }
                val context = LocalContext.current
                IconButton(onClick = { context.shareLog(viewModel.logs.value.format()) }) {
                    Icon(
                        painter = painterResource(Res.drawable.ic_share_24),
                        contentDescription = stringResource(Res.string.action_share)
                    )
                }
            }
        )
        if (logs.isEmpty()) {
            CenteredLargeTitleHint(stringResource(Res.string.no_search_results))
        } else {
            val insets = WindowInsets.safeDrawing.only(
                WindowInsetsSides.Horizontal + WindowInsetsSides.Bottom
            ).asPaddingValues()
            LazyColumn(
                state = listState,
                contentPadding = insets,
                modifier = Modifier.consumeWindowInsets(insets)
            ) {
                itemsIndexed(logs) { index, item ->
                    if (index > 0) Divider()
                    LogsRow(item, modifier = Modifier.padding(horizontal = 16.dp, vertical = 8.dp))
                }
            }
        }
    }

    if (showFiltersDialog) {
        LogsFiltersDialog(
            initialFilters = filters,
            onDismissRequest = { showFiltersDialog = false },
            onApplyFilters = {
                showFiltersDialog = false
                viewModel.setFilters(it)
            }
        )
    }
}

@Composable
private fun FiltersCounter(count: Int, modifier: Modifier = Modifier) {
    Text(
        text = count.toString(),
        modifier = modifier
            .size(16.dp)
            .background(
                color = MaterialTheme.colors.secondary,
                shape = CircleShape
            ),
        textAlign = TextAlign.Center,
        style = MaterialTheme.typography.caption
    )
}

private fun Context.shareLog(logText: String) {
    val logTimestamp = LocalDateTime.now().toString()
    val logTitle = "${ApplicationConstants.NAME}_${BuildConfig.VERSION_NAME}_$logTimestamp.log"

    val shareIntent = Intent(Intent.ACTION_SEND).also {
        it.putExtra(Intent.EXTRA_TEXT, logText)
        it.putExtra(Intent.EXTRA_TITLE, logTitle)
        it.type = "text/plain"
    }

    startActivity(Intent.createChooser(shareIntent, null))
}
