import m, { Children, Component, Vnode } from "mithril"
import { Button, ButtonType } from "../gui/base/Button.js"
import { showUserError } from "../misc/ErrorHandlerImpl.js"
import { locator } from "../api/main/CommonLocator.js"
import { InfoLink } from "../misc/LanguageViewModel.js"
import { newMailEditorFromTemplate } from "../../mail-app/mail/editor/MailEditor.js"
import { UserError } from "../api/main/UserError.js"
import { clientInfoString, getLogAttachments } from "../misc/ErrorReporter.js"
import { ExternalLink } from "../gui/base/ExternalLink.js"
import { isApp } from "../api/common/Env.js"
import { px, size } from "../gui/size.js"
import { getTutaLogo } from "../gui/base/Logo.js"
import { prepareLogContent, showLogsDialog } from "../gui/LogDialogUtils"
import { client } from "../misc/ClientDetector"

interface AboutDialogAttrs {
	onShowSetupWizard: () => unknown
}

export class AboutDialog implements Component<AboutDialogAttrs> {
	view(vnode: Vnode<AboutDialogAttrs>): Children {
		return m(".flex.col", [
			m(".center.mt-16", "Powered by"),
			m(
				".center",
				// Our logo must be padded but at least a certain amount.
				// This might be a bit more than needed but it's safe.
				{
					style: {
						margin: px(size.spacing_48),
					},
				},
				m.trust(getTutaLogo()),
			),
			m(".flex.justify-center.flex-wrap", [
				m(ExternalLink, {
					href: InfoLink.HomePage,
					text: "Website",
					isCompanySite: true,
					specialType: "me",
					class: "mlr-12 mt-16",
				}),
				m(ExternalLink, {
					href: "https://github.com/tutao/tutanota/releases",
					text: "Releases",
					isCompanySite: false,
					class: "mlr-12 mt-16",
				}),
			]),
			m(".flex.justify-center.selectable.flex-wrap", [
				m("p.center.mt-16.mlr-12", `v${env.versionNumber}`),
				m("p.text-center.mlr-12", "GPL-v3"),
				m("p", "© 2025 Tutao GmbH"),
			]),
			this.logsLink(),
			// wrap it in a div so that it's not filling the whole width
			isApp()
				? m(
						"",
						m(Button, {
							label: "showWelcomeDialog_action",
							type: ButtonType.Primary,
							click: vnode.attrs.onShowSetupWizard,
						}),
					)
				: null,
		])
	}

	logsLink(): Children {
		if (client.isMailApp()) {
			// Sending logs opens a mail editor, we only want to do this in an app that handles mail
			return this.sendLogsLink()
		} else {
			return this.getLogsLink()
		}
	}

	sendLogsLink(): Children {
		return m(
			".mt-16",
			m(Button, {
				label: "sendLogs_action",
				click: () => this._sendDeviceLogs(),
				type: ButtonType.Primary,
			}),
		)
	}

	getLogsLink(): Children {
		return m(
			".mt-16",
			m(Button, {
				label: "getLogs_action",
				click: () => prepareLogContent().then((logInfo) => showLogsDialog(logInfo)),
				type: ButtonType.Primary,
			}),
		)
	}

	async _sendDeviceLogs(): Promise<void> {
		const timestamp = new Date()
		const attachments = await getLogAttachments(timestamp)
		const mailboxDetails = await locator.mailboxModel.getUserMailboxDetails()
		let { message, type, client } = clientInfoString(timestamp, true)
		message = message
			.split("\n")
			.filter(Boolean)
			.map((l) => `<div>${l}<br></div>`)
			.join("")

		try {
			const editor = await newMailEditorFromTemplate(
				mailboxDetails,
				{},
				`Device logs v${env.versionNumber} - ${type} - ${client}`,
				message,
				attachments,
				true,
			)
			editor?.show()
		} catch (e) {
			if (e instanceof UserError) {
				await showUserError(e)
			} else {
				throw e
			}
		}
	}
}
