import { NewsListItem } from "../NewsListItem.js"
import m, { Children } from "mithril"
import { NewsId } from "../../../api/entities/tutanota/TypeRefs.js"
import { Button, ButtonAttrs, ButtonType } from "../../../gui/base/Button.js"
import { NewsModel } from "../NewsModel.js"
import { getReferralLink, ReferralLinkViewer } from "./ReferralLinkViewer.js"
import { DateProvider } from "../../../api/common/DateProvider.js"
import { generatedIdToTimestamp } from "../../../api/common/utils/EntityUtils.js"
import { getDayShifted, neverNull } from "@tutao/tutanota-utils"
import { UserController } from "../../../api/main/UserController.js"

const REFERRAL_NEWS_DISPLAY_THRESHOLD_DAYS = 7

/**
 * News item that informs users about option to refer friends. Only shown after the customer exists at least 7 days.
 *
 * Not shown for non-admin users.
 */
export class ReferralLinkNews implements NewsListItem {
	private referralLink: string = ""

	constructor(
		private readonly newsModel: NewsModel,
		private readonly dateProvider: DateProvider,
		private readonly userController: UserController,
	) {}

	async isShown(): Promise<boolean> {
		// Do not show this for business customers yet (not allowed to create referral links)
		if ((await this.userController.loadCustomer()).businessUse === true) {
			return false
		}

		// Create the referral link
		this.referralLink = await getReferralLink(this.userController)

		// Decode the date the user was generated from the timestamp in the user ID
		const customerCreatedTime = generatedIdToTimestamp(neverNull(this.userController.user.customer))
		return (
			this.userController.isGlobalAdmin() &&
			getDayShifted(new Date(customerCreatedTime), REFERRAL_NEWS_DISPLAY_THRESHOLD_DAYS) <= new Date(this.dateProvider.now())
		)
	}

	render(newsId: NewsId): Children {
		const buttonAttrs: Array<ButtonAttrs> = [
			{
				label: "close_alt",
				click: () => this.newsModel.acknowledgeNews(newsId.newsItemId).then(m.redraw),
				type: ButtonType.Secondary,
			},
		]

		return m(".full-width", [
			m(ReferralLinkViewer, { referralLink: this.referralLink }),
			m(
				".flex-end.flex-no-grow-no-shrink-auto.flex-wrap",
				buttonAttrs.map((a) => m(Button, a)),
			),
		])
	}
}
