import m, { Component, Vnode } from "mithril"
import { BaseButton } from "./BaseButton.js"
import { AllIcons, Icon, IconSize } from "../Icon.js"
import { ClickHandler } from "../GuiUtils.js"
import { AriaRole } from "../../AriaUtils.js"
import { theme } from "../../theme.js"
import { lang, MaybeTranslation } from "../../../misc/LanguageViewModel.js"

export interface RowButtonAttrs {
	/** accessibility & tooltip description */
	label: MaybeTranslation
	/** visible text inside button */
	text?: MaybeTranslation
	icon?: AllIcons | "none"
	selected?: boolean
	onclick: ClickHandler
	style?: Record<string, any>
	class?: string
	role?: AriaRole
}

/** A button that is styled the same as a `NavButton`. */
export class RowButton implements Component<RowButtonAttrs> {
	view(vnode: Vnode<RowButtonAttrs>) {
		const attrs = vnode.attrs
		const label = lang.getTranslationText(attrs.label)
		const text = lang.getTranslationText(attrs.text ?? attrs.label)
		const color = attrs.selected ? theme.primary : theme.on_surface
		return m(BaseButton, {
			label: attrs.label,
			text: m(
				".plr-8.text-ellipsis",
				{
					style: { color },
					// When the label doesn't match content, screen readers read both
					ariaHidden: label !== text, // this prevents that
				},
				text,
			),
			role: attrs.role,
			selected: attrs.selected,
			icon:
				attrs.icon && attrs.icon !== "none"
					? m(Icon, {
							icon: attrs.icon,
							container: "div",
							class: "mr-8",
							style: { fill: color },
							size: IconSize.PX24,
						})
					: attrs.icon === "none"
						? m(".icon-24.mr-8")
						: null,
			class: "flex items-center state-bg button-content plr-8 " + attrs.class,
			style: {
				...attrs.style,
				color,
			},
			onclick: attrs.onclick,
		})
	}
}
