import { arrayEquals } from "./ArrayUtils.js";
import { deepEqual } from "./Utils.js";
/**
 * Function which accepts another function. On first invocation
 * of this resulting function result will be remembered and returned
 * on consequent invocations.
 */
export function lazyMemoized(source) {
    // Using separate variable for tracking because value can be undefined and we want to the function call only once
    let cached = false;
    let value;
    return () => {
        if (cached) {
            return value;
        }
        else {
            cached = true;
            return (value = source());
        }
    };
}
/**
 * Returns a cached version of {@param fn}.
 * Cached function checks that argument is the same (with ===) and if it is then it returns the cached result.
 * If the cached argument has changed then {@param fn} will be called with new argument and result will be cached again.
 * Only remembers the last argument.
 */
export function memoized(fn) {
    let lastArgs;
    let lastResult;
    let didCache = false;
    const memoizedFunction = (...args) => {
        if (!didCache || !arrayEquals(lastArgs, args)) {
            lastArgs = args;
            didCache = true;
            lastResult = fn(...args);
        }
        return lastResult;
    };
    return memoizedFunction;
}
/**
 * Returns a cached version of {@param fn}.
 * Cached function checks that argument is deeply the same and if it is then it returns the cached result.
 * If the cached argument has changed then {@param fn} will be called with new argument and result will be cached again.
 * Only remembers the last argument.
 */
export function deepMemoized(fn) {
    let lastArgs;
    let lastResult;
    let didCache = false;
    const memoizedFunction = (...args) => {
        if (!didCache || !deepEqual(lastArgs, args)) {
            lastArgs = args;
            didCache = true;
            lastResult = fn(...args);
        }
        return lastResult;
    };
    return memoizedFunction;
}
/**
 * Like {@link memoized} but the argument is passed in via {@param argumentProvider}.
 * Useful for the cases where we want to keep only one field around e.g. for lazy getters
 */
export function memoizedWithHiddenArgument(argumentProvider, computationFunction) {
    const memoizedComputation = memoized(computationFunction);
    return () => memoizedComputation(argumentProvider());
}
