/**
 * General interface for WASM exports, whether from native WASM or a fallback.
 */
export interface WASMExports {
    /**
     * Allocates a set number of bytes on the heap.
     *
     * This data will be permanently allocated until manually freed with free. Be careful with this function!
     *
     * See https://en.cppreference.com/w/c/memory/malloc
     *
     * @param len number of bytes
     * @return pointer to data or 0 if allocation failed
     */
    malloc(len: number): Ptr;
    /**
     * Frees data allocated with malloc.
     *
     * `what` MUST point to something allocated with malloc, or it must be 0 (which will be a no-op). Passing a pointer not allocated with malloc or a pointer
     * that was already freed (with the exception of 0 in both cases) will result in undefined behavior. Be careful with this function!
     *
     * See https://en.cppreference.com/w/c/memory/free
     *
     * @param what data to free or 0
     */
    free(what: Ptr): void;
    /**
     * WebAssembly memory/heap
     */
    memory: MemoryIF;
}
/**
 * Call the WebAssembly function with the given arguments.
 *
 * Automatically allocates strings and buffers and frees them while passing booleans and numbers as-is.
 *
 * @param func function to call
 * @param exports WASM module instance's exports
 * @param args arguments to pass
 *
 * @return return value of the function
 */
export declare function callWebAssemblyFunctionWithArguments<T>(func: (...args: number[]) => T, exports: WASMExports, ...args: (string | number | Uint8Array | Int8Array | MutableUint8Array | SecureFreeUint8Array | boolean | null)[]): T;
/**
 * Allocate memory on the heap of the WebAssembly instance.
 *
 * Be sure to call `free` on the byteOffset when you are done!
 *
 * @param length length of data to allocate
 * @param exports WASM module instance's exports
 */
export declare function allocateBuffer(length: number, exports: WASMExports): Uint8Array;
/**
 * Wrapper to be passed to a WebAssembly function.
 *
 * The contents of the array will be updated when the function finishes.
 */
export declare class MutableUint8Array {
    readonly uint8ArrayInputOutput: Uint8Array | SecureFreeUint8Array;
    constructor(uint8ArrayInputOutput: Uint8Array | SecureFreeUint8Array);
}
/**
 * Wrapper to be passed to a WebAssembly function.
 *
 * The copy allocated on the VM will be filled with zero bytes. This is slower, but it will ensure that its contents won't linger after being freed.
 *
 * Note that the buffer pointed to by uint8ArrayInput is *not* zeroed out automatically, as it is not a deep copy, so remember to zero out the original buffer
 * when you are done with it, too!
 */
export declare class SecureFreeUint8Array {
    readonly uint8ArrayInput: Uint8Array;
    constructor(uint8ArrayInput: Uint8Array);
}
/**
 * Convenience function for wrapping an array as a MutableUint8Array.
 *
 * Data from the WASM module will be copied back to the array once finished.
 * @param array array to wrap
 * @return wrapper
 */
export declare function mutable(array: Uint8Array): MutableUint8Array;
/**
 * Convenience function for wrapping an array as a MutableUint8Array and SecureFreeUint8Array.
 *
 * Data from the WASM module will be copied back to the array once finished, and then it will be erased from the module.
 * @param array array to wrap
 * @return wrapper
 */
export declare function mutableSecureFree(array: Uint8Array): MutableUint8Array;
/**
 * Convenience function for wrapping an array as a MutableUint8Array and SecureFreeUint8Array.
 *
 * Data from the WASM module will be erased once finished.
 * @param array array to wrap
 * @return wrapper
 */
export declare function secureFree(array: Uint8Array): SecureFreeUint8Array;
/**
 * Defines a pointer type
 */
export type Ptr = number;
/**
 * Defines a pointer type for immutable memory
 */
export type ConstPtr = number;
/**
 * Free function interface
 */
export type FreeFN = (what: Ptr) => void;
export type MemoryIF = WebAssembly.Memory;
