export interface ErrorInfo {
    readonly name: string | null;
    readonly message: string | null;
    readonly stack?: string | null;
}
export type lazy<T> = () => T;
export type lazyAsync<T> = () => Promise<T>;
export type Thunk = () => unknown;
/**
 * Integer constraint from 0 to n (using tail-recursion elimination)
 */
type Enumerate<N extends number, Acc extends number[] = []> = Acc["length"] extends N ? Acc[number] : Enumerate<N, [...Acc, Acc["length"]]>;
/**
 * A key version must be an integer between 0 and 100.
 *
 * The constraint to < 100 is arbitrary and must be changed when we rotate keys more often.
 */
export type KeyVersion = Enumerate<100>;
export declare function isKeyVersion(version: number): version is KeyVersion;
/**
 * A group key and its version.
 */
export type Versioned<T> = {
    object: T;
    version: KeyVersion;
};
/**
 * Create a versioned object with version 0
 */
export declare function freshVersioned<T>(object: T): Versioned<T>;
/** specifies a set of keys to be required, even if they're originally optional on a type.
 * requires nullable fields to be non-null, this may not be desired for all use cases.
 * Use "RequireNullable<K, T>" for cases where null is a meaningful value.
 *
 * `Require<"uid", Partial<CalendarEvent>>` */
export type Require<K extends keyof T, T> = T & {
    [P in K]-?: NonNullable<T[P]>;
};
export type DeferredObject<T> = {
    resolve: (arg0: T | PromiseLike<T>) => void;
    reject: (arg0: Error) => void;
    promise: Promise<T>;
};
export type DeferredObjectWithHandler<T, U> = {
    resolve: (arg0: T) => void;
    reject: (arg0: Error) => void;
    promise: Promise<U>;
};
export declare function defer<T>(): DeferredObject<T>;
export declare function deferWithHandler<T, U>(handler: (arg0: T) => U): DeferredObjectWithHandler<T, U>;
export declare function asyncFind<T>(array: ReadonlyArray<T>, finder: (item: T, index: number, arrayLength: number) => Promise<boolean>): Promise<T | null | undefined>;
export declare function asyncFindAndMap<T, R>(array: ReadonlyArray<T>, finder: (item: T, index: number, arrayLength: number) => Promise<R | null>): Promise<R | null | undefined>;
/**
 * Calls an executor function for slices of nbrOfElementsInGroup items of the given array until the executor function returns false.
 */
export declare function executeInGroups<T>(array: T[], nbrOfElementsInGroup: number, executor: (items: T[]) => Promise<boolean>): Promise<void>;
export declare function neverNull<T>(object: T): NonNullable<T>;
/**
 * returns its argument if it is not null, throws otherwise.
 * @param value the value to check
 * @param message optional error message
 */
export declare function assertNotNull<T>(value: T | null | undefined, message?: string): T;
/**
 * throws if the value is not null.
 * @param value the value to check
 * @param message optional error message
 */
export declare function assertNull<T>(value: T | null | undefined, message?: string): void;
/**
 * assertion function that only returns if the argument is non-null
 * (acts as a type guard)
 * @param value the value to check
 * @param message optional error message
 */
export declare function assertNonNull<T>(value: T | null | undefined, message?: string): asserts value is T;
export declare function isNotNull<T>(t: T | null | undefined): t is T;
export declare function assert(assertion: MaybeLazy<boolean>, message: string): void;
export declare function downcast<R = any>(object: any): R;
export declare function clone<T>(instance: T): T;
export type Callback<T> = (arg: T) => void;
/**
 * accept a function taking exactly one argument and returning nothing and return a version of it
 * that will call the original function on the first call and ignore any further calls.
 * @param fn a function taking one argument and returning nothing
 */
export declare function makeSingleUse<T>(fn: Callback<T>): Callback<T>;
/**
 * Function which returns what was passed into it
 */
export declare function identity<T>(t: T): T;
/**
 * Function which does nothing.
 */
export declare function noOp(): void;
/**
 * Return a function, which executed {@param toThrottle} only after it is not invoked for {@param timeout} ms.
 * Executes function with the last passed arguments
 * @return {Function}
 */
export declare function debounce<F extends (...args: any) => void>(timeout: number, toThrottle: F): F;
/**
 * Returns a debounced function. When invoked for the first time, will just invoke
 * {@param toThrottle}. On subsequent invocations it will either invoke it right away
 * (if {@param timeout} has passed) or will schedule it to be run after {@param timeout}.
 * So the first and the last invocations in a series of invocations always take place
 * but ones in the middle (which happen too often) are discarded.
 */
export declare function debounceStart<F extends (...args: any) => void>(timeout: number, toThrottle: F): F;
/**
 * Returns a throttled function. When invoked for the first time will schedule {@param toThrottle}
 * to be called after {@param periodMs}. On subsequent invocations before {@param periodMs} amount of
 * time passes it will replace the arguments for the scheduled call (without rescheduling). After
 * {@param period} amount of time passes it will finally call {@param toThrottle} with the arguments
 * of the last call. New calls after that will behave like described in the beginning.
 *
 * This makes sure that the function is called not more often but also at most after {@param periodMs}
 * amount of time. Unlike {@link debounce}, it will get called after {@param periodMs} even if it
 * is being called repeatedly.
 */
export declare function throttle<F extends (...args: any) => void>(periodMs: number, toThrottle: F): F;
/**
 * Returns a throttled function. On the first call it is called immediately. For subsequent calls if the next call
 * happens after {@param periodMs} it is invoked immediately. For subsequent calls it will schedule the function to
 * run after {@param periodMs} after the last run of {@param toThrottle}. Only one invocation is scheduled, with the
 * latest arguments.
 *
 * 1--2-34
 * 1---2---4
 *
 * In this case, the first invocation happens immediately. 2 happens shortly before the interval expires
 * so it is run at the end of the interval. Within the next interval, both 3 and 4 are called so at the end of the
 * interval only 4 is called.
 */
export declare function throttleStart<F extends (...args: any[]) => Promise<any>>(periodMs: number, toThrottle: F): F;
/**
 * Returns an async function that will only be executed once until it has settled. Subsequent calls will return the
 * original promise if it hasn't yet resolved.
 *
 * If the function throws before it can be awaited, it will not be caught.
 */
export declare function singleAsync<T, R>(fn: () => Promise<R>): () => Promise<R>;
export declare function randomIntFromInterval(min: number, max: number): number;
export declare function errorToString(error: ErrorInfo): string;
/**
 * Like {@link Object.entries} but preserves the type of the key and value
 */
export declare function objectEntries<A extends string | symbol, B>(object: Record<A, B>): Array<[A, B]>;
/**
 * modified deepEquals from ospec is only needed as long as we use custom classes (TypeRef) and Date is not properly handled
 */
export declare function deepEqual(a: any, b: any): boolean;
/**
 * returns an array of top-level properties that are in both objA and objB, but differ in value
 * does not handle functions or circular references
 * treats undefined and null as equal
 */
export declare function getChangedProps(objA: any, objB: any): Array<string>;
/**
 * Disallow set, delete and clear on Map.
 * Important: It is *not* a deep freeze.
 * @param myMap
 * @return {unknown}
 */
export declare function freezeMap<K, V>(myMap: ReadonlyMap<K, V>): ReadonlyMap<K, V>;
export declare function addressDomain(senderAddress: string): string;
/**
 * Ignores the fact that Object.keys returns also not owned properties.
 */
export declare function typedKeys<K extends string, V>(obj: Record<K, V>): Array<K>;
/**
 * Ignores the fact that Object.keys returns also not owned properties.
 */
export declare function typedEntries<K extends string, V>(obj: Record<K, V>): Array<[K, V]>;
/**
 * Ignores the fact that Object.keys returns also not owned properties.
 */
export declare function typedValues<K extends string, V>(obj: Record<K, V>): Array<V>;
export type MaybeLazy<T> = T | lazy<T>;
export declare function resolveMaybeLazy<T>(maybe: MaybeLazy<T>): T;
export declare function getAsLazy<T>(maybe: MaybeLazy<T>): lazy<T>;
export declare function mapLazily<T, U>(maybe: MaybeLazy<T>, mapping: (arg0: T) => U): lazy<U>;
/**
 * Stricter version of parseInt() from MDN. parseInt() allows some arbitrary characters at the end of the string.
 * Returns NaN in case there's anything non-number in the string.
 */
export declare function filterInt(value: string): number;
interface Positioned {
    x: number;
    y: number;
}
interface Sized {
    top: number;
    left: number;
    bottom: number;
    right: number;
}
export declare function insideRect(point: Positioned, rect: Sized): boolean;
/**
 * If val is non null, returns the result of val passed to action, else null
 */
export declare function mapNullable<T, U>(val: T | null | undefined, action: (arg0: T) => U | null | undefined): U | null;
/** Helper to take instead of `typeof setTimeout` which is hellish to reproduce */
export type TimeoutSetter = (fn: () => unknown, arg1: number) => ReturnType<typeof setTimeout>;
export declare function mapObject<K extends string | number | symbol, V, R>(mapper: (arg0: V) => R, obj: Record<K, V>): Record<K, R>;
/**
 * Run jobs with defined max parallelism.
 */
export declare class BoundedExecutor {
    private readonly maxParallelJobs;
    private runningJobsCount;
    private currentJob;
    constructor(maxParallelJobs: number);
    run<T>(job: () => Promise<T>): Promise<T>;
}
export declare function assertValidURL(url: string): false | URL;
/**
 * Excessive resizing of an observed element can result in one or more resize events being deferred to the next render cycle.
 * When this happens, the browser sends a `ResizeObserver loop completed with undelivered notifications` error.
 * To avoid this, we handle resize events in a `requestAnimationFrame` making sure to cancel any pending requests
 */
export declare function createResizeObserver(cb: ResizeObserverCallback): ResizeObserver;
export type Nullable<T> = T | null;
/**
 * Factory method to allow tracing unresolved promises.
 */
export declare function newPromise<T>(executor: (resolve: (value: T | PromiseLike<T>) => void, reject: (reason?: any) => void) => void, tag?: string): Promise<T>;
export declare function isSessionStorageAvailable(): boolean;
export declare function isAsciiChar(char: string): boolean;
export {};
