import type { lazyAsync } from "./Utils.js";
/**
 * A wrapper for an object that shall be lazy loaded asynchronously. If loading the object is triggered in parallel (getAsync()) the object is actually only loaded once but returned to all calls of getAsync().
 * If the object was loaded once it is not loaded again.
 */
export declare class LazyLoaded<T> {
    private readonly loadFunction;
    private defaultValue;
    private state;
    /**
     * @param loadFunction The function that actually loads the object as soon as getAsync() is called the first time.
     * @param defaultValue The value that shall be returned by getSync() or getLoaded() as long as the object is not loaded yet.
     */
    constructor(loadFunction: lazyAsync<T>, defaultValue?: T | null);
    /**
     * Returns a LazyLoaded object that has already been loaded and can be retrieved with getSync()
     */
    static newLoaded<T>(object: T): LazyLoaded<T>;
    load(): this;
    isLoaded(): boolean;
    isLoadedOrLoading(): boolean;
    /**
     * Loads the object if it is not loaded yet. May be called in parallel and takes care that the load function is only called once.
     */
    getAsync(): Promise<T>;
    /**
     * Returns null if the object is not loaded yet.
     */
    getSync(): T | null;
    /**
     * Only call this function if you know that the object is already loaded.
     */
    getLoaded(): T;
    /**
     * Removes the currently loaded object, so it will be loaded again with the next getAsync() call. Does not set any default value.
     */
    reset(): void;
    /**
     * Loads the object again and replaces the current one
     */
    reload(): Promise<T>;
}
