import otest from "@tutao/otest";
import * as td from "testdouble";
import { lastThrow, mapObject } from "@tutao/tutanota-utils";
/**
 * Mocks an attribute (function or object) on an object and makes sure that it can be restored to the original attribute by calling unmockAttribute() later.
 * Additionally creates a spy for the attribute if the attribute is a function.
 * @param object The object on which the attribute exists.
 * @param attributeOnObject The attribute to mock.
 * @param attributeMock The attribute mock.
 * @returns An object to be passed to unmockAttribute() in order to restore the original attribute.
 */
export function mockAttribute(object, attributeOnObject, attributeMock) {
    if (attributeOnObject == null)
        throw new Error("attributeOnObject is undefined");
    let attributeName = Object.getOwnPropertyNames(object).find((key) => object[key] === attributeOnObject);
    if (!attributeName) {
        attributeName = Object.getOwnPropertyNames(Object.getPrototypeOf(object)).find((key) => object[key] === attributeOnObject);
    }
    if (!attributeName) {
        throw new Error("attribute not found on object");
    }
    object[attributeName] = typeof attributeOnObject === "function" ? spy(attributeMock) : attributeMock;
    return {
        _originalObject: object,
        _originalAttribute: attributeOnObject,
        _attributeName: attributeName,
    };
}
export function unmockAttribute(mock) {
    mock._originalObject[mock._attributeName] = mock._originalAttribute;
}
export function spy(producer) {
    const invocations = [];
    // make it *non* arrow function so that it can be bound to objects
    const s = function (...args) {
        invocations.push(args);
        return producer && producer.apply(this, args);
    };
    s.invocations = invocations;
    Object.defineProperty(s, "callCount", {
        get() {
            return s.invocations.length;
        },
    });
    Object.defineProperty(s, "args", {
        get() {
            return lastThrow(s.invocations);
        },
    });
    Object.defineProperty(s, "calls", {
        get() {
            return s.invocations;
        },
    });
    return s;
}
/**
 * Create partial mock, i.e. allows mocking attributes or functions on actual instances
 * @param obj The base mock object on which mocker may overwrite attributes or functions
 * @param mocker This function receives obj and can overwrite attributes or functions.
 * @returns {T}
 */
export const mock = (obj, mocker) => {
    mocker(obj);
    return obj;
};
export function mapToObject(map) {
    const obj = {};
    for (const [key, value] of map.entries()) {
        obj[key] = value;
    }
    return obj;
}
export function replaceAllMaps(toReplace) {
    return toReplace instanceof Map
        ? replaceAllMaps(mapToObject(toReplace))
        : toReplace instanceof Array
            ? toReplace.map(replaceAllMaps)
            : toReplace != null && Object.getPrototypeOf(toReplace) === Object.prototype // plain object
                ? mapObject(replaceAllMaps, toReplace)
                : toReplace;
}
export async function assertThrows(expected, fn) {
    try {
        await fn();
    }
    catch (e) {
        otest.check(e instanceof expected).equals(true)("AssertThrows failed: Expected a " + expected.name + " to be thrown, but got a " + e);
        return e;
    }
    throw new Error("AssertThrows failed: Expected a " + expected + " to be thrown, but nothing was");
}
export function makeTimeoutMock() {
    let timeoutId = 1;
    let scheduledFn;
    const timeoutMock = function (fn) {
        scheduledFn = fn;
        timeoutId++;
        return timeoutId;
    };
    const spiedMock = spy(timeoutMock);
    spiedMock.next = function () {
        scheduledFn?.();
    };
    return spiedMock;
}
function delay(ms) {
    return new Promise((resolve) => {
        setTimeout(resolve, ms);
    });
}
/** Verify using testdouble, but register as an otest assertion */
export function verify(demonstration, config) {
    function check(demonstration) {
        try {
            td.verify(demonstration, config);
            return {
                pass: true,
                message: "Successful verification",
            };
        }
        catch (e) {
            return {
                pass: false,
                message: e.toString(),
            };
        }
    }
    otest(demonstration).satisfies(check);
}
export function throwsErrorWithMessage(errorClass, message) {
    return (fn) => {
        try {
            fn();
            return { pass: false, message: "Did not throw!" };
        }
        catch (e) {
            const pass = e instanceof errorClass && typeof e.message === "string" && e.message.includes(message);
            return { pass, message: `Error of type ${errorClass} w/ message ${message}, instead got ${e}` };
        }
    };
}
