"use strict";
const {
  __spreadValues,
  __export,
  __toESM,
  __toCommonJS,
  __async
} = require('./esblib.cjs');


// src/core.ts
var core_exports = {};
__export(core_exports, {
  $: () => $,
  ProcessOutput: () => ProcessOutput,
  ProcessPromise: () => ProcessPromise,
  cd: () => cd,
  defaults: () => defaults,
  kill: () => kill,
  log: () => log,
  syncProcessCwd: () => syncProcessCwd,
  useBash: () => useBash,
  usePowerShell: () => usePowerShell,
  usePwsh: () => usePwsh,
  within: () => within
});
module.exports = __toCommonJS(core_exports);
var import_node_assert = __toESM(require("assert"), 1);
var import_node_child_process = require("child_process");
var import_node_async_hooks = require("async_hooks");
var import_node_util = require("util");
var import_node_os = require("os");
var import_vendor_core = require("./vendor-core.cjs");
var import_util = require("./util.cjs");
var processCwd = Symbol("processCwd");
var syncExec = Symbol("syncExec");
var eol = Buffer.from(import_node_os.EOL);
var storage = new import_node_async_hooks.AsyncLocalStorage();
var cwdSyncHook;
function syncProcessCwd(flag = true) {
  cwdSyncHook = cwdSyncHook || (0, import_node_async_hooks.createHook)({
    init: syncCwd,
    before: syncCwd,
    promiseResolve: syncCwd,
    after: syncCwd,
    destroy: syncCwd
  });
  if (flag) cwdSyncHook.enable();
  else cwdSyncHook.disable();
}
var defaults = {
  [processCwd]: process.cwd(),
  [syncExec]: false,
  verbose: false,
  env: process.env,
  sync: false,
  shell: true,
  stdio: ["inherit", "pipe", "pipe"],
  nothrow: false,
  quiet: false,
  prefix: "",
  postfix: "",
  detached: false,
  preferLocal: false,
  spawn: import_node_child_process.spawn,
  spawnSync: import_node_child_process.spawnSync,
  log,
  kill
};
function usePowerShell() {
  $.shell = import_vendor_core.which.sync("powershell.exe");
  $.prefix = "";
  $.postfix = "; exit $LastExitCode";
  $.quote = import_util.quotePowerShell;
}
function usePwsh() {
  $.shell = import_vendor_core.which.sync("pwsh");
  $.prefix = "";
  $.postfix = "; exit $LastExitCode";
  $.quote = import_util.quotePowerShell;
}
function useBash() {
  $.shell = import_vendor_core.which.sync("bash");
  $.prefix = "set -euo pipefail;";
  $.postfix = "";
  $.quote = import_util.quote;
}
function checkShell() {
  if (!$.shell)
    throw new Error(`No shell is available: https://\xEF.at/zx-no-shell`);
}
function checkQuote() {
  if (!$.quote)
    throw new Error("No quote function is defined: https://\xEF.at/no-quote-func");
}
function getStore() {
  return storage.getStore() || defaults;
}
var $ = new Proxy(
  function(pieces, ...args) {
    if (!Array.isArray(pieces)) {
      return function(...args2) {
        const self = this;
        return within(() => {
          return Object.assign($, pieces).apply(self, args2);
        });
      };
    }
    const from = (0, import_util.getCallerLocation)();
    if (pieces.some((p) => p == void 0)) {
      throw new Error(`Malformed command at ${from}`);
    }
    checkShell();
    checkQuote();
    let resolve, reject;
    const promise = new ProcessPromise((...args2) => [resolve, reject] = args2);
    const cmd = (0, import_vendor_core.buildCmd)(
      $.quote,
      pieces,
      args
    );
    const snapshot = getStore();
    const sync = snapshot[syncExec];
    const callback = () => promise.isHalted() || promise.run();
    promise._bind(
      cmd,
      from,
      resolve,
      (v) => {
        reject(v);
        if (sync) throw v;
      },
      snapshot
    );
    sync ? callback() : setImmediate(callback);
    return sync ? promise.output : promise;
  },
  {
    set(_, key, value) {
      const target = key in Function.prototype ? _ : getStore();
      Reflect.set(target, key === "sync" ? syncExec : key, value);
      return true;
    },
    get(_, key) {
      if (key === "sync") return $({ sync: true });
      const target = key in Function.prototype ? _ : getStore();
      return Reflect.get(target, key);
    }
  }
);
try {
  useBash();
} catch (err) {
}
var ProcessPromise = class _ProcessPromise extends Promise {
  constructor() {
    super(...arguments);
    this._command = "";
    this._from = "";
    this._resolve = import_util.noop;
    this._reject = import_util.noop;
    this._snapshot = getStore();
    this._timeoutSignal = "SIGTERM";
    this._resolved = false;
    this._halted = false;
    this._piped = false;
    this._zurk = null;
    this._output = null;
    this._prerun = import_util.noop;
    this._postrun = import_util.noop;
  }
  _bind(cmd, from, resolve, reject, options) {
    this._command = cmd;
    this._from = from;
    this._resolve = resolve;
    this._reject = reject;
    this._snapshot = __spreadValues({ ac: new AbortController() }, options);
  }
  run() {
    var _a, _b, _c, _d;
    if (this.child) return this;
    this._prerun();
    const $2 = this._snapshot;
    const self = this;
    const input = (_b = (_a = $2.input) == null ? void 0 : _a.stdout) != null ? _b : $2.input;
    if (input) this.stdio("pipe");
    if ($2.timeout) this.timeout($2.timeout, $2.timeoutSignal);
    if ($2.preferLocal) $2.env = (0, import_util.preferNmBin)($2.env, $2.cwd, $2[processCwd]);
    $2.log({
      kind: "cmd",
      cmd: this._command,
      verbose: self.isVerbose()
    });
    this._zurk = (0, import_vendor_core.exec)({
      input,
      cmd: $2.prefix + self._command + $2.postfix,
      cwd: (_c = $2.cwd) != null ? _c : $2[processCwd],
      ac: $2.ac,
      signal: $2.signal,
      shell: typeof $2.shell === "string" ? $2.shell : true,
      env: $2.env,
      spawn: $2.spawn,
      spawnSync: $2.spawnSync,
      store: $2.store,
      stdio: (_d = self._stdio) != null ? _d : $2.stdio,
      sync: $2[syncExec],
      detached: $2.detached,
      run: (cb) => cb(),
      on: {
        start: () => {
          if (self._timeout) {
            const t = setTimeout(
              () => self.kill(self._timeoutSignal),
              self._timeout
            );
            self.finally(() => clearTimeout(t)).catch(import_util.noop);
          }
        },
        stdout: (data) => {
          if (self._piped) return;
          $2.log({ kind: "stdout", data, verbose: self.isVerbose() });
        },
        stderr: (data) => {
          $2.log({ kind: "stderr", data, verbose: !self.isQuiet() });
        },
        end: ({ error, stdout, stderr, stdall, status, signal }, c) => {
          var _a2, _b2, _c2, _d2;
          self._resolved = true;
          if (stderr && !stderr.endsWith("\n")) (_b2 = (_a2 = c.on).stderr) == null ? void 0 : _b2.call(_a2, eol, c);
          if (stdout && !stdout.endsWith("\n")) (_d2 = (_c2 = c.on).stdout) == null ? void 0 : _d2.call(_c2, eol, c);
          if (error) {
            const message = ProcessOutput.getErrorMessage(error, self._from);
            const output = new ProcessOutput(
              null,
              null,
              stdout,
              stderr,
              stdall,
              message
            );
            self._output = output;
            self._reject(output);
          } else {
            const message = ProcessOutput.getExitMessage(
              status,
              signal,
              stderr,
              self._from
            );
            const output = new ProcessOutput(
              status,
              signal,
              stdout,
              stderr,
              stdall,
              message
            );
            self._output = output;
            if (status === 0 || self.isNothrow()) {
              self._resolve(output);
            } else {
              self._reject(output);
            }
          }
        }
      }
    });
    this._postrun();
    return this;
  }
  get child() {
    var _a;
    return (_a = this._zurk) == null ? void 0 : _a.child;
  }
  get stdin() {
    this.stdio("pipe");
    this.run();
    (0, import_node_assert.default)(this.child);
    if (this.child.stdin == null)
      throw new Error("The stdin of subprocess is null.");
    return this.child.stdin;
  }
  get stdout() {
    this.run();
    (0, import_node_assert.default)(this.child);
    if (this.child.stdout == null)
      throw new Error("The stdout of subprocess is null.");
    return this.child.stdout;
  }
  get stderr() {
    this.run();
    (0, import_node_assert.default)(this.child);
    if (this.child.stderr == null)
      throw new Error("The stderr of subprocess is null.");
    return this.child.stderr;
  }
  get exitCode() {
    return this.then(
      (p) => p.exitCode,
      (p) => p.exitCode
    );
  }
  json() {
    return this.then((p) => p.json());
  }
  text(encoding) {
    return this.then((p) => p.text(encoding));
  }
  lines() {
    return this.then((p) => p.lines());
  }
  buffer() {
    return this.then((p) => p.buffer());
  }
  blob(type) {
    return this.then((p) => p.blob(type));
  }
  then(onfulfilled, onrejected) {
    if (this.isHalted() && !this.child) {
      throw new Error("The process is halted!");
    }
    return super.then(onfulfilled, onrejected);
  }
  catch(onrejected) {
    return super.catch(onrejected);
  }
  pipe(dest) {
    if ((0, import_util.isString)(dest))
      throw new Error("The pipe() method does not take strings. Forgot $?");
    if (this._resolved) {
      if (dest instanceof _ProcessPromise) dest.stdin.end();
      throw new Error(
        "The pipe() method shouldn't be called after promise is already resolved!"
      );
    }
    this._piped = true;
    if (dest instanceof _ProcessPromise) {
      dest.stdio("pipe");
      dest._prerun = this.run.bind(this);
      dest._postrun = () => {
        if (!dest.child)
          throw new Error(
            "Access to stdin of pipe destination without creation a subprocess."
          );
        this.stdout.pipe(dest.stdin);
      };
      return dest;
    } else {
      this._postrun = () => this.stdout.pipe(dest);
      return this;
    }
  }
  abort(reason) {
    var _a, _b;
    if (this.signal !== ((_a = this._snapshot.ac) == null ? void 0 : _a.signal))
      throw new Error("The signal is controlled by another process.");
    if (!this.child)
      throw new Error("Trying to abort a process without creating one.");
    (_b = this._zurk) == null ? void 0 : _b.ac.abort(reason);
  }
  get signal() {
    var _a;
    return this._snapshot.signal || ((_a = this._snapshot.ac) == null ? void 0 : _a.signal);
  }
  kill(signal = "SIGTERM") {
    return __async(this, null, function* () {
      if (!this.child)
        throw new Error("Trying to kill a process without creating one.");
      if (!this.child.pid) throw new Error("The process pid is undefined.");
      return $.kill(this.child.pid, signal);
    });
  }
  stdio(stdin, stdout = "pipe", stderr = "pipe") {
    this._stdio = [stdin, stdout, stderr];
    return this;
  }
  nothrow() {
    this._nothrow = true;
    return this;
  }
  quiet(v = true) {
    this._quiet = v;
    return this;
  }
  verbose(v = true) {
    this._verbose = v;
    return this;
  }
  isQuiet() {
    var _a;
    return (_a = this._quiet) != null ? _a : this._snapshot.quiet;
  }
  isVerbose() {
    var _a;
    return ((_a = this._verbose) != null ? _a : this._snapshot.verbose) && !this.isQuiet();
  }
  isNothrow() {
    var _a;
    return (_a = this._nothrow) != null ? _a : this._snapshot.nothrow;
  }
  timeout(d, signal = "SIGTERM") {
    this._timeout = (0, import_util.parseDuration)(d);
    this._timeoutSignal = signal;
    return this;
  }
  halt() {
    this._halted = true;
    return this;
  }
  isHalted() {
    return this._halted;
  }
  get output() {
    return this._output;
  }
};
var ProcessOutput = class extends Error {
  constructor(code, signal, stdout, stderr, combined, message) {
    super(message);
    this._code = code;
    this._signal = signal;
    this._stdout = stdout;
    this._stderr = stderr;
    this._combined = combined;
  }
  toString() {
    return this._combined;
  }
  json() {
    return JSON.parse(this._combined);
  }
  buffer() {
    return Buffer.from(this._combined);
  }
  blob(type = "text/plain") {
    if (!globalThis.Blob)
      throw new Error(
        "Blob is not supported in this environment. Provide a polyfill"
      );
    return new Blob([this.buffer()], { type });
  }
  text(encoding = "utf8") {
    return encoding === "utf8" ? this.toString() : this.buffer().toString(encoding);
  }
  lines() {
    return this.valueOf().split(/\r?\n/);
  }
  valueOf() {
    return this._combined.trim();
  }
  get stdout() {
    return this._stdout;
  }
  get stderr() {
    return this._stderr;
  }
  get exitCode() {
    return this._code;
  }
  get signal() {
    return this._signal;
  }
  static getExitMessage(code, signal, stderr, from) {
    let message = `exit code: ${code}`;
    if (code != 0 || signal != null) {
      message = `${stderr || "\n"}    at ${from}`;
      message += `
    exit code: ${code}${(0, import_util.exitCodeInfo)(code) ? " (" + (0, import_util.exitCodeInfo)(code) + ")" : ""}`;
      if (signal != null) {
        message += `
    signal: ${signal}`;
      }
    }
    return message;
  }
  static getErrorMessage(err, from) {
    return `${err.message}
    errno: ${err.errno} (${(0, import_util.errnoMessage)(err.errno)})
    code: ${err.code}
    at ${from}`;
  }
  [import_node_util.inspect.custom]() {
    let stringify = (s, c) => s.length === 0 ? "''" : c((0, import_node_util.inspect)(s));
    return `ProcessOutput {
  stdout: ${stringify(this.stdout, import_vendor_core.chalk.green)},
  stderr: ${stringify(this.stderr, import_vendor_core.chalk.red)},
  signal: ${(0, import_node_util.inspect)(this.signal)},
  exitCode: ${(this.exitCode === 0 ? import_vendor_core.chalk.green : import_vendor_core.chalk.red)(this.exitCode)}${(0, import_util.exitCodeInfo)(this.exitCode) ? import_vendor_core.chalk.grey(" (" + (0, import_util.exitCodeInfo)(this.exitCode) + ")") : ""}
}`;
  }
};
function within(callback) {
  return storage.run(__spreadValues({}, getStore()), callback);
}
function syncCwd() {
  if ($[processCwd] != process.cwd()) process.chdir($[processCwd]);
}
function cd(dir) {
  if (dir instanceof ProcessOutput) {
    dir = dir.toString().trim();
  }
  $.log({ kind: "cd", dir });
  process.chdir(dir);
  $[processCwd] = process.cwd();
}
function kill(pid, signal) {
  return __async(this, null, function* () {
    let children = yield import_vendor_core.ps.tree({ pid, recursive: true });
    for (const p of children) {
      try {
        process.kill(+p.pid, signal);
      } catch (e) {
      }
    }
    try {
      process.kill(-pid, signal);
    } catch (e) {
      try {
        process.kill(+pid, signal);
      } catch (e2) {
      }
    }
  });
}
function log(entry) {
  var _a;
  if (!((_a = entry.verbose) != null ? _a : $.verbose)) return;
  switch (entry.kind) {
    case "cmd":
      process.stderr.write((0, import_util.formatCmd)(entry.cmd));
      break;
    case "stdout":
    case "stderr":
    case "custom":
      process.stderr.write(entry.data);
      break;
    case "cd":
      process.stderr.write("$ " + import_vendor_core.chalk.greenBright("cd") + ` ${entry.dir}
`);
      break;
    case "fetch":
      const init = entry.init ? " " + (0, import_node_util.inspect)(entry.init) : "";
      process.stderr.write(
        "$ " + import_vendor_core.chalk.greenBright("fetch") + ` ${entry.url}${init}
`
      );
      break;
    case "retry":
      process.stderr.write(entry.error + "\n");
  }
}
/* c8 ignore next 100 */
// Annotate the CommonJS export names for ESM import in node:
0 && (module.exports = {
  $,
  ProcessOutput,
  ProcessPromise,
  cd,
  defaults,
  kill,
  log,
  syncProcessCwd,
  useBash,
  usePowerShell,
  usePwsh,
  within
});