#!/usr/bin/env node
"use strict";
const {
  __export,
  __toESM,
  __toCommonJS,
  __async,
  __forAwait
} = require('./esblib.cjs');

const import_meta_url =
  typeof document === 'undefined'
    ? new (require('url'.replace('', '')).URL)('file:' + __filename).href
    : (document.currentScript && document.currentScript.src) ||
      new URL('main.js', document.baseURI).href


// src/cli.ts
var cli_exports = {};
__export(cli_exports, {
  argv: () => argv,
  getVersion: () => getVersion,
  importPath: () => importPath,
  injectGlobalRequire: () => injectGlobalRequire,
  isMain: () => isMain,
  main: () => main,
  printUsage: () => printUsage,
  runScript: () => runScript,
  scriptFromHttp: () => scriptFromHttp,
  scriptFromStdin: () => scriptFromStdin,
  transformMarkdown: () => transformMarkdown,
  writeAndImport: () => writeAndImport
});
module.exports = __toCommonJS(cli_exports);
var import_node_url = __toESM(require("url"), 1);
var import_index = require("./index.cjs");
var import_deps = require("./deps.cjs");
var import_util = require("./util.cjs");
var import_vendor = require("./vendor.cjs");
var import_meta = {};
isMain() && main().catch((err) => {
  if (err instanceof import_index.ProcessOutput) {
    console.error("Error:", err.message);
  } else {
    console.error(err);
  }
  process.exitCode = 1;
});
function printUsage() {
  console.log(`
 ${import_index.chalk.bold("zx " + getVersion())}
   A tool for writing better scripts

 ${import_index.chalk.bold("Usage")}
   zx [options] <script>

 ${import_index.chalk.bold("Options")}
   --quiet              suppress any outputs
   --verbose            enable verbose mode
   --shell=<path>       custom shell binary
   --prefix=<command>   prefix all commands
   --postfix=<command>  postfix all commands
   --cwd=<path>         set current directory
   --eval=<js>, -e      evaluate script 
   --install, -i        install dependencies
   --version, -v        print current zx version
   --help, -h           print help
   --repl               start repl
   --experimental       enables experimental features (deprecated)
 
 ${import_index.chalk.italic("Full documentation:")} ${import_index.chalk.underline("https://google.github.io/zx/")}
`);
}
var argv = (0, import_index.minimist)(process.argv.slice(2), {
  string: ["shell", "prefix", "postfix", "eval", "cwd"],
  boolean: [
    "version",
    "help",
    "quiet",
    "verbose",
    "install",
    "repl",
    "experimental"
  ],
  alias: { e: "eval", i: "install", v: "version", h: "help" },
  stopEarly: true
});
function main() {
  return __async(this, null, function* () {
    yield import("./globals.cjs");
    if (argv.cwd) import_index.$.cwd = argv.cwd;
    if (argv.verbose) import_index.$.verbose = true;
    if (argv.quiet) import_index.$.quiet = true;
    if (argv.shell) import_index.$.shell = argv.shell;
    if (argv.prefix) import_index.$.prefix = argv.prefix;
    if (argv.postfix) import_index.$.postfix = argv.postfix;
    if (argv.version) {
      console.log(getVersion());
      return;
    }
    if (argv.help) {
      printUsage();
      return;
    }
    if (argv.repl) {
      yield (yield import("./repl.cjs")).startRepl();
      return;
    }
    if (argv.eval) {
      yield runScript(argv.eval);
      return;
    }
    const firstArg = argv._[0];
    (0, import_index.updateArgv)(argv._.slice(firstArg === void 0 ? 0 : 1));
    if (!firstArg || firstArg === "-") {
      const success = yield scriptFromStdin();
      if (!success) {
        printUsage();
        process.exitCode = 1;
      }
      return;
    }
    if (/^https?:/.test(firstArg)) {
      yield scriptFromHttp(firstArg);
      return;
    }
    const filepath = firstArg.startsWith("file:///") ? import_node_url.default.fileURLToPath(firstArg) : import_index.path.resolve(firstArg);
    yield importPath(filepath);
  });
}
function runScript(script) {
  return __async(this, null, function* () {
    var _a;
    const filepath = import_index.path.join((_a = import_index.$.cwd) != null ? _a : process.cwd(), `zx-${(0, import_util.randomId)()}.mjs`);
    yield writeAndImport(script, filepath);
  });
}
function scriptFromStdin() {
  return __async(this, null, function* () {
    let script = "";
    if (!process.stdin.isTTY) {
      process.stdin.setEncoding("utf8");
      try {
        for (var iter = __forAwait(process.stdin), more, temp, error; more = !(temp = yield iter.next()).done; more = false) {
          const chunk = temp.value;
          script += chunk;
        }
      } catch (temp) {
        error = [temp];
      } finally {
        try {
          more && (temp = iter.return) && (yield temp.call(iter));
        } finally {
          if (error)
            throw error[0];
        }
      }
      if (script.length > 0) {
        yield runScript(script);
        return true;
      }
    }
    return false;
  });
}
function scriptFromHttp(remote) {
  return __async(this, null, function* () {
    var _a;
    const res = yield (0, import_index.fetch)(remote);
    if (!res.ok) {
      console.error(`Error: Can't get ${remote}`);
      process.exit(1);
    }
    const script = yield res.text();
    const pathname = new URL(remote).pathname;
    const name = import_index.path.basename(pathname);
    const ext = import_index.path.extname(pathname) || ".mjs";
    const cwd = (_a = import_index.$.cwd) != null ? _a : process.cwd();
    const filepath = import_index.path.join(cwd, `${name}-${(0, import_util.randomId)()}${ext}`);
    yield writeAndImport(script, filepath);
  });
}
function writeAndImport(_0, _1) {
  return __async(this, arguments, function* (script, filepath, origin = filepath) {
    yield import_index.fs.writeFile(filepath, script.toString());
    try {
      yield importPath(filepath, origin);
    } finally {
      yield import_index.fs.rm(filepath);
    }
  });
}
function importPath(_0) {
  return __async(this, arguments, function* (filepath, origin = filepath) {
    const ext = import_index.path.extname(filepath);
    const base = import_index.path.basename(filepath);
    const dir = import_index.path.dirname(filepath);
    if (ext === "") {
      const tmpFilename = import_index.fs.existsSync(`${filepath}.mjs`) ? `${base}-${(0, import_util.randomId)()}.mjs` : `${base}.mjs`;
      return writeAndImport(
        yield import_index.fs.readFile(filepath),
        import_index.path.join(dir, tmpFilename),
        origin
      );
    }
    if (ext === ".md") {
      return writeAndImport(
        transformMarkdown(yield import_index.fs.readFile(filepath)),
        import_index.path.join(dir, base + ".mjs"),
        origin
      );
    }
    if (argv.install) {
      const deps = (0, import_deps.parseDeps)(yield import_index.fs.readFile(filepath));
      yield (0, import_deps.installDeps)(deps, dir);
    }
    injectGlobalRequire(origin);
    yield import(import_node_url.default.pathToFileURL(filepath).toString());
  });
}
function injectGlobalRequire(origin) {
  const __filename = import_index.path.resolve(origin);
  const __dirname = import_index.path.dirname(__filename);
  const require2 = (0, import_vendor.createRequire)(origin);
  Object.assign(globalThis, { __filename, __dirname, require: require2 });
}
function transformMarkdown(buf) {
  const source = buf.toString();
  const output = [];
  let state = "root";
  let codeBlockEnd = "";
  let prevLineIsEmpty = true;
  const jsCodeBlock = /^(```+|~~~+)(js|javascript)$/;
  const shCodeBlock = /^(```+|~~~+)(sh|bash)$/;
  const otherCodeBlock = /^(```+|~~~+)(.*)$/;
  for (let line of source.split(/\r?\n/)) {
    switch (state) {
      case "root":
        if (/^( {4}|\t)/.test(line) && prevLineIsEmpty) {
          output.push(line);
          state = "tab";
        } else if (jsCodeBlock.test(line)) {
          output.push("");
          state = "js";
          codeBlockEnd = line.match(jsCodeBlock)[1];
        } else if (shCodeBlock.test(line)) {
          output.push("await $`");
          state = "bash";
          codeBlockEnd = line.match(shCodeBlock)[1];
        } else if (otherCodeBlock.test(line)) {
          output.push("");
          state = "other";
          codeBlockEnd = line.match(otherCodeBlock)[1];
        } else {
          prevLineIsEmpty = line === "";
          output.push("// " + line);
        }
        break;
      case "tab":
        if (/^( +|\t)/.test(line)) {
          output.push(line);
        } else if (line === "") {
          output.push("");
        } else {
          output.push("// " + line);
          state = "root";
        }
        break;
      case "js":
        if (line === codeBlockEnd) {
          output.push("");
          state = "root";
        } else {
          output.push(line);
        }
        break;
      case "bash":
        if (line === codeBlockEnd) {
          output.push("`");
          state = "root";
        } else {
          output.push(line);
        }
        break;
      case "other":
        if (line === codeBlockEnd) {
          output.push("");
          state = "root";
        } else {
          output.push("// " + line);
        }
        break;
    }
  }
  return output.join("\n");
}
function getVersion() {
  return (0, import_vendor.createRequire)(import_meta_url)("../package.json").version;
}
function isMain(metaurl = import_meta_url, scriptpath = process.argv[1]) {
  if (metaurl.startsWith("file:")) {
    const modulePath = import_node_url.default.fileURLToPath(metaurl).replace(/\.\w+$/, "");
    const mainPath = import_index.fs.realpathSync(scriptpath).replace(/\.\w+$/, "");
    return mainPath === modulePath;
  }
  return false;
}
/* c8 ignore next 100 */
// Annotate the CommonJS export names for ESM import in node:
0 && (module.exports = {
  argv,
  getVersion,
  importPath,
  injectGlobalRequire,
  isMain,
  main,
  printUsage,
  runScript,
  scriptFromHttp,
  scriptFromStdin,
  transformMarkdown,
  writeAndImport
});