export type X25519PrivateKey = Uint8Array;
export type X25519PublicKey = Uint8Array;
/**
 * Contains a public key and its corresponding private key
 *
 * NOTE: Keys should be cleared from memory once they are no longer needed!
 */
export type X25519KeyPair = {
    publicKey: X25519PublicKey;
    privateKey: X25519PrivateKey;
};
/**
 * Contains all information for deriving AES keys
 *
 * The shared secret MUST NEVER be used as a key directly as it is a biased (some bits are more likely to be set than others).
 * The involved public keys should also be included when deriving an AES key from these shared secrets.
 */
export type X25519SharedSecrets = {
    ephemeralSharedSecret: Uint8Array;
    authSharedSecret: Uint8Array;
};
/**
 * @return randomly generated X25519 key pair
 */
export declare function generateX25519KeyPair(): X25519KeyPair;
/**
 * Derive a shared secret from the sender's private key and the recipient's public key to encrypt a message
 * @param senderIdentityPrivateKey    the sender's private identity key
 * @param ephemeralPrivateKey  the ephemeral private key generated by the sender for just one message (to one or more recipients)
 * @param recipientIdentityPublicKey the recipient's public identity key
 * @return the shared secrets
 */
export declare function x25519Encapsulate(senderIdentityPrivateKey: X25519PrivateKey, ephemeralPrivateKey: X25519PrivateKey, recipientIdentityPublicKey: X25519PublicKey): X25519SharedSecrets;
/**
 * Derive a shared secret from the recipient's private key and the sender's public key to decrypt a message
 * @param senderIdentityPublicKey    the sender's public identity key
 * @param ephemeralPublicKey  the ephemeral public key generated by the sender for just one message (to one or more recipients)
 * @param recipientIdentityPrivateKey the recipient's private identity key
 * @return shared secret and the sender's public key
 */
export declare function x25519Decapsulate(senderIdentityPublicKey: X25519PublicKey, ephemeralPublicKey: X25519PublicKey, recipientIdentityPrivateKey: X25519PrivateKey): X25519SharedSecrets;
/**
 * Derives the public key from the private key, without any integrity checks.
 * @param privateKey a valid clamped private key
 */
export declare function deriveX25519PublicKey(privateKey: X25519PrivateKey): X25519PublicKey;
