import { BitArray } from "../misc/Utils.js";
import { Base64 } from "@tutao/tutanota-utils";
export declare const ENABLE_MAC = true;
export declare const IV_BYTE_LENGTH = 16;
export declare const KEY_LENGTH_BYTES_AES_256 = 32;
export declare const KEY_LENGTH_BITS_AES_256: number;
export declare const KEY_LENGTH_BYTES_AES_128 = 16;
export declare const MAC_ENABLED_PREFIX = 1;
export type Aes256Key = BitArray;
export type Aes128Key = BitArray;
export type AesKey = Aes128Key | Aes256Key;
/**
 * @return the key length in bytes
 */
export declare function getKeyLengthBytes(key: AesKey): number;
export declare function aes256RandomKey(): Aes256Key;
export declare function generateIV(): Uint8Array;
/**
 * Encrypts bytes with AES128 or AES256 in CBC mode.
 * @param key The key to use for the encryption.
 * @param bytes The plain text.
 * @param iv The initialization vector.
 * @param usePadding If true, padding is used, otherwise no padding is used and the encrypted data must have the key size.
 * @param useMac If true, use HMAC (note that this is required for AES-256)
 * @return The encrypted bytes
 */
export declare function aesEncrypt(key: AesKey, bytes: Uint8Array, iv?: Uint8Array, usePadding?: boolean, useMac?: boolean): Uint8Array<ArrayBufferLike>;
/**
 * Encrypts bytes with AES 256 in CBC mode without mac. This is legacy code and should be removed once the index has been migrated.
 * @param key The key to use for the encryption.
 * @param bytes The plain text.
 * @param iv The initialization vector (only to be passed for testing).
 * @param usePadding If true, padding is used, otherwise no padding is used and the encrypted data must have the key size.
 * @return The encrypted text as words (sjcl internal structure)..
 */
export declare function aes256EncryptSearchIndexEntry(key: Aes256Key, bytes: Uint8Array, iv?: Uint8Array, usePadding?: boolean): Uint8Array;
/**
 * Decrypts the given words with AES-128/256 in CBC mode (with HMAC-SHA-256 as mac). The mac is enforced for AES-256 but optional for AES-128.
 * @param key The key to use for the decryption.
 * @param encryptedBytes The ciphertext encoded as bytes.
 * @param usePadding If true, padding is used, otherwise no padding is used and the encrypted data must have the key size.
 * @return The decrypted bytes.
 */
export declare function aesDecrypt(key: AesKey, encryptedBytes: Uint8Array, usePadding?: boolean): Uint8Array;
/**
 * Decrypts the given words with AES-128/ AES-256 in CBC mode with HMAC-SHA-256 as mac. Enforces the mac.
 * @param key The key to use for the decryption.
 * @param encryptedBytes The ciphertext encoded as bytes.
 * @param usePadding If true, padding is used, otherwise no padding is used and the encrypted data must have the key size.
 * @return The decrypted bytes.
 */
export declare function authenticatedAesDecrypt(key: AesKey, encryptedBytes: Uint8Array, usePadding?: boolean): Uint8Array;
/**
 * Decrypts the given words with AES-128/256 in CBC mode. Does not enforce a mac.
 * We always must enforce macs. This only exists for backward compatibility in some exceptional cases like search index entry encryption.
 *
 * @param key The key to use for the decryption.
 * @param encryptedBytes The ciphertext encoded as bytes.
 * @param usePadding If true, padding is used, otherwise no padding is used and the encrypted data must have the key size.
 * @return The decrypted bytes.
 */
export declare function unauthenticatedAesDecrypt(key: Aes256Key, encryptedBytes: Uint8Array, usePadding?: boolean): Uint8Array;
export declare function verifyKeySize(key: AesKey, bitLength: number[]): void;
/************************ Legacy AES128 ************************/
/**
 * @private visible for tests
 * @deprecated
 * */
export declare function _aes128RandomKey(): Aes128Key;
export declare function getAesSubKeys(key: AesKey, mac: boolean): {
    mKey: AesKey | null | undefined;
    cKey: AesKey;
};
export declare function extractIvFromCipherText(encrypted: Base64): Uint8Array;
