import path from "node:path";
import { generateImportCode, runCommand } from "./WasmHandler.js";
import * as fs from "node:fs";
function parseLibraries(libraries) {
    if (!Array.isArray(libraries)) {
        throw new Error(`Invalid webassemblyLibraries, expected array, got: ${libraries}`);
    }
    const librariesMap = new Map();
    for (const library of libraries) {
        if (typeof library.name !== "string" ||
            typeof library.command !== "string" ||
            (library.workingDir != null && typeof library.workingDir !== "string") ||
            (library.env != null && typeof library.env !== "object") ||
            typeof library.outputPath !== "string") {
            throw new Error(`Invalid library: ${JSON.stringify(library)}`);
        }
        if ("fallback" in library) {
            const fallback = library.fallback;
            if (typeof fallback !== "object" || (typeof fallback.outputPath !== "string" && typeof fallback.command !== "string")) {
                throw new Error(`Invalid library fallback for ${library.name}: ${JSON.stringify(fallback)}`);
            }
        }
        librariesMap.set(library.name, library);
    }
    return librariesMap;
}
function parseOptions(options) {
    return {
        libraries: parseLibraries(options.webassemblyLibraries),
    };
}
function findLib({ libraries }, requestedLib) {
    const lib = libraries.get(requestedLib);
    if (!lib)
        throw new Error(`${requestedLib} isn't included in build options`);
    return lib;
}
function createOutputFolderStructure(output) {
    if (!fs.existsSync(output)) {
        fs.mkdirSync(output, { recursive: true });
    }
}
export function rollupWasmLoader(options) {
    const normalizedOptions = parseOptions(options);
    return {
        name: "wasm",
        async resolveId(source) {
            if (normalizedOptions.libraries.has(source)) {
                return { id: `\0wasm-loader:${source}`, external: false };
            }
        },
        async resolveDynamicImport(specifier, importer) {
            if (importer.includes("wasm-loader") && specifier.startsWith("node:")) {
                // rollup chokes on node: imports for some reason
                return { external: true, id: specifier.substring("node:".length) };
            }
            else if (importer.includes("wasm-loader")) {
                return {
                    id: `\0${specifier}`,
                    external: false,
                };
            }
        },
        async load(id) {
            if (id.startsWith("\0wasm-loader")) {
                const wasmLib = id.replaceAll("\0wasm-loader:", "");
                const lib = findLib(normalizedOptions, wasmLib);
                createOutputFolderStructure(path.dirname(lib.outputPath));
                await runCommand(lib.command, {
                    workingDir: lib.workingDir,
                    env: { WASM: lib.outputPath },
                });
                console.log(`Trying to read file from lib.outputPath: ${lib.outputPath}, wasmLib: ${wasmLib}`);
                return await generateImportCode(wasmLib, lib.fallback != null);
            }
            else if (id.startsWith("\0wasm-fallback")) {
                const wasmLib = id.replaceAll("\0wasm-fallback:", "");
                const lib = findLib(normalizedOptions, wasmLib);
                if (lib.fallback) {
                    await runCommand(lib.fallback.command, {
                        workingDir: lib.fallback.workingDir,
                        env: {
                            WASM: lib.outputPath,
                            WASM_FALLBACK: lib.fallback.outputPath,
                        },
                    });
                    return fs.promises.readFile(lib.fallback.outputPath, "utf-8");
                }
            }
        },
    };
}
