import { Assertion } from "./Assertion.js";
import { RunResult } from "./TestResult.js";
declare class OTest {
    private static readonly DEFAULT_TIMEOUT_MS;
    private taskTree;
    private currentSpec;
    private currentTest;
    /**
     * Define a group of tests.
     * Spec may contain:
     *  * tests
     *  * before/beforeEach/after/afterEach clauses
     *  * other specs
     *
     *  Example:
     *  ```ts
     *  o.spec("testableFunction", () => {
     *      o.test("it works", () => {
     *          o.check(testableFunction(1)).equals(2)
     *      })
     *  })
     *  ```
     */
    spec(name: string, definition: () => void): void;
    /**
     * Define a test.
     * Tests may be async in which case they should either await or return a promise.
     * Timeouts apply and can be changed with {@link timeout}.
     */
    test(name: string, definition: (() => Promise<void>) | (() => void)): void;
    /**
     * Start an assertion.
     */
    check<T>(value: T): Assertion<T>;
    /**
     * Define a task to be executed before any test in the spec (once per spec).
     */
    before(task: () => Promise<void> | void): void;
    /**
     * Define a task to be executed after all test in the spec (once per spec).
     */
    after(task: () => Promise<void> | void): void;
    /**
     * Define a task to be executed before each test in the spec (once per test).
     * Also applies to tests in nested specs.
     */
    beforeEach(task: () => Promise<void> | void): void;
    /**
     * Define a task to be executed after each test in the spec (once per test).
     * Also applies to tests in nested specs.
     */
    afterEach(task: () => Promise<void> | void): void;
    /**
     * Set a timeout (in ms) for the currently running test.
     */
    timeout(ms: number): void;
    /**
     * Run the tests that were previously defined.
     * @param {string} filter: only run tests that match the filter string in either spec name or a test name.
     */
    run({ filter }?: {
        filter?: string;
    }): Promise<RunResult>;
    private runSpec;
    /**
     * Output the result of the test run.
     * @param result
     */
    printReport(result: RunResult): void;
    /**
     * A utility to exit the process with the appropriate exit code.
     * only runs in node, no-op otherwise.
     */
    terminateProcess(result: RunResult): void;
    private runTest;
}
export type CallableOTest = OTest & {
    /** An alias for {@link OTest.test}. Discouraged. */
    (name: string, definition: () => Promise<void> | void): void;
    /** An alias for {@link OTest.check}. Discouraged. */ <T>(actual: T): Assertion<T>;
};
declare const _default: CallableOTest;
export default _default;
