/**
 * @license
 * Copyright 2020 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import { convertToTensor } from '../tensor_util_env';
import * as util from '../util';
import { matMul } from './mat_mul';
import { op } from './operation';
import { reshape } from './reshape';
/**
 * Computes the dot product of two matrices and/or vectors, `t1` and `t2`.
 *
 * ```js
 * const a = tf.tensor1d([1, 2]);
 * const b = tf.tensor2d([[1, 2], [3, 4]]);
 * const c = tf.tensor2d([[1, 2, 3], [4, 5, 6]]);
 *
 * a.dot(b).print();  // or tf.dot(a, b)
 * b.dot(a).print();
 * b.dot(c).print();
 * ```
 * @param t1 The first tensor in the dot operation.
 * @param t2 The second tensor in the dot operation.
 *
 * @doc {heading: 'Operations', subheading: 'Matrices'}
 */
function dot_(t1, t2) {
    const $t1 = convertToTensor(t1, 't1', 'dot');
    const $t2 = convertToTensor(t2, 't2', 'dot');
    util.assert(($t1.rank === 1 || $t1.rank === 2) && ($t2.rank === 1 || $t2.rank === 2), () => `Error in dot: inputs must all be rank 1 or 2, but got ranks ` +
        `${$t1.rank} and ${$t2.rank}.`);
    const t1Inner = ($t1.rank === 1 ? $t1.size : $t1.shape[1]);
    const t2Inner = ($t2.rank === 1 ? $t2.size : $t2.shape[0]);
    util.assert(t1Inner === t2Inner, () => `Error in dot: inner dimensions of inputs must match, but got ` +
        `${t1Inner} and ${t2Inner}.`);
    if ($t1.rank === 1 && $t2.rank === 1) {
        const t12D = reshape($t1, [1, -1]);
        const t22D = reshape($t2, [-1, 1]);
        const t1t2 = matMul(t12D, t22D);
        return reshape(t1t2, []);
    }
    else if ($t1.rank === 1 && $t2.rank === 2) {
        const t12D = reshape($t1, [1, -1]);
        const t22D = reshape($t2, [$t2.shape[0], $t2.shape[1]]);
        const t1t2 = matMul(t12D, t22D);
        return reshape(t1t2, [t1t2.size]);
    }
    else if ($t1.rank === 2 && $t2.rank === 1) {
        const t22D = reshape($t2, [-1, 1]);
        const t1t2 = matMul($t1, t22D);
        return reshape(t1t2, [t1t2.size]);
    }
    else {
        const t22D = reshape($t2, [$t2.shape[0], $t2.shape[1]]);
        const t1t2 = matMul($t1, t22D);
        return t1t2;
    }
}
export const dot = /* @__PURE__ */ op({ dot_ });
//# sourceMappingURL=data:application/json;base64,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