/**
 * @license
 * Copyright 2017 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import * as tf from '../index';
import { ALL_ENVS, describeWithFlags } from '../jasmine_util';
import { expectArraysClose } from '../test_util';
describeWithFlags('depthwiseConv2D', ALL_ENVS, () => {
    it('input=1x3x3x1,f=2,s=1,d=1,p=valid,chMul=1', async () => {
        const fSize = 2;
        const pad = 'valid';
        const stride = 1;
        const chMul = 1;
        const inDepth = 1;
        const x = tf.tensor4d([
            0.230664, 0.987388, 0.0685208, 0.419224, 0.887861, 0.731641,
            0.0741907, 0.409265, 0.351377
        ], [1, 3, 3, inDepth]);
        const w = tf.tensor4d([0.303873, 0.229223, 0.144333, 0.803373], [fSize, fSize, inDepth, chMul]);
        const result = tf.depthwiseConv2d(x, w, stride, pad);
        expect(result.shape).toEqual([1, 2, 2, 1]);
        const expected = [1.07022, 1.03167, 0.67041, 0.778863];
        expectArraysClose(await result.data(), expected);
    });
    it('input=1x3x3x1,f=2,s=1,d=1,p=explicit,chMul=1', async () => {
        const fSize = 2;
        const pad = [[0, 0], [1, 2], [0, 1], [0, 0]];
        const stride = 1;
        const chMul = 1;
        const inDepth = 1;
        const x = tf.tensor4d([
            0.230664, 0.987388, 0.0685208, 0.419224, 0.887861, 0.731641,
            0.0741907, 0.409265, 0.351377
        ], [1, 3, 3, inDepth]);
        const w = tf.tensor4d([0.303873, 0.229223, 0.144333, 0.803373], [fSize, fSize, inDepth, chMul]);
        const result = tf.depthwiseConv2d(x, w, stride, pad);
        expect(result.shape).toEqual([1, 5, 3, 1]);
        const expected = [
            0.826533, 0.197560, 0.0098898, 1.070216, 1.031675, 0.126422, 0.6704096,
            0.778863, 0.273041, 0.116357, 0.204908, 0.106774, 0, 0, 0
        ];
        expectArraysClose(await result.data(), expected);
    });
    it('input=1x5x5x1,f=3,s=1,d=1,p=valid,chMul=1', async () => {
        const fSize = 3;
        const pad = 'valid';
        const stride = 1;
        const chMul = 1;
        const inDepth = 1;
        const x = tf.tensor4d([
            0.149194, 0.089009, 0.654891, 0.083324, 0.537043, 0.644331, 0.563037,
            0.211859, 0.633501, 0.186427, 0.777034, 0.50001, 0.607341, 0.95303,
            0.696479, 0.050387, 0.62045, 0.728049, 0.028043, 0.437009, 0.712881,
            0.741935, 0.974474, 0.621102, 0.171411
        ], [1, 5, 5, inDepth]);
        const w = tf.tensor4d([
            0.125386, 0.975199, 0.640437, 0.281895, 0.990968, 0.347208, 0.889702,
            0.180695, 0.691992
        ], [fSize, fSize, inDepth, chMul]);
        const result = tf.depthwiseConv2d(x, w, stride, pad);
        expect(result.shape).toEqual([1, 3, 3, 1]);
        const expected = [
            2.540022, 2.505885, 2.454062, 2.351701, 2.459601, 3.076421, 3.29848,
            3.437421, 2.93419
        ];
        expectArraysClose(await result.data(), expected);
    });
    it('input=1x3x3x1,f=2,s=1,d=2,p=valid,chMul=1', async () => {
        const fSize = 2;
        const pad = 'valid';
        const stride = 1;
        const dilation = 2;
        const chMul = 1;
        const inDepth = 1;
        const x = tf.tensor4d([
            0.230664, 0.987388, 0.0685208, 0.419224, 0.887861, 0.731641,
            0.0741907, 0.409265, 0.351377
        ], [1, 3, 3, inDepth]);
        const w = tf.tensor4d([0.303873, 0.229223, 0.144333, 0.803373], [fSize, fSize, inDepth, chMul]);
        // adding a dilation rate is equivalent to using a filter
        // with 0s for the dilation rate
        const fSizeDilated = fSize + (fSize - 1) * (dilation - 1);
        const wDilated = tf.tensor4d([0.303873, 0, 0.229223, 0, 0, 0, 0.144333, 0, 0.803373], [fSizeDilated, fSizeDilated, inDepth, chMul]);
        const result = tf.depthwiseConv2d(x, w, stride, pad, 'NHWC', dilation);
        const expectedResult = tf.depthwiseConv2d(x, wDilated, stride, pad);
        expect(result.shape).toEqual(expectedResult.shape);
        expectArraysClose(await result.data(), await expectedResult.data());
    });
    it('input=1x5x5x1,f=3,s=1,d=2,p=valid,chMul=1', async () => {
        const fSize = 3;
        const pad = 'valid';
        const stride = 1;
        const dilation = 2;
        const chMul = 1;
        const inDepth = 1;
        const x = tf.tensor4d([
            0.149194, 0.089009, 0.654891, 0.083324, 0.537043, 0.644331, 0.563037,
            0.211859, 0.633501, 0.186427, 0.777034, 0.50001, 0.607341, 0.95303,
            0.696479, 0.050387, 0.62045, 0.728049, 0.028043, 0.437009, 0.712881,
            0.741935, 0.974474, 0.621102, 0.171411
        ], [1, 5, 5, inDepth]);
        const w = tf.tensor4d([
            0.125386, 0.975199, 0.640437, 0.281895, 0.990968, 0.347208, 0.889702,
            0.180695, 0.691992
        ], [fSize, fSize, inDepth, chMul]);
        // adding a dilation rate is equivalent to using a filter
        // with 0s for the dilation rate
        const fSizeDilated = fSize + (fSize - 1) * (dilation - 1);
        const wDilated = tf.tensor4d([
            0.125386, 0, 0.975199, 0, 0.640437, 0, 0, 0, 0, 0,
            0.281895, 0, 0.990968, 0, 0.347208, 0, 0, 0, 0, 0,
            0.889702, 0, 0.180695, 0, 0.691992
        ], [fSizeDilated, fSizeDilated, inDepth, chMul]);
        const result = tf.depthwiseConv2d(x, w, stride, pad, 'NHWC', dilation);
        const expectedResult = tf.depthwiseConv2d(x, wDilated, stride, pad);
        expect(result.shape).toEqual(expectedResult.shape);
        expectArraysClose(await result.data(), await expectedResult.data());
    });
    it('input=1x5x5x1,f=2,s=1,d=4,p=valid,chMul=1', async () => {
        const fSize = 2;
        const pad = 'valid';
        const stride = 1;
        const dilation = 4;
        const chMul = 1;
        const inDepth = 1;
        const x = tf.tensor4d([
            0.149194, 0.089009, 0.654891, 0.083324, 0.537043, 0.644331, 0.563037,
            0.211859, 0.633501, 0.186427, 0.777034, 0.50001, 0.607341, 0.95303,
            0.696479, 0.050387, 0.62045, 0.728049, 0.028043, 0.437009, 0.712881,
            0.741935, 0.974474, 0.621102, 0.171411
        ], [1, 5, 5, inDepth]);
        const w = tf.tensor4d([0.125386, 0.975199, 0.640437, 0.281895], [fSize, fSize, inDepth, chMul]);
        // adding a dilation rate is equivalent to using a filter
        // with 0s for the dilation rate
        const fSizeDilated = fSize + (fSize - 1) * (dilation - 1);
        const wDilated = tf.tensor4d([
            0.125386, 0, 0, 0, 0.975199, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0.640437, 0, 0, 0, 0.281895
        ], [fSizeDilated, fSizeDilated, inDepth, chMul]);
        const result = tf.depthwiseConv2d(x, w, stride, pad, 'NHWC', dilation);
        const expectedResult = tf.depthwiseConv2d(x, wDilated, stride, pad);
        expect(result.shape).toEqual(expectedResult.shape);
        expectArraysClose(await result.data(), await expectedResult.data());
    });
    it('input=1x3x3x2,f=2,s=1,d=1,p=same,chMul=1', async () => {
        const fSize = 2;
        const pad = 'same';
        const stride = 1;
        const chMul = 1;
        const inDepth = 2;
        const x = tf.tensor4d([
            0.111057, 0.661818, 0.701979, 0.424362, 0.992854, 0.417599, 0.423036,
            0.500499, 0.368484, 0.714135, 0.456693, 0.531058, 0.636636, 0.345024,
            0.0506303, 0.789682, 0.177473, 0.793569
        ], [1, 3, 3, inDepth]);
        const w = tf.tensor4d([
            0.614293, 0.0648011, 0.101113, 0.452887, 0.0582746, 0.426481,
            0.872743, 0.765767
        ], [fSize, fSize, inDepth, chMul]);
        const result = tf.depthwiseConv2d(x, w, stride, pad);
        expect(result.shape).toEqual([1, 3, 3, 2]);
        const expected = [
            0.485445, 0.995389, 0.95166, 0.927856, 0.636516, 0.253547, 0.378414,
            1.10771, 0.430373, 1.23126, 0.290885, 0.372855, 0.3962, 0.379995,
            0.0490466, 0.410569, 0.10902, 0.0514242
        ];
        expectArraysClose(await result.data(), expected);
    });
    it('input=1x5x5x1,f=3,s=1,d=1,p=same,chMul=1', async () => {
        const fSize = 3;
        const pad = 'same';
        const stride = 1;
        const chMul = 1;
        const inDepth = 1;
        const x = tf.tensor4d([
            0.149194, 0.089009, 0.654891, 0.083324, 0.537043, 0.644331, 0.563037,
            0.211859, 0.633501, 0.186427, 0.777034, 0.50001, 0.607341, 0.95303,
            0.696479, 0.050387, 0.62045, 0.728049, 0.028043, 0.437009, 0.712881,
            0.741935, 0.974474, 0.621102, 0.171411
        ], [1, 5, 5, inDepth]);
        const w = tf.tensor4d([
            0.125386, 0.975199, 0.640437, 0.281895, 0.990968, 0.347208, 0.889702,
            0.180695, 0.691992
        ], [fSize, fSize, inDepth, chMul]);
        const result = tf.depthwiseConv2d(x, w, stride, pad);
        expect(result.shape).toEqual([1, 5, 5, 1]);
        const expected = [
            0.684796, 1.179251, 1.680593, 0.885615, 1.152995, 1.52291, 2.540022,
            2.505885, 2.454062, 1.871258, 2.371015, 2.351701, 2.459601, 3.076421,
            1.323994, 1.985572, 3.29848, 3.437421, 2.93419, 1.823238, 1.410545,
            2.352186, 2.19622, 1.348218, 0.774635
        ];
        expectArraysClose(await result.data(), expected);
    });
    it('input=1x3x3x2,f=2,s=1,d=2,p=same,chMul=1', async () => {
        const fSize = 2;
        const pad = 'same';
        const stride = 1;
        const dilation = 2;
        const inDepth = 2;
        const x = tf.tensor4d([
            0.111057, 0.661818, 0.701979, 0.424362, 0.992854, 0.417599, 0.423036,
            0.500499, 0.368484, 0.714135, 0.456693, 0.531058, 0.636636, 0.345024,
            0.0506303, 0.789682, 0.177473, 0.793569
        ], [1, 3, 3, inDepth]);
        const w = tf.stack([
            tf.tensor2d([0.614293, 0.0648011, 0.101113, 0.452887], [fSize, fSize]),
            tf.tensor2d([0.0582746, 0.426481, 0.872743, 0.765767], [fSize, fSize])
        ], 2)
            .expandDims(3);
        // adding a dilation rate is equivalent to using a filter
        // with 0s for the dilation rate
        const fSizeDilated = fSize + (fSize - 1) * (dilation - 1);
        const wDilated = tf.stack([
            tf.tensor2d([0.614293, 0, 0.0648011, 0, 0, 0, 0.101113, 0, 0.452887], [fSizeDilated, fSizeDilated]),
            tf.tensor2d([0.0582746, 0, 0.426481, 0, 0, 0, 0.872743, 0, 0.765767], [fSizeDilated, fSizeDilated])
        ], 2)
            .expandDims(3);
        expect(wDilated.shape).toEqual([fSizeDilated, fSizeDilated, inDepth, 1]);
        const result = tf.depthwiseConv2d(x, w, stride, pad, 'NHWC', dilation);
        const expectedResult = tf.depthwiseConv2d(x, wDilated, stride, pad);
        expect(result.shape).toEqual(expectedResult.shape);
        expectArraysClose(await result.data(), await expectedResult.data());
    });
    it('input=1x5x5x1,f=3,s=1,d=2,p=valid,chMul=1', async () => {
        const fSize = 3;
        const pad = 'valid';
        const stride = 1;
        const chMul = 1;
        const inDepth = 1;
        const x = tf.tensor4d([
            0.149194, 0.089009, 0.654891, 0.083324, 0.537043, 0.644331, 0.563037,
            0.211859, 0.633501, 0.186427, 0.777034, 0.50001, 0.607341, 0.95303,
            0.696479, 0.050387, 0.62045, 0.728049, 0.028043, 0.437009, 0.712881,
            0.741935, 0.974474, 0.621102, 0.171411
        ], [1, 5, 5, inDepth]);
        const w = tf.tensor4d([
            0.125386, 0.975199, 0.640437, 0.281895, 0.990968, 0.347208, 0.889702,
            0.180695, 0.691992
        ], [fSize, fSize, inDepth, chMul]);
        const result = tf.depthwiseConv2d(x, w, stride, pad);
        expect(result.shape).toEqual([1, 3, 3, 1]);
        const expected = [
            2.540022, 2.505885, 2.454062, 2.351701, 2.459601, 3.076421, 3.29848,
            3.437421, 2.93419
        ];
        expectArraysClose(await result.data(), expected);
    });
    it('input=1x5x5x4,f=3,s=1,d=1,p=same,chMul=1', async () => {
        const fSize = 3;
        const pad = 'same';
        const stride = 1;
        const chMul = 1;
        const inDepth = 4;
        const x = tf.tensor4d([
            0.149194, 0.089009, 0.654891, 0.083324, 0.537043, 0.644331,
            0.563037, 0.211859, 0.633501, 0.186427, 0.777034, 0.50001,
            0.607341, 0.95303, 0.696479, 0.050387, 0.62045, 0.728049,
            0.028043, 0.437009, 0.712881, 0.741935, 0.974474, 0.621102,
            0.171411, 0.675707, 0.758567, 0.413529, 0.963967, 0.217291,
            0.101335, 0.804231, 0.329673, 0.924503, 0.728742, 0.180217,
            0.210459, 0.133869, 0.650827, 0.047613, 0.554795, 0.653365,
            0.442196, 0.261945, 0.0528113, 0.656698, 0.127345, 0.610039,
            0.169131, 0.458647, 0.0988288, 0.966109, 0.0421747, 0.82035,
            0.274711, 0.359377, 0.512113, 0.689682, 0.941571, 0.31961,
            0.743826, 0.858147, 0.984766, 0.926973, 0.579597, 0.444104,
            0.505969, 0.241437, 0.937999, 0.0957074, 0.773611, 0.46023,
            0.469379, 0.363789, 0.269745, 0.486136, 0.894215, 0.794299,
            0.724615, 0.261945, 0.0528113, 0.656698, 0.127345, 0.610039,
            0.169131, 0.458647, 0.0988288, 0.966109, 0.0421747, 0.82035,
            0.274711, 0.359377, 0.512113, 0.689682, 0.941571, 0.31961,
            0.743826, 0.858147, 0.984766, 0.926973
        ], [1, 5, 5, inDepth]);
        const w = tf.tensor4d([
            0.6511372, 0.8699447, 0.6511372, 0.8699447, 0.267792, 0.9981787,
            0.267792, 0.9981787, 0.4913572, 0.3321196, 0.4913572, 0.3321196,
            0.5286497, 0.4241803, 0.5286497, 0.4241803, 0.0175446, 0.8365464,
            0.0175446, 0.8365464, 0.1768399, 0.2874831, 0.1768399, 0.2874831,
            0.0933998, 0.5764548, 0.0933998, 0.5764548, 0.0661623, 0.8850273,
            0.0661623, 0.8850273, 0.8700929, 0.205422, 0.8700929, 0.205422
        ], [fSize, fSize, inDepth, chMul]);
        const result = tf.depthwiseConv2d(x, w, stride, pad);
        expect(result.shape).toEqual([1, 5, 5, 4]);
        const expected = [
            0.29389750957489014, 1.055132269859314, 0.8355544209480286,
            0.7652503848075867, 1.116986632347107, 1.7007107734680176,
            0.7228718996047974, 1.2455471754074097, 0.7690584063529968,
            1.4749835729599, 1.1460752487182617, 1.5098011493682861,
            0.7502411007881165, 2.056602716445923, 1.0519171953201294,
            1.012758731842041, 0.37667199969291687, 1.6647151708602905,
            0.4798099994659424, 0.532977283000946, 0.4293096363544464,
            1.8309053182601929, 0.7433272004127502, 1.1491419076919556,
            1.3050479888916016, 2.7769954204559326, 1.6411027908325195,
            2.1799824237823486, 1.0364032983779907, 2.7503039836883545,
            1.7060394287109375, 2.880652904510498, 1.8967751264572144,
            3.3914175033569336, 1.734355092048645, 2.076633930206299,
            0.7774094939231873, 3.1432321071624756, 0.9456352591514587,
            1.0863502025604248, 0.8477171659469604, 2.5510711669921875,
            1.169355869293213, 2.0218098163604736, 2.23183274269104,
            3.257829189300537, 1.939490556716919, 2.96195650100708,
            1.0946838855743408, 2.4252827167510986, 1.329919695854187,
            3.0390005111694336, 1.8967963457107544, 2.775693416595459,
            1.5250799655914307, 2.4470155239105225, 0.40530526638031006,
            2.775503158569336, 0.8836789727210999, 1.1361782550811768,
            0.4407186806201935, 2.3912413120269775, 0.38215696811676025,
            2.047299861907959, 1.080580234527588, 3.09224534034729,
            1.2943278551101685, 3.1656715869903564, 0.9704407453536987,
            2.8066811561584473, 1.419780969619751, 3.1822099685668945,
            1.720312237739563, 3.279745578765869, 2.0871992111206055,
            2.6629819869995117, 0.5254714488983154, 3.3779194355010986,
            0.73943030834198, 2.0616414546966553, 0.5148154497146606,
            1.6852912902832031, 0.5320349931716919, 1.7935365438461304,
            1.1387810707092285, 2.119696617126465, 1.2744661569595337,
            2.3705403804779053, 1.0399315357208252, 1.6817822456359863,
            0.8927359580993652, 1.6332063674926758, 1.3386595249176025,
            1.8818190097808838, 1.267898440361023, 1.6589205265045166,
            0.8288722038269043, 2.119757890701294, 0.8847255706787109,
            1.5954076051712036
        ];
        expectArraysClose(await result.data(), expected);
    });
    it('input=1x5x5x4,f=5,s=2,d=1,p=same,chMul=1', async () => {
        const fSize = 5;
        const pad = 'same';
        const stride = 2;
        const chMul = 1;
        const inDepth = 4;
        const x = tf.tensor4d([
            0.149194, 0.089009, 0.654891, 0.083324, 0.537043, 0.644331,
            0.563037, 0.211859, 0.633501, 0.186427, 0.777034, 0.50001,
            0.607341, 0.95303, 0.696479, 0.050387, 0.62045, 0.728049,
            0.028043, 0.437009, 0.712881, 0.741935, 0.974474, 0.621102,
            0.171411, 0.675707, 0.758567, 0.413529, 0.963967, 0.217291,
            0.101335, 0.804231, 0.329673, 0.924503, 0.728742, 0.180217,
            0.210459, 0.133869, 0.650827, 0.047613, 0.554795, 0.653365,
            0.442196, 0.261945, 0.0528113, 0.656698, 0.127345, 0.610039,
            0.169131, 0.458647, 0.0988288, 0.966109, 0.0421747, 0.82035,
            0.274711, 0.359377, 0.512113, 0.689682, 0.941571, 0.31961,
            0.743826, 0.858147, 0.984766, 0.926973, 0.579597, 0.444104,
            0.505969, 0.241437, 0.937999, 0.0957074, 0.773611, 0.46023,
            0.469379, 0.363789, 0.269745, 0.486136, 0.894215, 0.794299,
            0.724615, 0.261945, 0.0528113, 0.656698, 0.127345, 0.610039,
            0.169131, 0.458647, 0.0988288, 0.966109, 0.0421747, 0.82035,
            0.274711, 0.359377, 0.512113, 0.689682, 0.941571, 0.31961,
            0.743826, 0.858147, 0.984766, 0.926973
        ], [1, 5, 5, inDepth]);
        const w = tf.tensor4d([
            0.6511372, 0.8699447, 0.6511372, 0.8699447, 0.267792, 0.9981787,
            0.267792, 0.9981787, 0.4913572, 0.3321196, 0.4913572, 0.3321196,
            0.5286497, 0.4241803, 0.5286497, 0.4241803, 0.0175446, 0.8365464,
            0.0175446, 0.8365464, 0.1768399, 0.2874831, 0.1768399, 0.2874831,
            0.0933998, 0.5764548, 0.0933998, 0.5764548, 0.0661623, 0.8850273,
            0.0661623, 0.8850273, 0.8700929, 0.205422, 0.8700929, 0.205422,
            0.149194, 0.089009, 0.654891, 0.083324, 0.537043, 0.644331,
            0.563037, 0.211859, 0.633501, 0.186427, 0.777034, 0.50001,
            0.607341, 0.95303, 0.696479, 0.050387, 0.62045, 0.728049,
            0.028043, 0.437009, 0.712881, 0.741935, 0.974474, 0.621102,
            0.171411, 0.125386, 0.975199, 0.640437, 0.281895, 0.990968,
            0.347208, 0.889702, 0.180695, 0.691992, 0.347154, 0.386692,
            0.327191, 0.483784, 0.591807, 0.24263, 0.95182, 0.174353,
            0.592136, 0.623469, 0.988244, 0.660731, 0.946534, 0.0801365,
            0.864889, 0.874602, 0.240347, 0.906352, 0.478657, 0.825918,
            0.380769, 0.184705, 0.238241, 0.201907, 0.294087, 0.181165,
            0.191303, 0.7225, 0.430064, 0.900622
        ], [fSize, fSize, inDepth, chMul]);
        const result = tf.depthwiseConv2d(x, w, stride, pad);
        expect(result.shape).toEqual([1, 3, 3, 4]);
        const expected = [
            2.2883458137512207, 2.5740344524383545, 2.3246560096740723,
            2.27826189994812, 3.0600292682647705, 5.021538734436035,
            4.432307720184326, 2.6976213455200195, 1.8467353582382202,
            3.617821216583252, 2.0940940380096436, 1.3091316223144531,
            2.4892354011535645, 4.767732620239258, 3.126866579055786,
            3.4326541423797607, 4.181705474853516, 8.082467079162598,
            6.922453880310059, 5.922790050506592, 2.819075345993042,
            5.9510369300842285, 3.7211103439331055, 2.7263708114624023,
            1.164026141166687, 3.3068809509277344, 1.6575196981430054,
            2.738445997238159, 2.288442850112915, 5.463253021240234,
            2.840029239654541, 3.8579823970794678, 1.440760612487793,
            3.862100839614868, 2.3826799392700195, 2.2323575019836426
        ];
        expectArraysClose(await result.data(), expected);
    });
    it('input=1x5x5x1,f=3,s=1,d=2,p=explicit,chMul=1', async () => {
        const fSize = 3;
        const pad = [[0, 0], [0, 0], [0, 1], [0, 1]];
        const stride = 1;
        const chMul = 1;
        const inDepth = 1;
        const x = tf.tensor4d([
            0.149194, 0.089009, 0.654891, 0.083324, 0.537043, 0.644331, 0.563037,
            0.211859, 0.633501, 0.186427, 0.777034, 0.50001, 0.607341, 0.95303,
            0.696479, 0.050387, 0.62045, 0.728049, 0.028043, 0.437009, 0.712881,
            0.741935, 0.974474, 0.621102, 0.171411
        ], [1, 5, 5, inDepth]);
        const w = tf.tensor4d([
            0.125386, 0.975199, 0.640437, 0.281895, 0.990968, 0.347208, 0.889702,
            0.180695, 0.691992
        ], [fSize, fSize, inDepth, chMul]);
        const result = tf.depthwiseConv2d(x, w, stride, pad);
        expect(result.shape).toEqual([1, 3, 4, 1]);
        const expected = [
            2.540022, 2.505885, 2.454062, 1.871258, 2.35170, 2.459601, 3.076421,
            1.32399, 3.298480, 3.437421, 2.93419, 1.823238
        ];
        expectArraysClose(await result.data(), expected);
    });
    it('input=1x3x3x4,f=3,s=1,d=2,p=same,chMul=1', async () => {
        const fSize = 3;
        const pad = 'same';
        const stride = 1;
        const chMul = 1;
        const inDepth = 4;
        const dilation = 2;
        const x = tf.tensor4d([
            0.5227615, 0.3477598, 0.5227615, 0.3477598, 0.4690094, 0.408161,
            0.4690094, 0.408161, 0.3239015, 0.2372907, 0.3239015, 0.2372907,
            0.6136674, 0.7918105, 0.6136674, 0.7918105, 0.9145211, 0.218611,
            0.9145211, 0.218611, 0.3778793, 0.2392365, 0.3778793, 0.2392365,
            0.2340134, 0.1251984, 0.2340134, 0.1251984, 0.6222534, 0.1327361,
            0.6222534, 0.1327361, 0.7697753, 0.1216059, 0.7697753, 0.1216059
        ], [1, 3, 3, inDepth]);
        const w = tf.tensor4d([
            0.6511372, 0.8699447, 0.6511372, 0.8699447, 0.267792, 0.9981787,
            0.267792, 0.9981787, 0.4913572, 0.3321196, 0.4913572, 0.3321196,
            0.5286497, 0.4241803, 0.5286497, 0.4241803, 0.0175446, 0.8365464,
            0.0175446, 0.8365464, 0.1768399, 0.2874831, 0.1768399, 0.2874831,
            0.0933998, 0.5764548, 0.0933998, 0.5764548, 0.0661623, 0.8850273,
            0.0661623, 0.8850273, 0.8700929, 0.205422, 0.8700929, 0.205422
        ], [fSize, fSize, inDepth, chMul]);
        const result = tf.depthwiseConv2d(x, w, stride, pad, 'NHWC', dilation);
        expect(result.shape).toEqual([1, 3, 3, 4]);
        const expected = [
            0.7517092227935791, 0.4949187934398651, 0.7517092227935791,
            0.4949187934398651, 0.04939830303192139, 0.4589206874370575,
            0.04939830303192139, 0.4589206874370575, 0.3548273742198944,
            0.5258132815361023, 0.3548273742198944, 0.5258132815361023,
            0.0775906890630722, 0.7311626672744751, 0.0775906890630722,
            0.7311626672744751, 0.01604490540921688, 0.1828782558441162,
            0.01604490540921688, 0.1828782558441162, 0.3310448229312897,
            0.5360028743743896, 0.3310448229312897, 0.5360028743743896,
            0.4393753409385681, 0.565629243850708, 0.4393753409385681,
            0.565629243850708, 0.13651414215564728, 0.5184575319290161,
            0.13651414215564728, 0.5184575319290161, 0.5643441677093506,
            0.6942259669303894, 0.5643441677093506, 0.6942259669303894
        ];
        expectArraysClose(await result.data(), expected);
    });
    it('input=1x3x3x2,f=2,s=1,p=same,chMul=2', async () => {
        const fSize = 2;
        const pad = 'same';
        const stride = 1;
        const chMul = 2;
        const inDepth = 2;
        const x = tf.tensor4d([
            0.675707, 0.758567, 0.413529, 0.963967, 0.217291, 0.101335, 0.804231,
            0.329673, 0.924503, 0.728742, 0.180217, 0.210459, 0.133869, 0.650827,
            0.047613, 0.554795, 0.653365, 0.442196
        ], [1, 3, 3, inDepth]);
        const w = tf.tensor4d([
            0.347154, 0.386692, 0.327191, 0.483784, 0.591807, 0.24263, 0.95182,
            0.174353, 0.592136, 0.623469, 0.988244, 0.660731, 0.946534, 0.0801365,
            0.864889, 0.874602
        ], [fSize, fSize, inDepth, chMul]);
        const result = tf.depthwiseConv2d(x, w, stride, pad);
        expect(result.shape).toEqual([1, 3, 3, 4]);
        const expected = [
            1.83059, 0.937125, 2.1218, 1.39024, 0.990167, 0.803472,
            1.31405, 1.14959, 0.182147, 0.196385, 0.241141, 0.188081,
            0.950656, 0.622581, 1.92451, 1.20179, 1.07422, 0.483268,
            1.36948, 1.14256, 0.449444, 0.477042, 0.505857, 0.393989,
            0.0746509, 0.0633184, 0.74101, 0.41159, 0.403195, 0.176938,
            0.602415, 0.345499, 0.226819, 0.252651, 0.144682, 0.213927
        ];
        expectArraysClose(await result.data(), expected);
    });
    it('input=2x3x3x2,f=2,s=1,p=same,chMul=2', async () => {
        const fSize = 2;
        const pad = 'same';
        const stride = 1;
        const chMul = 2;
        const inDepth = 2;
        const x = tf.tensor4d([
            0.261945, 0.0528113, 0.656698, 0.127345, 0.610039, 0.169131,
            0.458647, 0.0988288, 0.966109, 0.0421747, 0.82035, 0.274711,
            0.359377, 0.512113, 0.689682, 0.941571, 0.31961, 0.743826,
            0.858147, 0.984766, 0.926973, 0.579597, 0.444104, 0.505969,
            0.241437, 0.937999, 0.0957074, 0.773611, 0.46023, 0.469379,
            0.363789, 0.269745, 0.486136, 0.894215, 0.794299, 0.724615
        ], [2, 3, 3, inDepth]);
        const w = tf.tensor4d([
            0.240347, 0.906352, 0.478657, 0.825918, 0.380769, 0.184705, 0.238241,
            0.201907, 0.294087, 0.181165, 0.191303, 0.7225, 0.430064, 0.900622,
            0.670338, 0.33478
        ], [fSize, fSize, inDepth, chMul]);
        const result = tf.depthwiseConv2d(x, w, stride, pad);
        expect(result.shape).toEqual([2, 3, 3, 4]);
        const expected = [
            0.863379, 1.3119, 0.102795, 0.154853, 1.02704, 1.62173, 0.293466,
            0.261764, 0.387876, 0.701529, 0.133508, 0.338167, 0.880395, 1.28039,
            0.786492, 0.775361, 0.884845, 1.43995, 0.764374, 1.0196, 0.291162,
            0.801428, 0.273788, 0.764303, 0.348985, 0.45311, 0.469447, 0.613073,
            0.287461, 0.684128, 0.627899, 0.927844, 0.0768174, 0.28968, 0.356037,
            0.614339, 0.67138, 1.07894, 1.30747, 1.86705, 0.617971, 1.35402,
            0.860607, 1.29693, 0.242087, 0.485892, 0.331979, 0.757015, 0.410527,
            0.740235, 1.28431, 1.42516, 0.68281, 0.975185, 1.13892, 1.62237,
            0.344208, 0.561029, 0.363292, 0.911203, 0.272541, 0.419513, 0.342154,
            0.403335, 0.419286, 0.587321, 0.600655, 0.884853, 0.190907, 0.719914,
            0.346842, 0.598472
        ];
        expectArraysClose(await result.data(), expected);
    });
    it('input=2x3x3x2,f=2,s=1,d=2,p=same,chMul=2', async () => {
        const fSize = 2;
        const pad = 'same';
        const stride = 1;
        const inDepth = 2;
        const dilation = 2;
        const noDilation = 1;
        const x = tf.tensor4d([
            0.261945, 0.0528113, 0.656698, 0.127345, 0.610039, 0.169131,
            0.458647, 0.0988288, 0.966109, 0.0421747, 0.82035, 0.274711,
            0.359377, 0.512113, 0.689682, 0.941571, 0.31961, 0.743826,
            0.858147, 0.984766, 0.926973, 0.579597, 0.444104, 0.505969,
            0.241437, 0.937999, 0.0957074, 0.773611, 0.46023, 0.469379,
            0.363789, 0.269745, 0.486136, 0.894215, 0.794299, 0.724615
        ], [2, 3, 3, inDepth]);
        const w = tf.stack([
            tf.stack([
                tf.tensor2d([0.240347, 0.906352, 0.478657, 0.825918], [fSize, fSize]),
                tf.tensor2d([0.380769, 0.184705, 0.238241, 0.201907], [fSize, fSize])
            ], 2),
            tf.stack([
                tf.tensor2d([0.294087, 0.181165, 0.191303, 0.7225], [fSize, fSize]),
                tf.tensor2d([0.430064, 0.900622, 0.670338, 0.33478], [fSize, fSize])
            ], 2)
        ], 3);
        const fSizeDilated = fSize + (fSize - 1) * (dilation - 1);
        const wDilated = tf.stack([
            tf.stack([
                tf.tensor2d([0.240347, 0, 0.906352, 0, 0, 0, 0.478657, 0, 0.825918], [fSizeDilated, fSizeDilated]),
                tf.tensor2d([0.380769, 0, 0.184705, 0, 0, 0, 0.238241, 0, 0.201907], [fSizeDilated, fSizeDilated])
            ], 2),
            tf.stack([
                tf.tensor2d([0.294087, 0, 0.181165, 0, 0, 0, 0.191303, 0, 0.7225], [fSizeDilated, fSizeDilated]),
                tf.tensor2d([0.430064, 0, 0.900622, 0, 0, 0, 0.670338, 0, 0.33478], [fSizeDilated, fSizeDilated])
            ], 2)
        ], 3);
        const result = tf.depthwiseConv2d(x, w, stride, pad, 'NHWC', dilation);
        const expectedResult = tf.depthwiseConv2d(x, wDilated, stride, pad, 'NHWC', noDilation);
        expect(result.shape).toEqual(expectedResult.shape);
        expectArraysClose(await result.data(), await expectedResult.data());
    });
    it('input=2x3x3x2,f=3,s=1,d=2,p=same,chMul=2', async () => {
        const fSize = 3;
        const pad = 'same';
        const stride = 1;
        const inDepth = 2;
        const dilation = 2;
        const x = tf.tensor4d([[
                [
                    [0.52276146, 0.34775984], [0.4690094, 0.40816104],
                    [0.32390153, 0.23729074], [0.61366737, 0.7918105],
                    [0.9145211, 0.218611], [0.37787926, 0.23923647],
                    [0.23401344, 0.12519836]
                ],
                [
                    [0.6222534, 0.13273609], [0.7697753, 0.12160587],
                    [0.0448128, 0.94806635], [0.4199953, 0.7140714],
                    [0.01420832, 0.47453713], [0.02061439, 0.37226152],
                    [0.62741446, 0.23167181]
                ],
                [
                    [0.7257557, 0.14352751], [0.3011638, 0.3869065],
                    [0.09286129, 0.25151742], [0.7566397, 0.13099921],
                    [0.65324724, 0.38959372], [0.65826, 0.7505318],
                    [0.35919082, 0.85470796]
                ],
                [
                    [0.24827361, 0.2826661], [0.24717247, 0.27446854],
                    [0.27112448, 0.68068564], [0.11082292, 0.7948675],
                    [0.41535318, 0.659986], [0.22165525, 0.18149579],
                    [0.42273378, 0.9558281]
                ],
                [
                    [0.943074, 0.6799041], [0.78851473, 0.07249606],
                    [0.771909, 0.7925967], [0.9551083, 0.03087568],
                    [0.82589805, 0.94797385], [0.5895462, 0.5045923],
                    [0.9667754, 0.24292922]
                ],
                [
                    [0.67123663, 0.109761], [0.04002762, 0.51942277],
                    [0.37868536, 0.8467603], [0.77171385, 0.51604605],
                    [0.8192849, 0.38843668], [0.19607484, 0.5591624],
                    [0.45990825, 0.35768318]
                ],
                [
                    [0.67443585, 0.6256168], [0.9373623, 0.6498393],
                    [0.7623085, 0.13218105], [0.9349631, 0.7660191],
                    [0.50054944, 0.7738123], [0.30201948, 0.525643],
                    [0.30896342, 0.21111596]
                ]
            ]], [1, 7, 7, inDepth]);
        const w = tf.tensor4d([
            [
                [[0.65113723], [0.8699447]],
                [[0.267792], [0.9981787]],
                [[0.4913572], [0.33211958]]
            ],
            [
                [[0.5286497], [0.42418027]],
                [[0.01754463], [0.8365464]],
                [[0.17683995], [0.2874831]]
            ],
            [
                [[0.09339976], [0.57645476]],
                [[0.06616235], [0.8850273]],
                [[0.87009287], [0.20542204]]
            ]
        ], [fSize, fSize, inDepth, 1]);
        const result = tf.depthwiseConv2d(x, w, stride, pad, 'NHWC', dilation);
        expect(result.shape).toEqual([1, 7, 7, 2]);
        expectArraysClose(await result.data(), [
            0.19526604, 0.5378273, 0.795022, 0.9384107, 1.0860794, 0.7942326,
            0.9764694, 1.3974442, 0.5930813, 0.9848901, 0.44526684, 1.275759,
            0.572345, 1.1784878, 0.27117175, 0.773588, 0.20055711, 0.71320784,
            0.73477566, 1.8867722, 0.64123434, 1.6549369, 0.55551285, 2.0385633,
            0.24740812, 1.233143, 0.08528192, 1.6214795, 1.062326, 1.3828603,
            1.4494176, 1.1022222, 2.2350664, 2.283423, 1.5940895, 1.8871424,
            1.6627852, 2.4903212, 1.0405337, 2.0754304, 1.1508893, 1.9568737,
            0.6148571, 1.1505995, 1.1105528, 1.3823687, 1.4342139, 2.9909487,
            1.0210396, 2.6467443, 1.0563798, 3.3963797, 0.42652097, 2.274134,
            0.51121074, 2.264094, 1.1009313, 1.6042703, 1.510688, 1.2317145,
            2.025515, 2.3658662, 1.6722159, 2.0787857, 1.3785586, 2.895031,
            1.2915218, 2.2051222, 1.0423074, 2.4303207, 0.27844793, 0.84346974,
            0.25781655, 1.1208354, 0.9447272, 2.0111258, 0.3689065, 1.9052455,
            0.79137695, 2.355344, 0.5429248, 1.5593178, 0.8248403, 1.9922242,
            0.77847, 1.5032601, 0.8622418, 0.84645665, 1.6850245, 2.2958806,
            1.6242284, 1.329045, 1.6652328, 2.480535, 1.2793491, 1.2951884,
            1.0667037, 1.5720158
        ]);
    });
    it('input=1x8x8x2,f=3,s=1,d=3,p=valid,chMul=1', async () => {
        const fSize = 3;
        const pad = 'valid';
        const stride = 1;
        const inDepth = 2;
        const dilation = 3;
        const x = tf.tensor4d([
            0.09941668063402176, 0.05248984694480896, 0.4567521810531616,
            0.8002573847770691, 0.810535192489624, 0.7010623216629028,
            0.5898630023002625, 0.05883334204554558, 0.2314797043800354,
            0.45427876710891724, 0.10960108041763306, 0.9710874557495117,
            0.18139968812465668, 0.8959258794784546, 0.35156702995300293,
            0.6495933532714844, 0.5185067653656006, 0.3260101079940796,
            0.7837356925010681, 0.9170011281967163, 0.465780109167099,
            0.0857422724366188, 0.38354963064193726, 0.8134718537330627,
            0.8768209218978882, 0.38151195645332336, 0.5045309066772461,
            0.8152258396148682, 0.2782581150531769, 0.545160174369812,
            0.1587309092283249, 0.5507456064224243, 0.2704062759876251,
            0.7736618518829346, 0.9871141314506531, 0.29300180077552795,
            0.3038032352924347, 0.36257433891296387, 0.967268168926239,
            0.7251133918762207, 0.6244085431098938, 0.8398842215538025,
            0.42696574330329895, 0.25569799542427063, 0.5784937143325806,
            0.22755105793476105, 0.8869972229003906, 0.05128923058509827,
            0.6748542785644531, 0.97468101978302, 0.5549167394638062,
            0.5639380812644958, 0.821204662322998, 0.5207878947257996,
            0.8831672668457031, 0.6721863746643066, 0.23375047743320465,
            0.040671784430742264, 0.24522553384304047, 0.6293181777000427,
            0.6886807680130005, 0.29527169466018677, 0.48199158906936646,
            0.5751473307609558, 0.817806601524353, 0.38846832513809204,
            0.5553714036941528, 0.1839468777179718, 0.5287416577339172,
            0.4813096523284912, 0.477756530046463, 0.641162633895874,
            0.03040425479412079, 0.20608118176460266, 0.7930338978767395,
            0.727353572845459, 0.42868077754974365, 0.6136374473571777,
            0.06312728673219681, 0.4346885681152344, 0.004786544945091009,
            0.4951920807361603, 0.588252604007721, 0.724294126033783,
            0.07830118387937546, 0.07353833317756653, 0.7818689346313477,
            0.8137099742889404, 0.6505773067474365, 0.5716961026191711,
            0.5416423678398132, 0.855529248714447, 0.8958709239959717,
            0.3598312437534332, 0.31329575181007385, 0.5971285104751587,
            0.034069616347551346, 0.6229354739189148, 0.24074052274227142,
            0.3356363773345947, 0.1049640029668808, 0.2543765604496002,
            0.1635538637638092, 0.8082090616226196, 0.9097364544868469,
            0.6435819268226624, 0.6100808382034302, 0.29750677943229675,
            0.0738643929362297, 0.8887753486633301, 0.7692861557006836,
            0.6412256360054016, 0.16205888986587524, 0.9414404034614563,
            0.5698712468147278, 0.6834514737129211, 0.41202589869499207,
            0.9096908569335938, 0.8094117045402527, 0.42103442549705505,
            0.8905773162841797, 0.069722980260849, 0.014392468146979809,
            0.22018849849700928, 0.30076053738594055, 0.8472294211387634,
            0.852762758731842, 0.5004454851150513
        ], [1, 8, 8, inDepth]);
        const w = tf.tensor4d([
            0.5785998106002808, 0.7439202666282654, 0.2178175300359726,
            0.8782838582992554, 0.6579487919807434, 0.6556791067123413,
            0.7341834306716919, 0.3332836329936981, 0.037182893604040146,
            0.7394348382949829, 0.04031887650489807, 0.19104436039924622,
            0.7014378309249878, 0.5309979319572449, 0.8485966920852661,
            0.6609954237937927, 0.021728534251451492, 0.9289031624794006
        ], [fSize, fSize, inDepth, 1]);
        const result = tf.depthwiseConv2d(x, w, stride, pad, 'NHWC', dilation);
        expect(result.shape).toEqual([1, 2, 2, 2]);
        expectArraysClose(await result.data(), [
            1.0257229804992676, 3.247040033340454, 1.9391249418258667,
            2.9474055767059326, 2.0091731548309326, 3.600433826446533,
            2.334312677383423, 2.548961877822876
        ]);
    });
    it('Tensor3D is allowed', async () => {
        const fSize = 2;
        const pad = 'same';
        const stride = 1;
        const chMul = 3;
        const inDepth = 2;
        const x = tf.zeros([3, 3, inDepth]);
        const w = tf.zeros([fSize, fSize, inDepth, chMul]);
        const result = tf.depthwiseConv2d(x, w, stride, pad);
        expect(result.shape).toEqual([3, 3, inDepth * chMul]);
    });
    it('Pass null for dilations, which defaults to [1, 1]', () => {
        const fSize = 2;
        const pad = 'same';
        const stride = 1;
        const chMul = 3;
        const inDepth = 2;
        const dilations = null;
        const x = tf.zeros([3, 3, inDepth]);
        const w = tf.zeros([fSize, fSize, inDepth, chMul]);
        const result = tf.depthwiseConv2d(x, w, stride, pad, 'NHWC', dilations);
        expect(result.shape).toEqual([3, 3, inDepth * chMul]);
    });
    it('TensorLike', async () => {
        const pad = 'valid';
        const stride = 1;
        const x = [[
                [[0.230664], [0.987388], [0.0685208]],
                [[0.419224], [0.887861], [0.731641]],
                [[0.0741907], [0.409265], [0.351377]]
            ]];
        const w = [[[[0.303873]], [[0.229223]]], [[[0.144333]], [[0.803373]]]];
        const result = tf.depthwiseConv2d(x, w, stride, pad);
        const expected = [1.07022, 1.03167, 0.67041, 0.778863];
        expectArraysClose(await result.data(), expected);
    });
    it('TensorLike Chained', async () => {
        const pad = 'valid';
        const stride = 1;
        const inDepth = 1;
        const x = tf.tensor4d([
            0.230664, 0.987388, 0.0685208, 0.419224, 0.887861, 0.731641,
            0.0741907, 0.409265, 0.351377
        ], [1, 3, 3, inDepth]);
        const w = [[[[0.303873]], [[0.229223]]], [[[0.144333]], [[0.803373]]]];
        const result = x.depthwiseConv2d(w, stride, pad);
        expect(result.shape).toEqual([1, 2, 2, 1]);
        const expected = [1.07022, 1.03167, 0.67041, 0.778863];
        expectArraysClose(await result.data(), expected);
    });
    it('throws when passed x as a non-tensor', () => {
        const inputDepth = 1;
        const outputDepth = 1;
        const fSize = 1;
        const pad = 'same';
        const stride = 2;
        const dataFormat = 'NHWC';
        const dilation = 2;
        const w = tf.tensor4d([3], [fSize, fSize, inputDepth, outputDepth]);
        const e = /Argument 'x' passed to 'depthwiseConv2d' must be a Tensor/;
        expect(() => tf.depthwiseConv2d({}, w, stride, pad, dataFormat, dilation))
            .toThrowError(e);
    });
    it('throws when passed filter as a non-tensor', () => {
        const inputDepth = 1;
        const inputShape = [2, 2, inputDepth];
        const pad = 'same';
        const stride = 2;
        const dataFormat = 'NHWC';
        const dilation = 2;
        const x = tf.tensor3d([1, 2, 3, 4], inputShape);
        const e = /Argument 'filter' passed to 'depthwiseConv2d' must be a Tensor/;
        expect(() => tf.depthwiseConv2d(x, {}, stride, pad, dataFormat, dilation))
            .toThrowError(e);
    });
    it('throws when input is int32', async () => {
        const fSize = 2;
        const pad = 'valid';
        const stride = 1;
        const chMul = 1;
        const inDepth = 1;
        const x = tf.tensor4d([1, 2, 3, 4, 5, 6, 7, 8, 9], [1, 3, 3, inDepth], 'int32');
        const w = tf.tensor4d([0.303873, 0.229223, 0.144333, 0.803373], [fSize, fSize, inDepth, chMul]);
        const errRegex = /Argument 'x' passed to 'depthwiseConv2d' must be float32/;
        expect(() => tf.depthwiseConv2d(x, w, stride, pad)).toThrowError(errRegex);
    });
    it('throws when filter is int32', async () => {
        const fSize = 2;
        const pad = 'valid';
        const stride = 1;
        const chMul = 1;
        const inDepth = 1;
        const x = tf.tensor4d([1, 2, 3, 4, 5, 6, 7, 8, 9], [1, 3, 3, inDepth]);
        const w = tf.tensor4d([1, 2, 3, 4], [fSize, fSize, inDepth, chMul], 'int32');
        const errRegex = /Argument 'filter' passed to 'depthwiseConv2d' must be float32/;
        expect(() => tf.depthwiseConv2d(x, w, stride, pad)).toThrowError(errRegex);
    });
    it('throws when dimRoundingMode is set and pad is same', () => {
        const fSize = 2;
        const pad = 'same';
        const stride = 1;
        const chMul = 1;
        const inDepth = 1;
        const dimRoundingMode = 'round';
        const x = tf.tensor4d([
            0.230664, 0.987388, 0.0685208, 0.419224, 0.887861, 0.731641,
            0.0741907, 0.409265, 0.351377
        ], [1, 3, 3, inDepth]);
        const w = tf.tensor4d([0.303873, 0.229223, 0.144333, 0.803373], [fSize, fSize, inDepth, chMul]);
        expect(() => tf.depthwiseConv2d(x, w, stride, pad, 'NHWC', 1, dimRoundingMode))
            .toThrowError();
    });
    it('throws when dimRoundingMode is set and pad is valid', () => {
        const fSize = 2;
        const pad = 'valid';
        const stride = 1;
        const chMul = 1;
        const inDepth = 1;
        const dimRoundingMode = 'round';
        const x = tf.tensor4d([
            0.230664, 0.987388, 0.0685208, 0.419224, 0.887861, 0.731641,
            0.0741907, 0.409265, 0.351377
        ], [1, 3, 3, inDepth]);
        const w = tf.tensor4d([0.303873, 0.229223, 0.144333, 0.803373], [fSize, fSize, inDepth, chMul]);
        expect(() => tf.depthwiseConv2d(x, w, stride, pad, 'NHWC', 1, dimRoundingMode))
            .toThrowError();
    });
    it('throws when dimRoundingMode is set and pad is a non-integer number', () => {
        const fSize = 2;
        const pad = 1.2;
        const stride = 1;
        const chMul = 1;
        const inDepth = 1;
        const dimRoundingMode = 'round';
        const x = tf.tensor4d([
            0.230664, 0.987388, 0.0685208, 0.419224, 0.887861, 0.731641,
            0.0741907, 0.409265, 0.351377
        ], [1, 3, 3, inDepth]);
        const w = tf.tensor4d([0.303873, 0.229223, 0.144333, 0.803373], [fSize, fSize, inDepth, chMul]);
        expect(() => tf.depthwiseConv2d(x, w, stride, pad, 'NHWC', 1, dimRoundingMode))
            .toThrowError();
    });
    it('throws when dimRoundingMode is set and pad is explicit by non-integer ' +
        'number', () => {
        const fSize = 2;
        const pad = [[0, 0], [0, 2.1], [1, 1], [0, 0]];
        const stride = 1;
        const chMul = 1;
        const inDepth = 1;
        const dimRoundingMode = 'round';
        const x = tf.tensor4d([
            0.230664, 0.987388, 0.0685208, 0.419224, 0.887861, 0.731641,
            0.0741907, 0.409265, 0.351377
        ], [1, 3, 3, inDepth]);
        const w = tf.tensor4d([0.303873, 0.229223, 0.144333, 0.803373], [fSize, fSize, inDepth, chMul]);
        expect(() => tf.depthwiseConv2d(x, w, stride, pad, 'NHWC', 1, dimRoundingMode))
            .toThrowError();
    });
    it('accepts a tensor-like object', async () => {
        const pad = 'valid';
        const stride = 1;
        // 1x3x3x1
        const x = [[
                [[0.230664], [0.987388], [0.0685208]],
                [[0.419224], [0.887861], [0.731641]],
                [[0.0741907], [0.409265], [0.351377]]
            ]];
        // 2x2x1x1
        const w = [[[[0.303873]], [[0.229223]]], [[[0.144333]], [[0.803373]]]];
        const result = tf.depthwiseConv2d(x, w, stride, pad);
        expect(result.shape).toEqual([1, 2, 2, 1]);
        const expected = [1.07022, 1.03167, 0.67041, 0.778863];
        expectArraysClose(await result.data(), expected);
    });
});
describeWithFlags('depthwiseConv2d gradients', ALL_ENVS, () => {
    let images;
    let filter;
    let result;
    const stride = 1;
    const pad = 'same';
    beforeEach(() => {
        // two 2x2 RGB images => 2x2x2x3
        images = tf.tensor4d([
            [[[2, 3, 1], [3, 0, 2]], [[0, 4, 1], [3, 1, 3]]],
            [[[2, 1, 0], [0, 3, 3]], [[4, 0, 1], [1, 4, 1]]]
        ]);
        // 2x2 filters, chMul = 2 => 2x2x3x2
        filter = tf.tensor4d([
            [[[1, 1], [1, 1], [0, 0]], [[0, 1], [1, 1], [1, 1]]],
            [[[1, 0], [1, 1], [0, 0]], [[0, 1], [1, 0], [0, 0]]]
        ]);
        // result of convolution operation
        result = tf.tensor4d([
            [
                [[2, 8, 8, 7, 2, 2], [6, 3, 1, 1, 0, 0]],
                [[0, 3, 5, 5, 3, 3], [3, 3, 1, 1, 0, 0]]
            ],
            [
                [[6, 3, 8, 4, 3, 3], [1, 0, 7, 7, 0, 0]],
                [[4, 5, 4, 4, 1, 1], [1, 1, 4, 4, 0, 0]]
            ]
        ]);
    });
    it('wrt input', async () => {
        const { value, grad } = tf.valueAndGrad((x) => tf.depthwiseConv2d(x, filter, stride, pad))(images);
        expectArraysClose(await value.data(), await result.data());
        const expectedGrad = tf.tensor4d([
            [[[2., 2., 0.], [3., 4., 2.]], [[3., 4., 0.], [5., 7., 2.]]],
            [[[2., 2., 0.], [3., 4., 2.]], [[3., 4., 0.], [5., 7., 2.]]]
        ]);
        expectArraysClose(await grad.data(), await expectedGrad.data());
    });
    // The gradients of normal and depthwise 2D convolutions are actually the same
    // in the special case that dy = 1, so we also test the gradient of a function
    // of the output to disambiguate the two methods.
    it('wrt input, squared output', async () => {
        const grad = tf.grad((x) => tf.square(tf.depthwiseConv2d(x, filter, stride, pad)))(images);
        const expectedGrad = tf.tensor4d([
            [[[20., 30., 0.], [34., 34., 8.]], [[10., 50., 0.], [46., 44., 12.]]],
            [[[18., 24., 0.], [8., 52., 12.]], [[30., 40., 0.], [22., 76., 4.]]]
        ]);
        expectArraysClose(await grad.data(), await expectedGrad.data());
    });
    it('wrt filter', async () => {
        const { value, grad } = tf.valueAndGrad((f) => tf.depthwiseConv2d(images, f, stride, pad))(filter);
        expectArraysClose(await value.data(), await result.data());
        const expectedGrad = tf.tensor4d([
            [[[15., 15.], [16., 16.], [12., 12.]], [[7., 7.], [8., 8.], [9., 9.]]],
            [[[8., 8.], [9., 9.], [6., 6.]], [[4., 4.], [5., 5.], [4., 4.]]]
        ]);
        expectArraysClose(await grad.data(), await expectedGrad.data());
    });
    it('gradient with clones', async () => {
        const [dx, dFilter] = tf.grads((x, filter) => tf.depthwiseConv2d(x.clone(), filter.clone(), stride, pad).clone())([images, filter]);
        expect(dx.shape).toEqual(images.shape);
        expect(dFilter.shape).toEqual(filter.shape);
    });
    // Also disambiguate regular vs. depthwise filter gradients
    it('wrt filter, squared output', async () => {
        const grad = tf.grad((f) => tf.square(tf.depthwiseConv2d(images, f, stride, pad)))(filter);
        const expectedGrad = tf.tensor4d([
            [
                [[120., 122.], [180., 166.], [12., 12.]],
                [[20., 76.], [90., 66.], [46., 46.]]
            ],
            [
                [[86., 42.], [122., 114.], [10., 10.]],
                [[24., 54.], [80., 46.], [18., 18.]]
            ]
        ]);
        expectArraysClose(await grad.data(), await expectedGrad.data());
    });
    it('throws error on dilations > 1', () => {
        const grad = tf.grad((x) => tf.depthwiseConv2d(x, filter, stride, pad, 'NHWC', 2));
        expect(() => grad(images))
            .toThrowError(/dilation rates greater than 1 are not yet supported/);
    });
    it('wrt input, stride=2, pad=valid', async () => {
        const dx = tf.grad((x) => tf.depthwiseConv2d(x, filter, 2, 'valid'))(images);
        expectArraysClose(await dx.data(), [
            2., 2., 0., 1., 2., 2., 1., 2., 0., 1., 1., 0.,
            2., 2., 0., 1., 2., 2., 1., 2., 0., 1., 1., 0.
        ]);
        expect(dx.shape).toEqual([2, 2, 2, 3]);
    });
    it('wrt filter, stride=2, pad=valid', async () => {
        const df = tf.grad((f) => tf.depthwiseConv2d(images, f, 2, 'valid'))(filter);
        expectArraysClose(await df.data(), [
            4., 4., 4., 4., 1., 1., 3., 3., 3., 3., 5., 5.,
            4., 4., 4., 4., 2., 2., 4., 4., 5., 5., 4., 4.
        ]);
        expect(df.shape).toEqual([2, 2, 3, 2]);
    });
    it('gradient with clones', async () => {
        const fSize = 2;
        const pad = 'valid';
        const stride = 1;
        const chMul = 1;
        const inDepth = 1;
        const x = tf.tensor4d([
            0.230664, 0.987388, 0.0685208, 0.419224, 0.887861, 0.731641,
            0.0741907, 0.409265, 0.351377
        ], [1, 3, 3, inDepth]);
        const f = tf.tensor4d([0.303873, 0.229223, 0.144333, 0.803373], [fSize, fSize, inDepth, chMul]);
        const [dx, df] = tf.grads((x, f) => tf.depthwiseConv2d(x.clone(), f.clone(), stride, pad).clone())([x, f]);
        expectArraysClose(await dx.data(), [
            0.303873, 0.533096, 0.229223, 0.448206, 1.480802, 1.032596, 0.144333,
            0.947706, 0.803373
        ]);
        expect(dx.shape).toEqual([1, 3, 3, 1]);
        expectArraysClose(await df.data(), [2.525137, 2.6754108, 1.7905407, 2.380144]);
        expect(df.shape).toEqual([2, 2, 1, 1]);
    });
});
//# sourceMappingURL=data:application/json;base64,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