/**
 * @license
 * Copyright 2017 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import * as broadcast_util from './broadcast_util';
describe('broadcast_util.getBroadcastShape', () => {
    it('two scalars', () => {
        const res = broadcast_util.assertAndGetBroadcastShape([], []);
        expect(res).toEqual([]);
    });
    it('scalar and 1d', () => {
        const res = broadcast_util.assertAndGetBroadcastShape([6], []);
        expect(res).toEqual([6]);
    });
    it('scalar and 2d', () => {
        const res = broadcast_util.assertAndGetBroadcastShape([2, 6], []);
        expect(res).toEqual([2, 6]);
    });
    it('1d and 2d', () => {
        const res = broadcast_util.assertAndGetBroadcastShape([6], [2, 6]);
        expect(res).toEqual([2, 6]);
    });
    it('2d and 3d', () => {
        const res = broadcast_util.assertAndGetBroadcastShape([2, 6], [7, 2, 6]);
        expect(res).toEqual([7, 2, 6]);
    });
    it('3d and 3d', () => {
        const res = broadcast_util.assertAndGetBroadcastShape([1, 1, 6], [7, 2, 6]);
        expect(res).toEqual([7, 2, 6]);
    });
    it('incompatible inner shape', () => {
        const f = () => broadcast_util.assertAndGetBroadcastShape([7, 2, 5], [7, 2, 6]);
        expect(f).toThrowError();
    });
    it('incompatible middle shape', () => {
        const f = () => broadcast_util.assertAndGetBroadcastShape([7, 3, 6], [7, 2, 6]);
        expect(f).toThrowError();
    });
    it('compatible with broadcasting support', () => {
        const res = broadcast_util.assertAndGetBroadcastShape([7, 1, 1], [7, 1, 1]);
        expect(res).toEqual([7, 1, 1]);
    });
    it('3d and 3d, each gets broadcasted', () => {
        const res = broadcast_util.assertAndGetBroadcastShape([4, 1, 7], [1, 3, 1]);
        expect(res).toEqual([4, 3, 7]);
    });
    it('[0] and [1] = [0]', () => {
        const res = broadcast_util.assertAndGetBroadcastShape([0], [1]);
        expect(res).toEqual([0]);
        const res2 = broadcast_util.assertAndGetBroadcastShape([1], [0]);
        expect(res2).toEqual([0]);
    });
    it('[0] and [0] = [0]', () => {
        const res = broadcast_util.assertAndGetBroadcastShape([0], [0]);
        expect(res).toEqual([0]);
    });
    it('[0, 1] and [1, 3] = [0, 3]', () => {
        const res = broadcast_util.assertAndGetBroadcastShape([0, 1], [1, 3]);
        expect(res).toEqual([0, 3]);
    });
    it('[5, 0, 3] and [5, 1, 1] = [5, 0, 3]', () => {
        const res = broadcast_util.assertAndGetBroadcastShape([5, 0, 3], [5, 1, 1]);
        expect(res).toEqual([5, 0, 3]);
    });
    it('[1] and [0, 0, 4] = [0, 0, 4]', () => {
        const res = broadcast_util.assertAndGetBroadcastShape([1], [0, 0, 4]);
        expect(res).toEqual([0, 0, 4]);
    });
});
describe('broadcast_util.getBroadcastDims', () => {
    it('[] => []', () => {
        const dims = broadcast_util.getBroadcastDims([], []);
        expect(dims.length).toBe(0);
    });
    it('[] => [5, 4]', () => {
        const dims = broadcast_util.getBroadcastDims([], [5, 4]);
        expect(dims.length).toBe(0);
    });
    it('[1] => [5]', () => {
        const dims = broadcast_util.getBroadcastDims([1], [5]);
        expect(dims).toEqual([0]);
    });
    it('[5, 1] => [5, 3]', () => {
        const dims = broadcast_util.getBroadcastDims([5, 1], [5, 3]);
        expect(dims).toEqual([1]);
    });
    it('[1, 3] => [5, 3]', () => {
        const dims = broadcast_util.getBroadcastDims([1, 3], [5, 3]);
        expect(dims).toEqual([0]);
    });
    it('[1, 1] => [5, 3]', () => {
        const dims = broadcast_util.getBroadcastDims([1, 1], [5, 3]);
        expect(dims).toEqual([0, 1]);
    });
    it('[4, 1, 3] => [4, 5, 3]', () => {
        const dims = broadcast_util.getBroadcastDims([4, 1, 3], [4, 5, 3]);
        expect(dims).toEqual([1]);
    });
});
describe('broadcast_util.getReductionAxes', () => {
    it('[] => []', () => {
        const axes = broadcast_util.getReductionAxes([], []);
        expect(axes).toEqual([]);
    });
    it('[] => [5, 4]', () => {
        const axes = broadcast_util.getReductionAxes([], [5, 4]);
        expect(axes).toEqual([0, 1]);
    });
    it('[1] => [5]', () => {
        const axes = broadcast_util.getReductionAxes([1], [5]);
        expect(axes).toEqual([0]);
    });
    it('[5, 1] => [5, 3]', () => {
        const axes = broadcast_util.getReductionAxes([5, 1], [5, 3]);
        expect(axes).toEqual([1]);
    });
    it('[1, 3] => [5, 3]', () => {
        const axes = broadcast_util.getReductionAxes([1, 3], [5, 3]);
        expect(axes).toEqual([0]);
    });
    it('[1, 1] => [5, 3]', () => {
        const axes = broadcast_util.getReductionAxes([1, 1], [5, 3]);
        expect(axes).toEqual([0, 1]);
    });
    it('[4, 1, 3] => [4, 5, 3]', () => {
        const axes = broadcast_util.getReductionAxes([4, 1, 3], [4, 5, 3]);
        expect(axes).toEqual([1]);
    });
});
//# sourceMappingURL=data:application/json;base64,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