/**
 * @license
 * Copyright 2017 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import * as tf from '../index';
import { ALL_ENVS, describeWithFlags } from '../jasmine_util';
import { expectArraysClose } from '../test_util';
describeWithFlags('batchNorm4D', ALL_ENVS, () => {
    it('simple batchnorm4D, no offset or scale, 2x1x1x2', async () => {
        const xT = tf.tensor4d([2, 4, 9, 23], [2, 1, 1, 2]);
        const meanT = tf.tensor1d([1, 2]);
        const varianceT = tf.tensor1d([2, 3]);
        const varianceEpsilon = .001;
        const result = tf.batchNorm4d(xT, meanT, varianceT, undefined, undefined, varianceEpsilon);
        const x = await xT.array();
        const mean = await meanT.array();
        const variance = await varianceT.array();
        expectArraysClose(await result.data(), [
            (x[0][0][0][0] - mean[0]) * 1 / Math.sqrt(variance[0] + varianceEpsilon),
            (x[0][0][0][1] - mean[1]) * 1 / Math.sqrt(variance[1] + varianceEpsilon),
            (x[1][0][0][0] - mean[0]) * 1 / Math.sqrt(variance[0] + varianceEpsilon),
            (x[1][0][0][1] - mean[1]) * 1 / Math.sqrt(variance[1] + varianceEpsilon)
        ]);
    });
    it('simple batchnorm4D, no offset, 2x1x1x2', async () => {
        const xT = tf.tensor4d([2, 4, 9, 23], [2, 1, 1, 2]);
        const meanT = tf.tensor1d([1, 2]);
        const varianceT = tf.tensor1d([2, 3]);
        const scaleT = tf.tensor1d([4, 5]);
        const varianceEpsilon = .001;
        const result = tf.batchNorm4d(xT, meanT, varianceT, undefined, scaleT, varianceEpsilon);
        const x = await xT.buffer();
        const mean = await meanT.buffer();
        const variance = await varianceT.buffer();
        const scale = await scaleT.buffer();
        expectArraysClose(await result.data(), [
            (x.get(0, 0, 0, 0) - mean.get(0)) * scale.get(0) /
                Math.sqrt(variance.get(0) + varianceEpsilon),
            (x.get(0, 0, 0, 1) - mean.get(1)) * scale.get(1) /
                Math.sqrt(variance.get(1) + varianceEpsilon),
            (x.get(1, 0, 0, 0) - mean.get(0)) * scale.get(0) /
                Math.sqrt(variance.get(0) + varianceEpsilon),
            (x.get(1, 0, 0, 1) - mean.get(1)) * scale.get(1) /
                Math.sqrt(variance.get(1) + varianceEpsilon)
        ]);
    });
    it('simple batchnorm4D, no scale, 2x1x1x2', async () => {
        const xT = tf.tensor4d([2, 4, 9, 23], [2, 1, 1, 2]);
        const meanT = tf.tensor1d([1, 2]);
        const varianceT = tf.tensor1d([2, 3]);
        const offsetT = tf.tensor1d([4, 5]);
        const varianceEpsilon = .001;
        const result = tf.batchNorm4d(xT, meanT, varianceT, offsetT, undefined, varianceEpsilon);
        const x = await xT.buffer();
        const mean = await meanT.buffer();
        const variance = await varianceT.buffer();
        const offset = await offsetT.buffer();
        expectArraysClose(await result.data(), [
            offset.get(0) +
                (x.get(0, 0, 0, 0) - mean.get(0)) * 1 /
                    Math.sqrt(variance.get(0) + varianceEpsilon),
            offset.get(1) +
                (x.get(0, 0, 0, 1) - mean.get(1)) * 1 /
                    Math.sqrt(variance.get(1) + varianceEpsilon),
            offset.get(0) +
                (x.get(1, 0, 0, 0) - mean.get(0)) * 1 /
                    Math.sqrt(variance.get(0) + varianceEpsilon),
            offset.get(1) +
                (x.get(1, 0, 0, 1) - mean.get(1)) * 1 /
                    Math.sqrt(variance.get(1) + varianceEpsilon)
        ]);
    });
    it('simple batchnorm4D, 2x1x1x2', async () => {
        const xT = tf.tensor4d([2, 4, 9, 23], [2, 1, 1, 2]);
        const meanT = tf.tensor1d([1, 2]);
        const varianceT = tf.tensor1d([2, 3]);
        const offsetT = tf.tensor1d([3, 4]);
        const scaleT = tf.tensor1d([4, 5]);
        const varianceEpsilon = .001;
        const result = tf.batchNorm4d(xT, meanT, varianceT, offsetT, scaleT, varianceEpsilon);
        const x = await xT.buffer();
        const mean = await meanT.buffer();
        const variance = await varianceT.buffer();
        const scale = await scaleT.buffer();
        const offset = await offsetT.buffer();
        expectArraysClose(await result.data(), [
            offset.get(0) +
                (x.get(0, 0, 0, 0) - mean.get(0)) * scale.get(0) /
                    Math.sqrt(variance.get(0) + varianceEpsilon),
            offset.get(1) +
                (x.get(0, 0, 0, 1) - mean.get(1)) * scale.get(1) /
                    Math.sqrt(variance.get(1) + varianceEpsilon),
            offset.get(0) +
                (x.get(1, 0, 0, 0) - mean.get(0)) * scale.get(0) /
                    Math.sqrt(variance.get(0) + varianceEpsilon),
            offset.get(1) +
                (x.get(1, 0, 0, 1) - mean.get(1)) * scale.get(1) /
                    Math.sqrt(variance.get(1) + varianceEpsilon)
        ]);
    });
    it('accepts a tensor-like object', async () => {
        const x = [[[[2, 4]]], [[[9, 23]]]]; // 2x1x1x2
        const mean = [1, 2];
        const variance = [2, 3];
        const offset = [3, 4];
        const scale = [4, 5];
        const varianceEpsilon = .001;
        const result = tf.batchNorm4d(x, mean, variance, offset, scale, varianceEpsilon);
        expectArraysClose(await result.data(), [
            offset[0] +
                (x[0][0][0][0] - mean[0]) * scale[0] /
                    Math.sqrt(variance[0] + varianceEpsilon),
            offset[1] +
                (x[0][0][0][1] - mean[1]) * scale[1] /
                    Math.sqrt(variance[1] + varianceEpsilon),
            offset[0] +
                (x[1][0][0][0] - mean[0]) * scale[0] /
                    Math.sqrt(variance[0] + varianceEpsilon),
            offset[1] +
                (x[1][0][0][1] - mean[1]) * scale[1] /
                    Math.sqrt(variance[1] + varianceEpsilon)
        ]);
    });
    it('simple batchnorm4D gradients, 2x1x1x2', async () => {
        const x = tf.tensor4d([2, 4, 9, 23], [2, 1, 1, 2]);
        const mean = tf.tensor1d([1, 2]);
        const variance = tf.tensor1d([2, 3]);
        const offset = tf.tensor1d([3, 4]);
        const scale = tf.tensor1d([2, 5]);
        const varianceEpsilon = .001;
        const dy = tf.tensor4d([-1, -1, -1, -1], [2, 1, 1, 2]);
        const gradX = tf.grad((x) => tf.batchNorm4d(x, mean, variance, offset, scale, varianceEpsilon))(x, dy);
        expectArraysClose(await gradX.data(), [-1.414, -2.887, -1.414, -2.887]);
        expect(gradX.shape).toEqual([2, 1, 1, 2]);
        const gradMean = tf.grad((mean) => tf.batchNorm4d(x, mean, variance, offset, scale, varianceEpsilon))(mean, dy);
        expectArraysClose(await gradMean.data(), [2.828, 5.773]);
        expect(gradMean.shape).toEqual([2]);
        const gradVariance = tf.grad((variance) => tf.batchNorm4d(x, mean, variance, offset, scale, varianceEpsilon))(variance, dy);
        expectArraysClose(await gradVariance.data(), [3.180, 11.060]);
        expect(gradVariance.shape).toEqual([2]);
        const gradOffset = tf.grad((offset) => tf.batchNorm4d(x, mean, variance, offset, scale, varianceEpsilon))(offset, dy);
        expectArraysClose(await gradOffset.data(), await dy.sum([0, 1, 2]).data());
        expect(gradOffset.shape).toEqual([2]);
        const gradScale = tf.grad((scale) => tf.batchNorm4d(x, mean, variance, offset, scale, varianceEpsilon))(scale, dy);
        expectArraysClose(await gradScale.data(), [-6.362, -13.277]);
        expect(gradScale.shape).toEqual([2]);
    });
    it('batchnorm4D gradients, same shapes in x, mean and variance', async () => {
        const x = tf.tensor4d([10, 20, 30, 40], [2, 1, 1, 2]);
        const mean = tf.tensor4d([0, 5, 10, 15], [2, 1, 1, 2]);
        const variance = tf.tensor4d([2, 4, 6, 8], [2, 1, 1, 2]);
        const scale = tf.tensor4d([2, 5, 2, 5], [2, 1, 1, 2]);
        const offset = tf.tensor4d([0, 0, 0, 0], [2, 1, 1, 2]);
        const varianceEpsilon = .001;
        const dy = tf.tensor4d([-1, -1, -1, -1], [2, 1, 1, 2]);
        const gradX = tf.grad((x) => tf.batchNorm4d(x, mean, variance, offset, scale, varianceEpsilon))(x, dy);
        expectArraysClose(await gradX.data(), [-1.414, -2.500, -0.816, -1.768]);
        expect(gradX.shape).toEqual([2, 1, 1, 2]);
        const gradMean = tf.grad((mean) => tf.batchNorm4d(x, mean, variance, offset, scale, varianceEpsilon))(mean, dy);
        expectArraysClose(await gradMean.data(), [1.414, 2.500, 0.816, 1.768]);
        expect(gradMean.shape).toEqual([2, 1, 1, 2]);
        const gradVariance = tf.grad((variance) => tf.batchNorm4d(x, mean, variance, offset, scale, varianceEpsilon))(variance, dy);
        expectArraysClose(await gradVariance.data(), [3.533, 4.686, 1.360, 2.762]);
        expect(gradVariance.shape).toEqual([2, 1, 1, 2]);
        const gradOffset = tf.grad((offset) => tf.batchNorm4d(x, mean, variance, offset, scale, varianceEpsilon))(offset, dy);
        expectArraysClose(await gradOffset.data(), await dy.data());
        expect(gradOffset.shape).toEqual([2, 1, 1, 2]);
        const gradScale = tf.grad((scale) => tf.batchNorm4d(x, mean, variance, offset, scale, varianceEpsilon))(scale, dy);
        expectArraysClose(await gradScale.data(), [-7.069, -7.499, -8.164, -8.838]);
        expect(gradScale.shape).toEqual([2, 1, 1, 2]);
    });
});
describeWithFlags('batchNorm3D', ALL_ENVS, () => {
    it('simple batchnorm3D, no offset or scale, 2x1x2', async () => {
        const xT = tf.tensor3d([2, 4, 9, 23], [2, 1, 2]);
        const meanT = tf.tensor1d([1, 2]);
        const varianceT = tf.tensor1d([2, 3]);
        const varianceEpsilon = .001;
        const result = tf.batchNorm3d(xT, meanT, varianceT, undefined, undefined, varianceEpsilon);
        const x = await xT.buffer();
        const mean = await meanT.buffer();
        const variance = await varianceT.buffer();
        expectArraysClose(await result.data(), [
            (x.get(0, 0, 0) - mean.get(0)) * 1 /
                Math.sqrt(variance.get(0) + varianceEpsilon),
            (x.get(0, 0, 1) - mean.get(1)) * 1 /
                Math.sqrt(variance.get(1) + varianceEpsilon),
            (x.get(1, 0, 0) - mean.get(0)) * 1 /
                Math.sqrt(variance.get(0) + varianceEpsilon),
            (x.get(1, 0, 1) - mean.get(1)) * 1 /
                Math.sqrt(variance.get(1) + varianceEpsilon)
        ]);
    });
    it('simple batchnorm3D, no offset, 2x1x2', async () => {
        const xT = tf.tensor3d([2, 4, 9, 23], [2, 1, 2]);
        const meanT = tf.tensor1d([1, 2]);
        const varianceT = tf.tensor1d([2, 3]);
        const scaleT = tf.tensor1d([4, 5]);
        const varianceEpsilon = .001;
        const result = tf.batchNorm3d(xT, meanT, varianceT, undefined, scaleT, varianceEpsilon);
        const x = await xT.buffer();
        const mean = await meanT.buffer();
        const variance = await varianceT.buffer();
        const scale = await scaleT.buffer();
        expectArraysClose(await result.data(), [
            (x.get(0, 0, 0) - mean.get(0)) * scale.get(0) /
                Math.sqrt(variance.get(0) + varianceEpsilon),
            (x.get(0, 0, 1) - mean.get(1)) * scale.get(1) /
                Math.sqrt(variance.get(1) + varianceEpsilon),
            (x.get(1, 0, 0) - mean.get(0)) * scale.get(0) /
                Math.sqrt(variance.get(0) + varianceEpsilon),
            (x.get(1, 0, 1) - mean.get(1)) * scale.get(1) /
                Math.sqrt(variance.get(1) + varianceEpsilon)
        ]);
    });
    it('simple batchnorm3D, no scale, 2x1x2', async () => {
        const xT = tf.tensor3d([2, 4, 9, 23], [2, 1, 2]);
        const meanT = tf.tensor1d([1, 2]);
        const varianceT = tf.tensor1d([2, 3]);
        const offsetT = tf.tensor1d([4, 5]);
        const varianceEpsilon = .001;
        const result = tf.batchNorm3d(xT, meanT, varianceT, offsetT, undefined, varianceEpsilon);
        const x = await xT.buffer();
        const mean = await meanT.buffer();
        const variance = await varianceT.buffer();
        const offset = await offsetT.buffer();
        expectArraysClose(await result.data(), [
            offset.get(0) +
                (x.get(0, 0, 0) - mean.get(0)) * 1 /
                    Math.sqrt(variance.get(0) + varianceEpsilon),
            offset.get(1) +
                (x.get(0, 0, 1) - mean.get(1)) * 1 /
                    Math.sqrt(variance.get(1) + varianceEpsilon),
            offset.get(0) +
                (x.get(1, 0, 0) - mean.get(0)) * 1 /
                    Math.sqrt(variance.get(0) + varianceEpsilon),
            offset.get(1) +
                (x.get(1, 0, 1) - mean.get(1)) * 1 /
                    Math.sqrt(variance.get(1) + varianceEpsilon)
        ]);
    });
    it('simple batchnorm3D, 2x1x2', async () => {
        const xT = tf.tensor3d([2, 4, 9, 23], [2, 1, 2]);
        const meanT = tf.tensor1d([1, 2]);
        const varianceT = tf.tensor1d([2, 3]);
        const offsetT = tf.tensor1d([3, 4]);
        const scaleT = tf.tensor1d([4, 5]);
        const varianceEpsilon = .001;
        const result = tf.batchNorm3d(xT, meanT, varianceT, offsetT, scaleT, varianceEpsilon);
        const x = await xT.buffer();
        const mean = await meanT.buffer();
        const variance = await varianceT.buffer();
        const offset = await offsetT.buffer();
        const scale = await scaleT.buffer();
        expectArraysClose(await result.data(), [
            offset.get(0) +
                (x.get(0, 0, 0) - mean.get(0)) * scale.get(0) /
                    Math.sqrt(variance.get(0) + varianceEpsilon),
            offset.get(1) +
                (x.get(0, 0, 1) - mean.get(1)) * scale.get(1) /
                    Math.sqrt(variance.get(1) + varianceEpsilon),
            offset.get(0) +
                (x.get(1, 0, 0) - mean.get(0)) * scale.get(0) /
                    Math.sqrt(variance.get(0) + varianceEpsilon),
            offset.get(1) +
                (x.get(1, 0, 1) - mean.get(1)) * scale.get(1) /
                    Math.sqrt(variance.get(1) + varianceEpsilon)
        ]);
    });
    it('accepts a tensor-like object', async () => {
        const x = [[[2, 4]], [[9, 23]]]; // 2x1x2
        const mean = [1, 2];
        const variance = [2, 3];
        const offset = [3, 4];
        const scale = [4, 5];
        const varianceEpsilon = .001;
        const result = tf.batchNorm3d(x, mean, variance, offset, scale, varianceEpsilon);
        expectArraysClose(await result.data(), [
            offset[0] +
                (x[0][0][0] - mean[0]) * scale[0] /
                    Math.sqrt(variance[0] + varianceEpsilon),
            offset[1] +
                (x[0][0][1] - mean[1]) * scale[1] /
                    Math.sqrt(variance[1] + varianceEpsilon),
            offset[0] +
                (x[1][0][0] - mean[0]) * scale[0] /
                    Math.sqrt(variance[0] + varianceEpsilon),
            offset[1] +
                (x[1][0][1] - mean[1]) * scale[1] /
                    Math.sqrt(variance[1] + varianceEpsilon)
        ]);
    });
    it('batchnorm3D, x,mean,var,offset,scale are all 3D', async () => {
        const shape = [2, 1, 2];
        const xT = tf.tensor3d([2, 4, 9, 23], shape);
        const meanT = tf.tensor3d([1, 2, 3, 4], shape);
        const varianceT = tf.tensor3d([2, 3, 4, 5], shape);
        const offsetT = tf.tensor3d([3, 4, 5, 6], shape);
        const scaleT = tf.tensor3d([4, 5, 6, 7], shape);
        const varianceEpsilon = .001;
        const result = tf.batchNorm3d(xT, meanT, varianceT, offsetT, scaleT, varianceEpsilon);
        const x = await xT.buffer();
        const mean = await meanT.buffer();
        const variance = await varianceT.buffer();
        const offset = await offsetT.buffer();
        const scale = await scaleT.buffer();
        expectArraysClose(await result.data(), [
            offset.get(0, 0, 0) +
                (x.get(0, 0, 0) - mean.get(0, 0, 0)) * scale.get(0, 0, 0) /
                    Math.sqrt(variance.get(0, 0, 0) + varianceEpsilon),
            offset.get(0, 0, 1) +
                (x.get(0, 0, 1) - mean.get(0, 0, 1)) * scale.get(0, 0, 1) /
                    Math.sqrt(variance.get(0, 0, 1) + varianceEpsilon),
            offset.get(1, 0, 0) +
                (x.get(1, 0, 0) - mean.get(1, 0, 0)) * scale.get(1, 0, 0) /
                    Math.sqrt(variance.get(1, 0, 0) + varianceEpsilon),
            offset.get(1, 0, 1) +
                (x.get(1, 0, 1) - mean.get(1, 0, 1)) * scale.get(1, 0, 1) /
                    Math.sqrt(variance.get(1, 0, 1) + varianceEpsilon)
        ]);
    });
    it('simple batchnorm3D gradients, 2x1x2', async () => {
        const x = tf.tensor3d([2, 4, 9, 23], [2, 1, 2]);
        const mean = tf.tensor1d([1, 2]);
        const variance = tf.tensor1d([2, 3]);
        const offset = tf.tensor1d([3, 4]);
        const scale = tf.tensor1d([2, 5]);
        const varianceEpsilon = .001;
        const dy = tf.tensor3d([1, 1, 1, 1], [2, 1, 2]);
        const gradX = tf.grad((x) => tf.batchNorm3d(x, mean, variance, offset, scale, varianceEpsilon))(x, dy);
        expectArraysClose(await gradX.data(), [1.414, 2.887, 1.414, 2.887]);
        expect(gradX.shape).toEqual([2, 1, 2]);
        const gradMean = tf.grad((mean) => tf.batchNorm3d(x, mean, variance, offset, scale, varianceEpsilon))(mean, dy);
        expectArraysClose(await gradMean.data(), [-2.828, -5.773]);
        expect(gradMean.shape).toEqual([2]);
        const gradVariance = tf.grad((variance) => tf.batchNorm3d(x, mean, variance, offset, scale, varianceEpsilon))(variance, dy);
        expectArraysClose(await gradVariance.data(), [-3.180, -11.060]);
        expect(gradVariance.shape).toEqual([2]);
        const gradOffset = tf.grad((offset) => tf.batchNorm3d(x, mean, variance, offset, scale, varianceEpsilon))(offset, dy);
        expectArraysClose(await gradOffset.data(), [2, 2]);
        expect(gradOffset.shape).toEqual([2]);
        const gradScale = tf.grad((scale) => tf.batchNorm3d(x, mean, variance, offset, scale, varianceEpsilon))(scale, dy);
        expectArraysClose(await gradScale.data(), [6.362, 13.277]);
        expect(gradScale.shape).toEqual([2]);
    });
    it('batchnorm3D gradients, same shapes in x, mean and variance', async () => {
        const x = tf.tensor3d([10, 20, 30, 40], [2, 1, 2]);
        const mean = tf.tensor3d([0, 5, 10, 15], [2, 1, 2]);
        const variance = tf.tensor3d([2, 4, 6, 8], [2, 1, 2]);
        const scale = tf.tensor3d([2, 5, 2, 5], [2, 1, 2]);
        const offset = tf.tensor3d([0, 0, 0, 0], [2, 1, 2]);
        const varianceEpsilon = .001;
        const dy = tf.tensor3d([1, 1, 1, 1], [2, 1, 2]);
        const gradX = tf.grad((x) => tf.batchNorm3d(x, mean, variance, offset, scale, varianceEpsilon))(x, dy);
        expectArraysClose(await gradX.data(), [1.414, 2.500, 0.816, 1.768]);
        expect(gradX.shape).toEqual([2, 1, 2]);
        const gradMean = tf.grad((mean) => tf.batchNorm3d(x, mean, variance, offset, scale, varianceEpsilon))(mean, dy);
        expectArraysClose(await gradMean.data(), [-1.414, -2.500, -0.816, -1.768]);
        expect(gradMean.shape).toEqual([2, 1, 2]);
        const gradVariance = tf.grad((variance) => tf.batchNorm3d(x, mean, variance, offset, scale, varianceEpsilon))(variance, dy);
        expectArraysClose(await gradVariance.data(), [-3.533, -4.686, -1.360, -2.762]);
        expect(gradVariance.shape).toEqual([2, 1, 2]);
        const gradOffset = tf.grad((offset) => tf.batchNorm3d(x, mean, variance, offset, scale, varianceEpsilon))(offset, dy);
        expectArraysClose(await gradOffset.data(), [1, 1, 1, 1]);
        expect(gradOffset.shape).toEqual([2, 1, 2]);
        const gradScale = tf.grad((scale) => tf.batchNorm3d(x, mean, variance, offset, scale, varianceEpsilon))(scale, dy);
        expectArraysClose(await gradScale.data(), [7.069, 7.499, 8.164, 8.838]);
        expect(gradScale.shape).toEqual([2, 1, 2]);
    });
    it('batchnorm matches tensorflow, 2x3x3', async () => {
        const x = tf.tensor3d([
            0.49955603, 0.04158615, -1.09440524, 2.03854165, -0.61578344,
            2.87533573, 1.18105987, 0.807462, 1.87888837, 2.26563962, -0.37040935,
            1.35848753, -0.75347094, 0.15683117, 0.91925946, 0.34121279,
            0.92717143, 1.89683965
        ], [2, 3, 3]);
        const mean = tf.tensor1d([0.39745062, -0.48062894, 0.4847822]);
        const variance = tf.tensor1d([0.32375343, 0.67117643, 1.08334653]);
        const offset = tf.tensor1d([0.69398749, -1.29056387, 0.9429723]);
        const scale = tf.tensor1d([-0.5607271, 0.9878457, 0.25181573]);
        const varianceEpsilon = .001;
        const result = tf.batchNorm3d(x, mean, variance, offset, scale, varianceEpsilon);
        expectArraysClose(await result.data(), [
            0.59352049, -0.66135202, 0.5610874, -0.92077015, -1.45341019, 1.52106473,
            -0.07704776, 0.26144429, 1.28010017, -1.14422404, -1.15776136, 1.15425493,
            1.82644104, -0.52249442, 1.04803919, 0.74932291, 0.40568101, 1.2844412
        ]);
    });
});
describeWithFlags('batchNorm2D', ALL_ENVS, () => {
    it('simple batchnorm2D, no offset or scale, 2x2', async () => {
        const xT = tf.tensor2d([2, 4, 9, 23], [2, 2]);
        const meanT = tf.tensor1d([1, 2]);
        const varianceT = tf.tensor1d([2, 3]);
        const varianceEpsilon = .001;
        const result = tf.batchNorm2d(xT, meanT, varianceT, undefined, undefined, varianceEpsilon);
        const x = await xT.buffer();
        const mean = await meanT.buffer();
        const variance = await varianceT.buffer();
        expectArraysClose(await result.data(), [
            (x.get(0, 0) - mean.get(0)) * 1 /
                Math.sqrt(variance.get(0) + varianceEpsilon),
            (x.get(0, 1) - mean.get(1)) * 1 /
                Math.sqrt(variance.get(1) + varianceEpsilon),
            (x.get(1, 0) - mean.get(0)) * 1 /
                Math.sqrt(variance.get(0) + varianceEpsilon),
            (x.get(1, 1) - mean.get(1)) * 1 /
                Math.sqrt(variance.get(1) + varianceEpsilon)
        ]);
    });
    it('simple batchnorm2D, no offset, 2x2', async () => {
        const xT = tf.tensor2d([2, 4, 9, 23], [2, 2]);
        const meanT = tf.tensor1d([1, 2]);
        const varianceT = tf.tensor1d([2, 3]);
        const scaleT = tf.tensor1d([4, 5]);
        const varianceEpsilon = .001;
        const result = tf.batchNorm2d(xT, meanT, varianceT, undefined, scaleT, varianceEpsilon);
        const x = await xT.buffer();
        const mean = await meanT.buffer();
        const variance = await varianceT.buffer();
        const scale = await scaleT.buffer();
        expectArraysClose(await result.data(), [
            (x.get(0, 0) - mean.get(0)) * scale.get(0) /
                Math.sqrt(variance.get(0) + varianceEpsilon),
            (x.get(0, 1) - mean.get(1)) * scale.get(1) /
                Math.sqrt(variance.get(1) + varianceEpsilon),
            (x.get(1, 0) - mean.get(0)) * scale.get(0) /
                Math.sqrt(variance.get(0) + varianceEpsilon),
            (x.get(1, 1) - mean.get(1)) * scale.get(1) /
                Math.sqrt(variance.get(1) + varianceEpsilon)
        ]);
    });
    it('simple batchnorm2D, no scale, 2x2', async () => {
        const xT = tf.tensor2d([2, 4, 9, 23], [2, 2]);
        const meanT = tf.tensor1d([1, 2]);
        const varianceT = tf.tensor1d([2, 3]);
        const offsetT = tf.tensor1d([4, 5]);
        const varianceEpsilon = .001;
        const result = tf.batchNorm2d(xT, meanT, varianceT, offsetT, undefined, varianceEpsilon);
        const offset = await offsetT.array();
        const mean = await meanT.array();
        const variance = await varianceT.array();
        const x = await xT.array();
        expectArraysClose(await result.data(), [
            offset[0] +
                (x[0][0] - mean[0]) * 1 / Math.sqrt(variance[0] + varianceEpsilon),
            offset[1] +
                (x[0][1] - mean[1]) * 1 / Math.sqrt(variance[1] + varianceEpsilon),
            offset[0] +
                (x[1][0] - mean[0]) * 1 / Math.sqrt(variance[0] + varianceEpsilon),
            offset[1] +
                (x[1][1] - mean[1]) * 1 / Math.sqrt(variance[1] + varianceEpsilon)
        ]);
    });
    it('simple batchnorm2D, 2x2', async () => {
        const xT = tf.tensor2d([2, 4, 9, 23], [2, 2]);
        const meanT = tf.tensor1d([1, 2]);
        const varianceT = tf.tensor1d([2, 3]);
        const offsetT = tf.tensor1d([3, 4]);
        const scaleT = tf.tensor1d([4, 5]);
        const varianceEpsilon = .001;
        const result = tf.batchNorm2d(xT, meanT, varianceT, offsetT, scaleT, varianceEpsilon);
        const offset = await offsetT.array();
        const mean = await meanT.array();
        const variance = await varianceT.array();
        const scale = await scaleT.array();
        const x = await xT.array();
        expectArraysClose(await result.data(), [
            offset[0] +
                (x[0][0] - mean[0]) * scale[0] /
                    Math.sqrt(variance[0] + varianceEpsilon),
            offset[1] +
                (x[0][1] - mean[1]) * scale[1] /
                    Math.sqrt(variance[1] + varianceEpsilon),
            offset[0] +
                (x[1][0] - mean[0]) * scale[0] /
                    Math.sqrt(variance[0] + varianceEpsilon),
            offset[1] +
                (x[1][1] - mean[1]) * scale[1] /
                    Math.sqrt(variance[1] + varianceEpsilon)
        ]);
    });
    it('simple batchnorm2D gradients, 2x2', async () => {
        const x = tf.tensor2d([2, 4, 9, 23], [2, 2]);
        const mean = tf.tensor1d([1, 2]);
        const variance = tf.tensor1d([2, 3]);
        const offset = tf.tensor1d([3, 4]);
        const scale = tf.tensor1d([2, 5]);
        const varianceEpsilon = .001;
        const dy = tf.tensor2d([1, 1, 1, 1], [2, 2]);
        const [gradX, gradMean, gradVariance, gradOffset, gradScale] = tf.grads((x, mean, variance, offset, scale) => tf.batchNorm2d(x, mean, variance, offset, scale, varianceEpsilon))([x, mean, variance, offset, scale], dy);
        expectArraysClose(await gradX.data(), [1.414, 2.887, 1.414, 2.887]);
        expect(gradX.shape).toEqual([2, 2]);
        expectArraysClose(await gradMean.data(), [-2.828, -5.773]);
        expect(gradMean.shape).toEqual([2]);
        expectArraysClose(await gradVariance.data(), [-3.180, -11.060]);
        expect(gradVariance.shape).toEqual([2]);
        expectArraysClose(await gradOffset.data(), [2, 2]);
        expect(gradOffset.shape).toEqual([2]);
        expectArraysClose(await gradScale.data(), [6.362, 13.277]);
        expect(gradScale.shape).toEqual([2]);
    });
    it('gradient with clones batchnorm2D', async () => {
        const x = tf.tensor2d([2, 4, 9, 23], [2, 2]);
        const mean = tf.tensor1d([1, 2]);
        const variance = tf.tensor1d([2, 3]);
        const offset = tf.tensor1d([3, 4]);
        const scale = tf.tensor1d([2, 5]);
        const varianceEpsilon = .001;
        const dy = tf.tensor2d([1, 1, 1, 1], [2, 2]);
        const [gradX, gradMean, gradVariance, gradOffset, gradScale] = tf.grads((x, mean, variance, offset, scale) => tf.batchNorm2d(x.clone(), mean.clone(), variance.clone(), offset.clone(), scale.clone(), varianceEpsilon)
            .clone())([x, mean, variance, offset, scale], dy);
        expectArraysClose(await gradX.data(), [1.414, 2.887, 1.414, 2.887]);
        expect(gradX.shape).toEqual([2, 2]);
        expectArraysClose(await gradMean.data(), [-2.828, -5.773]);
        expect(gradMean.shape).toEqual([2]);
        expectArraysClose(await gradVariance.data(), [-3.180, -11.060]);
        expect(gradVariance.shape).toEqual([2]);
        expectArraysClose(await gradOffset.data(), [2, 2]);
        expect(gradOffset.shape).toEqual([2]);
        expectArraysClose(await gradScale.data(), [6.362, 13.277]);
        expect(gradScale.shape).toEqual([2]);
    });
    it('batchnorm2D gradients, same shapes in x, mean and variance', async () => {
        const x = tf.tensor2d([10, 20, 30, 40], [2, 2]);
        const mean = tf.tensor2d([0, 5, 10, 15], [2, 2]);
        const variance = tf.tensor2d([2, 4, 6, 8], [2, 2]);
        const scale = tf.tensor2d([2, 5, 2, 5], [2, 2]);
        const offset = tf.tensor2d([0, 0, 0, 0], [2, 2]);
        const varianceEpsilon = .001;
        const dy = tf.tensor2d([1, 1, 1, 1], [2, 2]);
        const gradX = tf.grad((x) => tf.batchNorm2d(x, mean, variance, offset, scale, varianceEpsilon))(x, dy);
        expectArraysClose(await gradX.data(), [1.414, 2.500, 0.816, 1.768]);
        expect(gradX.shape).toEqual([2, 2]);
        const gradMean = tf.grad((mean) => tf.batchNorm2d(x, mean, variance, offset, scale, varianceEpsilon))(mean, dy);
        expectArraysClose(await gradMean.data(), [-1.414, -2.500, -0.816, -1.768]);
        expect(gradMean.shape).toEqual([2, 2]);
        const gradVariance = tf.grad((variance) => tf.batchNorm2d(x, mean, variance, offset, scale, varianceEpsilon))(variance, dy);
        expectArraysClose(await gradVariance.data(), [-3.533, -4.686, -1.360, -2.762]);
        expect(gradVariance.shape).toEqual([2, 2]);
        const gradOffset = tf.grad((offset) => tf.batchNorm2d(x, mean, variance, offset, scale, varianceEpsilon))(offset, dy);
        expectArraysClose(await gradOffset.data(), [1, 1, 1, 1]);
        expect(gradOffset.shape).toEqual([2, 2]);
        const gradScale = tf.grad((scale) => tf.batchNorm2d(x, mean, variance, offset, scale, varianceEpsilon))(scale, dy);
        expectArraysClose(await gradScale.data(), [7.069, 7.499, 8.164, 8.838]);
        expect(gradScale.shape).toEqual([2, 2]);
    });
    it('gradient with clones', () => {
        const x = tf.zeros([2, 2]);
        const mean = tf.zeros([2, 2]);
        const variance = tf.zeros([2, 2]);
        const scale = tf.zeros([2, 2]);
        const offset = tf.zeros([2, 2]);
        const varianceEpsilon = .001;
        const gradF = tf.grads((x, mean, variance, offset, scale) => tf.batchNorm2d(x.clone(), mean.clone(), variance.clone(), offset.clone(), scale.clone(), varianceEpsilon)
            .clone());
        const [gradX, gradMean, gradVariance, gradOffset, gradScale] = gradF([x, mean, variance, offset, scale]);
        expect(gradX.shape).toEqual(x.shape);
        expect(gradMean.shape).toEqual(mean.shape);
        expect(gradVariance.shape).toEqual(variance.shape);
        expect(gradOffset.shape).toEqual(offset.shape);
        expect(gradScale.shape).toEqual(scale.shape);
    });
    it('batchnorm2D matches tensorflow, 3x3', async () => {
        const x = tf.tensor2d([
            0.3136892, 0.92389025, 0.594782, 0.05021042, 0.67545404, 0.93910035,
            0.13277993, 0.96474269, 0.88608916
        ], [3, 3]);
        const mean = tf.tensor1d([0.19526312, 0.74857256, 0.45166398]);
        const variance = tf.tensor1d([0.22963001, 0.61521992, 0.46623685]);
        const offset = tf.tensor1d([0.43098484, 0.77712237, 0.47916298]);
        const scale = tf.tensor1d([0.62186907, 0.85673736, 0.19201061]);
        const varianceEpsilon = .001;
        const result = tf.batchNorm2d(x, mean, variance, offset, scale, varianceEpsilon);
        expectArraysClose(await result.data(), [
            0.58433646, 0.96846228, 0.51936529, 0.24315402, 0.69732157, 0.61608542,
            0.35007446, 1.01304821, 0.60119441
        ]);
    });
    it('throws when passed x as a non-tensor', () => {
        const mean = tf.tensor1d([1, 2]);
        const variance = tf.tensor1d([2, 3]);
        expect(() => tf.batchNorm({}, mean, variance))
            .toThrowError(/Argument 'x' passed to 'batchNorm' must be a Tensor/);
    });
    it('throws when passed mean as a non-tensor', () => {
        const x = tf.tensor4d([2, 4, 9, 23], [2, 1, 1, 2]);
        const variance = tf.tensor1d([2, 3]);
        expect(() => tf.batchNorm(x, {}, variance))
            .toThrowError(/Argument 'mean' passed to 'batchNorm' must be a Tensor/);
    });
    it('throws when passed variance as a non-tensor', () => {
        const x = tf.tensor4d([2, 4, 9, 23], [2, 1, 1, 2]);
        const mean = tf.tensor1d([1, 2]);
        const e = /Argument 'variance' passed to 'batchNorm' must be a Tensor/;
        expect(() => tf.batchNorm(x, mean, {})).toThrowError(e);
    });
    it('throws when passed scale as a non-tensor', () => {
        const x = tf.tensor4d([2, 4, 9, 23], [2, 1, 1, 2]);
        const mean = tf.tensor1d([1, 2]);
        const variance = tf.tensor1d([2, 3]);
        const epsilon = .001;
        expect(() => tf.batchNorm(x, mean, variance, epsilon, {}))
            .toThrowError(/Argument 'scale' passed to 'batchNorm' must be a Tensor/);
    });
    it('throws when passed offset as a non-tensor', () => {
        const x = tf.tensor4d([2, 4, 9, 23], [2, 1, 1, 2]);
        const mean = tf.tensor1d([1, 2]);
        const variance = tf.tensor1d([2, 3]);
        const epsilon = .001;
        const scale = tf.tensor1d([0.62186907, 0.85673736, 0.19201061]);
        const e = /Argument 'offset' passed to 'batchNorm' must be a Tensor/;
        expect(() => tf.batchNorm(x, mean, variance, {}, scale, epsilon))
            .toThrowError(e);
    });
    it('accepts a tensor-like object', async () => {
        const x = [[2, 4], [9, 23]];
        const mean = [1, 2];
        const variance = [2, 3];
        const offset = [3, 4];
        const scale = [4, 5];
        const varianceEpsilon = .001;
        const result = tf.batchNorm2d(x, mean, variance, offset, scale, varianceEpsilon);
        expectArraysClose(await result.data(), [
            offset[0] +
                (x[0][0] - mean[0]) * scale[0] /
                    Math.sqrt(variance[0] + varianceEpsilon),
            offset[1] +
                (x[0][1] - mean[1]) * scale[1] /
                    Math.sqrt(variance[1] + varianceEpsilon),
            offset[0] +
                (x[1][0] - mean[0]) * scale[0] /
                    Math.sqrt(variance[0] + varianceEpsilon),
            offset[1] +
                (x[1][1] - mean[1]) * scale[1] /
                    Math.sqrt(variance[1] + varianceEpsilon)
        ]);
    });
    it('throws error when x is a string tensor', () => {
        const mean = [1, 2];
        const variance = [2, 3];
        const offset = [3, 4];
        const scale = [4, 5];
        const varianceEpsilon = .001;
        const f = () => tf.batchNorm2d([['a', 'b'], ['c', 'd']], mean, variance, offset, scale, varianceEpsilon);
        expect(f).toThrowError(/Argument 'x' passed to 'batchNorm' must be numeric/);
    });
    it('throws error when mean is a string tensor', () => {
        const x = [[2, 4], [9, 23]];
        const variance = [2, 3];
        const offset = [3, 4];
        const scale = [4, 5];
        const varianceEpsilon = .001;
        const f = () => tf.batchNorm2d(x, ['a', 'b'], variance, offset, scale, varianceEpsilon);
        expect(f).toThrowError(/Argument 'mean' passed to 'batchNorm' must be numeric/);
    });
    it('throws error when variance is a string tensor', () => {
        const x = [[2, 4], [9, 23]];
        const mean = [1, 2];
        const offset = [3, 4];
        const scale = [4, 5];
        const varianceEpsilon = .001;
        const f = () => tf.batchNorm2d(x, mean, ['a', 'b'], offset, scale, varianceEpsilon);
        expect(f).toThrowError(/'variance' passed to 'batchNorm' must be numeric/);
    });
    it('throws error when scale is a string tensor', () => {
        const x = [[2, 4], [9, 23]];
        const mean = [1, 2];
        const variance = [2, 3];
        const offset = [3, 4];
        const varianceEpsilon = .001;
        const f = () => tf.batchNorm2d(x, mean, variance, offset, ['a', 'b'], varianceEpsilon);
        expect(f).toThrowError(/'scale' passed to 'batchNorm' must be numeric/);
    });
    it('throws error when offset is a string tensor', () => {
        const x = [[2, 4], [9, 23]];
        const mean = [1, 2];
        const variance = [2, 3];
        const scale = [4, 5];
        const varianceEpsilon = .001;
        const f = () => tf.batchNorm2d(x, mean, variance, ['a', 'b'], scale, varianceEpsilon);
        expect(f).toThrowError(/'offset' passed to 'batchNorm' must be numeric/);
    });
});
//# sourceMappingURL=data:application/json;base64,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