/**
 * @license
 * Copyright 2019 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import { env } from '@tensorflow/tfjs-core';
import { FromPixels } from '@tensorflow/tfjs-core';
import { TextureUsage } from '../tex_util';
import { FromPixelsProgram } from './FromPixels_utils/from_pixels_gpu';
import { FromPixelsPackedProgram } from './FromPixels_utils/from_pixels_packed_gpu';
export const fromPixelsConfig = {
    kernelName: FromPixels,
    backendName: 'webgl',
    kernelFunc: fromPixels,
};
let fromPixels2DContext;
let willReadFrequently = env().getBool('CANVAS2D_WILL_READ_FREQUENTLY_FOR_GPU');
function fromPixels(args) {
    const { inputs, backend, attrs } = args;
    let { pixels } = inputs;
    const { numChannels } = attrs;
    const isVideo = typeof (HTMLVideoElement) !== 'undefined' &&
        pixels instanceof HTMLVideoElement;
    const isImage = typeof (HTMLImageElement) !== 'undefined' &&
        pixels instanceof HTMLImageElement;
    const [width, height] = isVideo ?
        [
            pixels.videoWidth,
            pixels.videoHeight
        ] :
        [pixels.width, pixels.height];
    const texShape = [height, width];
    const outShape = [height, width, numChannels];
    if (isImage || isVideo) {
        const newWillReadFrequently = env().getBool('CANVAS2D_WILL_READ_FREQUENTLY_FOR_GPU');
        if (fromPixels2DContext == null ||
            newWillReadFrequently !== willReadFrequently) {
            willReadFrequently = newWillReadFrequently;
            fromPixels2DContext =
                document.createElement('canvas').getContext('2d', { willReadFrequently });
        }
        fromPixels2DContext.canvas.width = width;
        fromPixels2DContext.canvas.height = height;
        fromPixels2DContext.drawImage(pixels, 0, 0, width, height);
        pixels = fromPixels2DContext.canvas;
    }
    const tempPixelHandle = backend.makeTensorInfo(texShape, 'int32');
    // This is a byte texture with pixels.
    backend.texData.get(tempPixelHandle.dataId).usage = TextureUsage.PIXELS;
    backend.gpgpu.uploadPixelDataToTexture(backend.getTexture(tempPixelHandle.dataId), pixels);
    const program = env().getBool('WEBGL_PACK') ?
        new FromPixelsPackedProgram(outShape) :
        new FromPixelsProgram(outShape);
    const res = backend.runWebGLProgram(program, [tempPixelHandle], 'int32');
    backend.disposeData(tempPixelHandle.dataId);
    return res;
}
//# sourceMappingURL=data:application/json;base64,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