/**
 * @license
 * Copyright 2023 Google LLC.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import { BitwiseAnd, env } from '@tensorflow/tfjs-core';
import { BinaryOpProgram } from '../binaryop_gpu';
import { BinaryOpPackedProgram } from '../binaryop_packed_gpu';
import { bitwiseAndImplCPU as cpuBitwiseAnd } from '../kernel_utils/shared';
export const BITWISEAND = `
  int r = int(a.r) & int(b.r);
  int g = int(a.g) & int(b.g);
  int rb = int(a.b) & int(b.b);
  int ra = int(a.a) & int(b.a);
  return vec4(r, g, rb, ra);
`;
export const BITWISEAND_UNPACKED = `
  return float(int(a.r) & int(b.r));
`;
export function bitwiseAnd(args) {
    const { inputs, backend } = args;
    const { a, b } = inputs;
    const shouldUsePackedProgram = env().getBool('WEBGL_PACK_BINARY_OPERATIONS');
    const versionNumber = env().getNumber('WEBGL_VERSION');
    // The type of a and b are ensured to be `int32` in core, therefore no need to
    // consider other type situations.
    if ((backend.shouldExecuteOnCPU([a, b])) || versionNumber === 1) {
        const aVals = backend.texData.get(a.dataId).values;
        const bVals = backend.texData.get(b.dataId).values;
        const [outValues, outShape] = cpuBitwiseAnd(a.shape, b.shape, aVals, bVals, a.dtype);
        const out = backend.makeTensorInfo(outShape, a.dtype);
        const outData = backend.texData.get(out.dataId);
        outData.values = outValues;
        return out;
    }
    let program;
    if (shouldUsePackedProgram) {
        program = new BinaryOpPackedProgram(BITWISEAND, a.shape, b.shape, false);
    }
    else {
        program = new BinaryOpProgram(BITWISEAND_UNPACKED, a.shape, b.shape);
    }
    return backend.runWebGLProgram(program, [a, b], a.dtype);
}
export const bitwiseAndConfig = {
    kernelName: BitwiseAnd,
    backendName: 'webgl',
    kernelFunc: bitwiseAnd
};
//# sourceMappingURL=data:application/json;base64,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