/**
 * @license
 * Copyright 2020 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import { backend_util, env, upcastType } from '@tensorflow/tfjs-core';
import { BinaryOpProgram } from '../binaryop_gpu';
import { BinaryOpPackedProgram } from '../binaryop_packed_gpu';
import { complex } from '../kernels/Complex';
import { LEAKYRELU, LEAKYRELU_PACKED } from '../kernels/LeakyRelu';
import { PRELU, PRELU_PACKED } from '../kernels/Prelu';
import * as unary_op from '../unaryop_gpu';
import { UnaryOpProgram } from '../unaryop_gpu';
import * as unary_packed_op from '../unaryop_packed_gpu';
import { UnaryOpPackedProgram } from '../unaryop_packed_gpu';
export const CHECK_NAN_SNIPPET_UNARY = `if (isnan(x)) return x;`;
/**
 * Template that creates a `KernelFunc` for unary ops.
 * @param opSnippet Op snippet to create `UnaryOpProgram`.
 * @param packedOpSnippet Op snippet to create `UnaryOpPackedProgram`.
 * @param dtype Optional. If set, the result has this dtype. Otherwise, the
 *     result has the same dtype as the first input. This is mainly used in
 *     comparison kernels, such as Equal, Less, Greater, etc.
 */
export function unaryKernelFunc({ opSnippet, packedOpSnippet, cpuKernelImpl, dtype }) {
    return ({ inputs, backend }) => {
        const { x } = inputs;
        const webglBackend = backend;
        const $dtype = dtype || x.dtype;
        if (webglBackend.shouldExecuteOnCPU([x]) && cpuKernelImpl != null) {
            const xData = webglBackend.texData.get(x.dataId);
            const outValues = cpuKernelImpl(xData.values, $dtype);
            return webglBackend.makeTensorInfo(x.shape, $dtype, outValues);
        }
        const shouldUsePackedProgram = env().getBool('WEBGL_PACK_UNARY_OPERATIONS') && packedOpSnippet != null;
        let program;
        if (shouldUsePackedProgram) {
            program = new UnaryOpPackedProgram(x.shape, packedOpSnippet);
        }
        else {
            program = new UnaryOpProgram(x.shape, opSnippet);
        }
        return webglBackend.runWebGLProgram(program, [x], $dtype);
    };
}
/**
 * Template that creates a `KernelFunc` for binary ops.
 * @param opSnippet Op snippet to create `BinaryOpProgram`.
 * @param packedOpSnippet Op snippet to create `BinaryOpPackedProgram`.
 * @param checkOutOfBoundsForPackedProgram Whether to set checkOutOfBounds=true
 *     when creating BinaryOpPackedProgram.
 * @param dtype Optional. If set, the result has this dtype. Otherwise, the
 *     result has the same dtype as the first input. This is mainly used in
 *     comparison kernels, such as Equal, Less, Greater, etc.
 */
export function binaryKernelFunc({ opSnippet, packedOpSnippet, checkOutOfBounds = false, supportsComplex = false, cpuKernelImpl, dtype }) {
    return ({ inputs, backend }) => {
        const { a, b } = inputs;
        const webglBackend = backend;
        if (supportsComplex && a.dtype === 'complex64') {
            const aData = webglBackend.texData.get(a.dataId);
            const bData = webglBackend.texData.get(b.dataId);
            const [real, imag] = [
                [aData.complexTensorInfos.real, bData.complexTensorInfos.real],
                [aData.complexTensorInfos.imag, bData.complexTensorInfos.imag]
            ].map(complexParts => {
                const [aPart, bPart] = complexParts;
                const aHandle = {
                    dataId: aPart.dataId,
                    dtype: aPart.dtype,
                    shape: a.shape
                };
                const bHandle = {
                    dataId: bPart.dataId,
                    dtype: bPart.dtype,
                    shape: b.shape
                };
                const program = new BinaryOpProgram(opSnippet, a.shape, b.shape);
                return webglBackend.runWebGLProgram(program, [aHandle, bHandle], upcastType(aPart.dtype, bPart.dtype));
            });
            const complexOutput = complex({ inputs: { real, imag }, backend: webglBackend });
            webglBackend.disposeIntermediateTensorInfo(real);
            webglBackend.disposeIntermediateTensorInfo(imag);
            // TODO(annxingyuan): Implement CPU forwarding for complex inputs.
            return complexOutput;
        }
        const $dtype = dtype || upcastType(a.dtype, b.dtype);
        if ((a.dtype === 'string' || b.dtype === 'string' ||
            webglBackend.shouldExecuteOnCPU([a, b])) &&
            cpuKernelImpl != null) {
            const aVals = webglBackend.texData.get(a.dataId).values;
            const bVals = webglBackend.texData.get(b.dataId).values;
            const decodedAVals = a.dtype === 'string' ?
                // tslint:disable-next-line: no-any
                backend_util.fromUint8ToStringArray(aVals) :
                aVals;
            const decodedBVals = a.dtype === 'string' ?
                // tslint:disable-next-line: no-any
                backend_util.fromUint8ToStringArray(bVals) :
                bVals;
            const [outValues, outShape] = cpuKernelImpl(a.shape, b.shape, decodedAVals, decodedBVals, $dtype);
            const out = webglBackend.makeTensorInfo(outShape, $dtype);
            const outData = webglBackend.texData.get(out.dataId);
            outData.values = outValues;
            return out;
        }
        const shouldUsePackedProgram = env().getBool('WEBGL_PACK_BINARY_OPERATIONS') &&
            packedOpSnippet != null;
        let program;
        if (shouldUsePackedProgram) {
            program = new BinaryOpPackedProgram(packedOpSnippet, a.shape, b.shape, checkOutOfBounds);
        }
        else {
            program = new BinaryOpProgram(opSnippet, a.shape, b.shape);
        }
        return webglBackend.runWebGLProgram(program, [a, b], $dtype);
    };
}
export function mapActivationToShaderProgram(activation, packed = false) {
    if (activation === 'linear') {
        if (packed) {
            return unary_packed_op.LINEAR;
        }
        return unary_op.LINEAR;
    }
    else if (activation === 'relu') {
        if (packed) {
            return unary_packed_op.RELU;
        }
        return unary_op.RELU;
    }
    else if (activation === 'elu') {
        if (packed) {
            return unary_packed_op.ELU;
        }
        return unary_op.ELU;
    }
    else if (activation === 'relu6') {
        if (packed) {
            return unary_packed_op.RELU6;
        }
        return unary_op.RELU6;
    }
    else if (activation === 'prelu') {
        if (packed) {
            return PRELU_PACKED;
        }
        return PRELU;
    }
    else if (activation === 'leakyrelu') {
        if (packed) {
            return LEAKYRELU_PACKED;
        }
        return LEAKYRELU;
    }
    else if (activation === 'sigmoid') {
        if (packed) {
            return unary_packed_op.SIGMOID;
        }
        return unary_op.SIGMOID;
    }
    throw new Error(`Activation ${activation} has not been implemented for the WebGL backend.`);
}
//# sourceMappingURL=data:application/json;base64,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