/**
 * @license
 * Copyright 2023 Google LLC.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import { Draw } from '@tensorflow/tfjs-core';
export function draw(args) {
    const { inputs, backend, attrs } = args;
    const { image } = inputs;
    const { canvas, options } = attrs;
    const { contextOptions, imageOptions } = options || {};
    const alpha = (imageOptions === null || imageOptions === void 0 ? void 0 : imageOptions.alpha) || 1;
    const contextType = (contextOptions === null || contextOptions === void 0 ? void 0 : contextOptions.contextType) || '2d';
    if (contextType !== '2d') {
        throw new Error(`Context type ${contextOptions.contextType} is not supported by the CPU backend.`);
    }
    const ctx = canvas.getContext(contextType, (contextOptions === null || contextOptions === void 0 ? void 0 : contextOptions.contextAttributes) || {});
    if (ctx == null) {
        throw new Error(`Could not get the context with ${contextType} type.`);
    }
    const [height, width] = image.shape.slice(0, 2);
    const depth = image.shape.length === 2 ? 1 : image.shape[2];
    const data = backend.data.get(image.dataId).values;
    const multiplier = image.dtype === 'float32' ? 255 : 1;
    const bytes = new Uint8ClampedArray(width * height * 4);
    for (let i = 0; i < height * width; ++i) {
        const rgba = [0, 0, 0, 255 * alpha];
        for (let d = 0; d < depth; d++) {
            const value = data[i * depth + d];
            if (image.dtype === 'float32') {
                if (value < 0 || value > 1) {
                    throw new Error(`Tensor values for a float32 Tensor must be in the ` +
                        `range [0 - 1] but encountered ${value}.`);
                }
            }
            else if (image.dtype === 'int32') {
                if (value < 0 || value > 255) {
                    throw new Error(`Tensor values for a int32 Tensor must be in the ` +
                        `range [0 - 255] but encountered ${value}.`);
                }
            }
            if (depth === 1) {
                rgba[0] = value * multiplier;
                rgba[1] = value * multiplier;
                rgba[2] = value * multiplier;
            }
            else {
                rgba[d] = value * multiplier;
            }
        }
        const j = i * 4;
        bytes[j + 0] = Math.round(rgba[0]);
        bytes[j + 1] = Math.round(rgba[1]);
        bytes[j + 2] = Math.round(rgba[2]);
        bytes[j + 3] = Math.round(rgba[3]);
    }
    canvas.width = width;
    canvas.height = height;
    const imageData = new ImageData(bytes, width, height);
    ctx.putImageData(imageData, 0, 0);
    return image;
}
export const drawConfig = {
    kernelName: Draw,
    backendName: 'cpu',
    kernelFunc: draw
};
//# sourceMappingURL=data:application/json;base64,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