/**
 * @license
 * Copyright 2020 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import { backend_util, Dilation2D, util } from '@tensorflow/tfjs-core';
export const dilation2DConfig = {
    kernelName: Dilation2D,
    backendName: 'cpu',
    kernelFunc: ({ inputs, backend, attrs }) => {
        const { x, filter } = inputs;
        const { strides, pad, dilations } = attrs;
        const cpuBackend = backend;
        const xVals = cpuBackend.data.get(x.dataId).values;
        const xRank = x.shape.length;
        const filterVals = cpuBackend.data.get(filter.dataId).values;
        const filterRank = filter.shape.length;
        const { batchSize, inHeight, inWidth, inChannels, outHeight, outWidth, padInfo, strideHeight, strideWidth, filterHeight, filterWidth, dilationHeight, dilationWidth, outShape } = backend_util.computeDilation2DInfo(x.shape, filter.shape, strides, pad, 'NHWC' /* dataFormat */, dilations);
        const outSize = util.sizeFromShape(outShape);
        const outRank = outShape.length;
        const outputVals = util.getArrayFromDType(x.dtype, outSize);
        // Upsampling the input by fill in `dilation size - 1` values between each
        // input value.
        // This implementation follows the TF c++ implementation:
        // https://github.com/tensorflow/tensorflow/blob/d9a3a849edc198e90172bc58eb293de457f9d986/tensorflow/core/kernels/dilation_ops.cc
        for (let b = 0; b < batchSize; ++b) {
            for (let hOut = 0; hOut < outHeight; ++hOut) {
                const hBeg = hOut * strideHeight - padInfo.top;
                for (let wOut = 0; wOut < outWidth; ++wOut) {
                    const wBeg = wOut * strideWidth - padInfo.left;
                    for (let d = 0; d < inChannels; ++d) {
                        let curVal = Number.MIN_SAFE_INTEGER;
                        for (let h = 0; h < filterHeight; ++h) {
                            const hIn = hBeg + h * dilationHeight;
                            if (hIn >= 0 && hIn < inHeight) {
                                for (let w = 0; w < filterWidth; ++w) {
                                    const wIn = wBeg + w * dilationWidth;
                                    if (wIn >= 0 && wIn < inWidth) {
                                        const xIndex = util.locToIndex([b, hIn, wIn, d], xRank, util.computeStrides(x.shape));
                                        const filterIndex = util.locToIndex([h, w, d], filterRank, util.computeStrides(filter.shape));
                                        const val = xVals[xIndex] + filterVals[filterIndex];
                                        if (val > curVal) {
                                            curVal = val;
                                        }
                                    }
                                }
                            }
                        }
                        const outputIndex = util.locToIndex([b, hOut, wOut, d], outRank, util.computeStrides(outShape));
                        outputVals[outputIndex] = curVal;
                    }
                }
            }
        }
        const dataId = cpuBackend.write(util.toTypedArray(outputVals, x.dtype), outShape, x.dtype);
        return { dataId, shape: outShape, dtype: x.dtype };
    }
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiRGlsYXRpb24yRC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uLy4uLy4uLy4uL3RmanMtYmFja2VuZC1jcHUvc3JjL2tlcm5lbHMvRGlsYXRpb24yRC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBQTs7Ozs7Ozs7Ozs7Ozs7O0dBZUc7QUFFSCxPQUFPLEVBQUMsWUFBWSxFQUFFLFVBQVUsRUFBK0QsSUFBSSxFQUFDLE1BQU0sdUJBQXVCLENBQUM7QUFJbEksTUFBTSxDQUFDLE1BQU0sZ0JBQWdCLEdBQWlCO0lBQzVDLFVBQVUsRUFBRSxVQUFVO0lBQ3RCLFdBQVcsRUFBRSxLQUFLO0lBQ2xCLFVBQVUsRUFBRSxDQUFDLEVBQUMsTUFBTSxFQUFFLE9BQU8sRUFBRSxLQUFLLEVBQUMsRUFBRSxFQUFFO1FBQ3ZDLE1BQU0sRUFBQyxDQUFDLEVBQUUsTUFBTSxFQUFDLEdBQUcsTUFBMEIsQ0FBQztRQUMvQyxNQUFNLEVBQUMsT0FBTyxFQUFFLEdBQUcsRUFBRSxTQUFTLEVBQUMsR0FBRyxLQUFtQyxDQUFDO1FBQ3RFLE1BQU0sVUFBVSxHQUFHLE9BQXlCLENBQUM7UUFFN0MsTUFBTSxLQUFLLEdBQUcsVUFBVSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxDQUFDLE1BQW9CLENBQUM7UUFDakUsTUFBTSxLQUFLLEdBQUcsQ0FBQyxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUM7UUFFN0IsTUFBTSxVQUFVLEdBQUcsVUFBVSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxDQUFDLE1BQW9CLENBQUM7UUFDM0UsTUFBTSxVQUFVLEdBQUcsTUFBTSxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUM7UUFFdkMsTUFBTSxFQUNKLFNBQVMsRUFDVCxRQUFRLEVBQ1IsT0FBTyxFQUNQLFVBQVUsRUFDVixTQUFTLEVBQ1QsUUFBUSxFQUNSLE9BQU8sRUFDUCxZQUFZLEVBQ1osV0FBVyxFQUNYLFlBQVksRUFDWixXQUFXLEVBQ1gsY0FBYyxFQUNkLGFBQWEsRUFDYixRQUFRLEVBQ1QsR0FDRyxZQUFZLENBQUMscUJBQXFCLENBQzlCLENBQUMsQ0FBQyxLQUF5QyxFQUMzQyxNQUFNLENBQUMsS0FBaUMsRUFBRSxPQUFPLEVBQUUsR0FBRyxFQUN0RCxNQUFNLENBQUMsZ0JBQWdCLEVBQUUsU0FBUyxDQUFDLENBQUM7UUFFNUMsTUFBTSxPQUFPLEdBQUcsSUFBSSxDQUFDLGFBQWEsQ0FBQyxRQUFRLENBQUMsQ0FBQztRQUM3QyxNQUFNLE9BQU8sR0FBRyxRQUFRLENBQUMsTUFBTSxDQUFDO1FBQ2hDLE1BQU0sVUFBVSxHQUFHLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDLENBQUMsS0FBSyxFQUFFLE9BQU8sQ0FBQyxDQUFDO1FBRTVELDBFQUEwRTtRQUMxRSxlQUFlO1FBQ2YseURBQXlEO1FBQ3pELGlJQUFpSTtRQUNqSSxLQUFLLElBQUksQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLEdBQUcsU0FBUyxFQUFFLEVBQUUsQ0FBQyxFQUFFO1lBQ2xDLEtBQUssSUFBSSxJQUFJLEdBQUcsQ0FBQyxFQUFFLElBQUksR0FBRyxTQUFTLEVBQUUsRUFBRSxJQUFJLEVBQUU7Z0JBQzNDLE1BQU0sSUFBSSxHQUFHLElBQUksR0FBRyxZQUFZLEdBQUcsT0FBTyxDQUFDLEdBQUcsQ0FBQztnQkFDL0MsS0FBSyxJQUFJLElBQUksR0FBRyxDQUFDLEVBQUUsSUFBSSxHQUFHLFFBQVEsRUFBRSxFQUFFLElBQUksRUFBRTtvQkFDMUMsTUFBTSxJQUFJLEdBQUcsSUFBSSxHQUFHLFdBQVcsR0FBRyxPQUFPLENBQUMsSUFBSSxDQUFDO29CQUMvQyxLQUFLLElBQUksQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLEdBQUcsVUFBVSxFQUFFLEVBQUUsQ0FBQyxFQUFFO3dCQUNuQyxJQUFJLE1BQU0sR0FBRyxNQUFNLENBQUMsZ0JBQWdCLENBQUM7d0JBQ3JDLEtBQUssSUFBSSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsR0FBRyxZQUFZLEVBQUUsRUFBRSxDQUFDLEVBQUU7NEJBQ3JDLE1BQU0sR0FBRyxHQUFHLElBQUksR0FBRyxDQUFDLEdBQUcsY0FBYyxDQUFDOzRCQUN0QyxJQUFJLEdBQUcsSUFBSSxDQUFDLElBQUksR0FBRyxHQUFHLFFBQVEsRUFBRTtnQ0FDOUIsS0FBSyxJQUFJLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxHQUFHLFdBQVcsRUFBRSxFQUFFLENBQUMsRUFBRTtvQ0FDcEMsTUFBTSxHQUFHLEdBQUcsSUFBSSxHQUFHLENBQUMsR0FBRyxhQUFhLENBQUM7b0NBQ3JDLElBQUksR0FBRyxJQUFJLENBQUMsSUFBSSxHQUFHLEdBQUcsT0FBTyxFQUFFO3dDQUM3QixNQUFNLE1BQU0sR0FBRyxJQUFJLENBQUMsVUFBVSxDQUMxQixDQUFDLENBQUMsRUFBRSxHQUFHLEVBQUUsR0FBRyxFQUFFLENBQUMsQ0FBQyxFQUFFLEtBQUssRUFBRSxJQUFJLENBQUMsY0FBYyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDO3dDQUMzRCxNQUFNLFdBQVcsR0FBRyxJQUFJLENBQUMsVUFBVSxDQUMvQixDQUFDLENBQUMsRUFBRSxDQUFDLEVBQUUsQ0FBQyxDQUFDLEVBQUUsVUFBVSxFQUNyQixJQUFJLENBQUMsY0FBYyxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDO3dDQUN2QyxNQUFNLEdBQUcsR0FBRyxLQUFLLENBQUMsTUFBTSxDQUFDLEdBQUcsVUFBVSxDQUFDLFdBQVcsQ0FBQyxDQUFDO3dDQUNwRCxJQUFJLEdBQUcsR0FBRyxNQUFNLEVBQUU7NENBQ2hCLE1BQU0sR0FBRyxHQUFHLENBQUM7eUNBQ2Q7cUNBQ0Y7aUNBQ0Y7NkJBQ0Y7eUJBQ0Y7d0JBQ0QsTUFBTSxXQUFXLEdBQUcsSUFBSSxDQUFDLFVBQVUsQ0FDL0IsQ0FBQyxDQUFDLEVBQUUsSUFBSSxFQUFFLElBQUksRUFBRSxDQUFDLENBQUMsRUFBRSxPQUFPLEVBQUUsSUFBSSxDQUFDLGNBQWMsQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDO3dCQUNoRSxVQUFVLENBQUMsV0FBVyxDQUFDLEdBQUcsTUFBTSxDQUFDO3FCQUNsQztpQkFDRjthQUNGO1NBQ0Y7UUFFRCxNQUFNLE1BQU0sR0FBRyxVQUFVLENBQUMsS0FBSyxDQUMzQixJQUFJLENBQUMsWUFBWSxDQUFDLFVBQVUsRUFBRSxDQUFDLENBQUMsS0FBSyxDQUFDLEVBQUUsUUFBUSxFQUFFLENBQUMsQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUUvRCxPQUFPLEVBQUMsTUFBTSxFQUFFLEtBQUssRUFBRSxRQUFRLEVBQUUsS0FBSyxFQUFFLENBQUMsQ0FBQyxLQUFLLEVBQUMsQ0FBQztJQUNuRCxDQUFDO0NBQ0YsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbIi8qKlxuICogQGxpY2Vuc2VcbiAqIENvcHlyaWdodCAyMDIwIEdvb2dsZSBMTEMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG4gKiBMaWNlbnNlZCB1bmRlciB0aGUgQXBhY2hlIExpY2Vuc2UsIFZlcnNpb24gMi4wICh0aGUgXCJMaWNlbnNlXCIpO1xuICogeW91IG1heSBub3QgdXNlIHRoaXMgZmlsZSBleGNlcHQgaW4gY29tcGxpYW5jZSB3aXRoIHRoZSBMaWNlbnNlLlxuICogWW91IG1heSBvYnRhaW4gYSBjb3B5IG9mIHRoZSBMaWNlbnNlIGF0XG4gKlxuICogaHR0cDovL3d3dy5hcGFjaGUub3JnL2xpY2Vuc2VzL0xJQ0VOU0UtMi4wXG4gKlxuICogVW5sZXNzIHJlcXVpcmVkIGJ5IGFwcGxpY2FibGUgbGF3IG9yIGFncmVlZCB0byBpbiB3cml0aW5nLCBzb2Z0d2FyZVxuICogZGlzdHJpYnV0ZWQgdW5kZXIgdGhlIExpY2Vuc2UgaXMgZGlzdHJpYnV0ZWQgb24gYW4gXCJBUyBJU1wiIEJBU0lTLFxuICogV0lUSE9VVCBXQVJSQU5USUVTIE9SIENPTkRJVElPTlMgT0YgQU5ZIEtJTkQsIGVpdGhlciBleHByZXNzIG9yIGltcGxpZWQuXG4gKiBTZWUgdGhlIExpY2Vuc2UgZm9yIHRoZSBzcGVjaWZpYyBsYW5ndWFnZSBnb3Zlcm5pbmcgcGVybWlzc2lvbnMgYW5kXG4gKiBsaW1pdGF0aW9ucyB1bmRlciB0aGUgTGljZW5zZS5cbiAqID09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09XG4gKi9cblxuaW1wb3J0IHtiYWNrZW5kX3V0aWwsIERpbGF0aW9uMkQsIERpbGF0aW9uMkRBdHRycywgRGlsYXRpb24yRElucHV0cywgS2VybmVsQ29uZmlnLCBUeXBlZEFycmF5LCB1dGlsfSBmcm9tICdAdGVuc29yZmxvdy90ZmpzLWNvcmUnO1xuXG5pbXBvcnQge01hdGhCYWNrZW5kQ1BVfSBmcm9tICcuLi9iYWNrZW5kX2NwdSc7XG5cbmV4cG9ydCBjb25zdCBkaWxhdGlvbjJEQ29uZmlnOiBLZXJuZWxDb25maWcgPSB7XG4gIGtlcm5lbE5hbWU6IERpbGF0aW9uMkQsXG4gIGJhY2tlbmROYW1lOiAnY3B1JyxcbiAga2VybmVsRnVuYzogKHtpbnB1dHMsIGJhY2tlbmQsIGF0dHJzfSkgPT4ge1xuICAgIGNvbnN0IHt4LCBmaWx0ZXJ9ID0gaW5wdXRzIGFzIERpbGF0aW9uMkRJbnB1dHM7XG4gICAgY29uc3Qge3N0cmlkZXMsIHBhZCwgZGlsYXRpb25zfSA9IGF0dHJzIGFzIHVua25vd24gYXMgRGlsYXRpb24yREF0dHJzO1xuICAgIGNvbnN0IGNwdUJhY2tlbmQgPSBiYWNrZW5kIGFzIE1hdGhCYWNrZW5kQ1BVO1xuXG4gICAgY29uc3QgeFZhbHMgPSBjcHVCYWNrZW5kLmRhdGEuZ2V0KHguZGF0YUlkKS52YWx1ZXMgYXMgVHlwZWRBcnJheTtcbiAgICBjb25zdCB4UmFuayA9IHguc2hhcGUubGVuZ3RoO1xuXG4gICAgY29uc3QgZmlsdGVyVmFscyA9IGNwdUJhY2tlbmQuZGF0YS5nZXQoZmlsdGVyLmRhdGFJZCkudmFsdWVzIGFzIFR5cGVkQXJyYXk7XG4gICAgY29uc3QgZmlsdGVyUmFuayA9IGZpbHRlci5zaGFwZS5sZW5ndGg7XG5cbiAgICBjb25zdCB7XG4gICAgICBiYXRjaFNpemUsXG4gICAgICBpbkhlaWdodCxcbiAgICAgIGluV2lkdGgsXG4gICAgICBpbkNoYW5uZWxzLFxuICAgICAgb3V0SGVpZ2h0LFxuICAgICAgb3V0V2lkdGgsXG4gICAgICBwYWRJbmZvLFxuICAgICAgc3RyaWRlSGVpZ2h0LFxuICAgICAgc3RyaWRlV2lkdGgsXG4gICAgICBmaWx0ZXJIZWlnaHQsXG4gICAgICBmaWx0ZXJXaWR0aCxcbiAgICAgIGRpbGF0aW9uSGVpZ2h0LFxuICAgICAgZGlsYXRpb25XaWR0aCxcbiAgICAgIG91dFNoYXBlXG4gICAgfSA9XG4gICAgICAgIGJhY2tlbmRfdXRpbC5jb21wdXRlRGlsYXRpb24yREluZm8oXG4gICAgICAgICAgICB4LnNoYXBlIGFzIFtudW1iZXIsIG51bWJlciwgbnVtYmVyLCBudW1iZXJdLFxuICAgICAgICAgICAgZmlsdGVyLnNoYXBlIGFzIFtudW1iZXIsIG51bWJlciwgbnVtYmVyXSwgc3RyaWRlcywgcGFkLFxuICAgICAgICAgICAgJ05IV0MnIC8qIGRhdGFGb3JtYXQgKi8sIGRpbGF0aW9ucyk7XG5cbiAgICBjb25zdCBvdXRTaXplID0gdXRpbC5zaXplRnJvbVNoYXBlKG91dFNoYXBlKTtcbiAgICBjb25zdCBvdXRSYW5rID0gb3V0U2hhcGUubGVuZ3RoO1xuICAgIGNvbnN0IG91dHB1dFZhbHMgPSB1dGlsLmdldEFycmF5RnJvbURUeXBlKHguZHR5cGUsIG91dFNpemUpO1xuXG4gICAgLy8gVXBzYW1wbGluZyB0aGUgaW5wdXQgYnkgZmlsbCBpbiBgZGlsYXRpb24gc2l6ZSAtIDFgIHZhbHVlcyBiZXR3ZWVuIGVhY2hcbiAgICAvLyBpbnB1dCB2YWx1ZS5cbiAgICAvLyBUaGlzIGltcGxlbWVudGF0aW9uIGZvbGxvd3MgdGhlIFRGIGMrKyBpbXBsZW1lbnRhdGlvbjpcbiAgICAvLyBodHRwczovL2dpdGh1Yi5jb20vdGVuc29yZmxvdy90ZW5zb3JmbG93L2Jsb2IvZDlhM2E4NDllZGMxOThlOTAxNzJiYzU4ZWIyOTNkZTQ1N2Y5ZDk4Ni90ZW5zb3JmbG93L2NvcmUva2VybmVscy9kaWxhdGlvbl9vcHMuY2NcbiAgICBmb3IgKGxldCBiID0gMDsgYiA8IGJhdGNoU2l6ZTsgKytiKSB7XG4gICAgICBmb3IgKGxldCBoT3V0ID0gMDsgaE91dCA8IG91dEhlaWdodDsgKytoT3V0KSB7XG4gICAgICAgIGNvbnN0IGhCZWcgPSBoT3V0ICogc3RyaWRlSGVpZ2h0IC0gcGFkSW5mby50b3A7XG4gICAgICAgIGZvciAobGV0IHdPdXQgPSAwOyB3T3V0IDwgb3V0V2lkdGg7ICsrd091dCkge1xuICAgICAgICAgIGNvbnN0IHdCZWcgPSB3T3V0ICogc3RyaWRlV2lkdGggLSBwYWRJbmZvLmxlZnQ7XG4gICAgICAgICAgZm9yIChsZXQgZCA9IDA7IGQgPCBpbkNoYW5uZWxzOyArK2QpIHtcbiAgICAgICAgICAgIGxldCBjdXJWYWwgPSBOdW1iZXIuTUlOX1NBRkVfSU5URUdFUjtcbiAgICAgICAgICAgIGZvciAobGV0IGggPSAwOyBoIDwgZmlsdGVySGVpZ2h0OyArK2gpIHtcbiAgICAgICAgICAgICAgY29uc3QgaEluID0gaEJlZyArIGggKiBkaWxhdGlvbkhlaWdodDtcbiAgICAgICAgICAgICAgaWYgKGhJbiA+PSAwICYmIGhJbiA8IGluSGVpZ2h0KSB7XG4gICAgICAgICAgICAgICAgZm9yIChsZXQgdyA9IDA7IHcgPCBmaWx0ZXJXaWR0aDsgKyt3KSB7XG4gICAgICAgICAgICAgICAgICBjb25zdCB3SW4gPSB3QmVnICsgdyAqIGRpbGF0aW9uV2lkdGg7XG4gICAgICAgICAgICAgICAgICBpZiAod0luID49IDAgJiYgd0luIDwgaW5XaWR0aCkge1xuICAgICAgICAgICAgICAgICAgICBjb25zdCB4SW5kZXggPSB1dGlsLmxvY1RvSW5kZXgoXG4gICAgICAgICAgICAgICAgICAgICAgICBbYiwgaEluLCB3SW4sIGRdLCB4UmFuaywgdXRpbC5jb21wdXRlU3RyaWRlcyh4LnNoYXBlKSk7XG4gICAgICAgICAgICAgICAgICAgIGNvbnN0IGZpbHRlckluZGV4ID0gdXRpbC5sb2NUb0luZGV4KFxuICAgICAgICAgICAgICAgICAgICAgICAgW2gsIHcsIGRdLCBmaWx0ZXJSYW5rLFxuICAgICAgICAgICAgICAgICAgICAgICAgdXRpbC5jb21wdXRlU3RyaWRlcyhmaWx0ZXIuc2hhcGUpKTtcbiAgICAgICAgICAgICAgICAgICAgY29uc3QgdmFsID0geFZhbHNbeEluZGV4XSArIGZpbHRlclZhbHNbZmlsdGVySW5kZXhdO1xuICAgICAgICAgICAgICAgICAgICBpZiAodmFsID4gY3VyVmFsKSB7XG4gICAgICAgICAgICAgICAgICAgICAgY3VyVmFsID0gdmFsO1xuICAgICAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgICB9XG4gICAgICAgICAgICB9XG4gICAgICAgICAgICBjb25zdCBvdXRwdXRJbmRleCA9IHV0aWwubG9jVG9JbmRleChcbiAgICAgICAgICAgICAgICBbYiwgaE91dCwgd091dCwgZF0sIG91dFJhbmssIHV0aWwuY29tcHV0ZVN0cmlkZXMob3V0U2hhcGUpKTtcbiAgICAgICAgICAgIG91dHB1dFZhbHNbb3V0cHV0SW5kZXhdID0gY3VyVmFsO1xuICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgICAgfVxuICAgIH1cblxuICAgIGNvbnN0IGRhdGFJZCA9IGNwdUJhY2tlbmQud3JpdGUoXG4gICAgICAgIHV0aWwudG9UeXBlZEFycmF5KG91dHB1dFZhbHMsIHguZHR5cGUpLCBvdXRTaGFwZSwgeC5kdHlwZSk7XG5cbiAgICByZXR1cm4ge2RhdGFJZCwgc2hhcGU6IG91dFNoYXBlLCBkdHlwZTogeC5kdHlwZX07XG4gIH1cbn07XG4iXX0=