/**
 * @license
 * Copyright 2020 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import { BatchMatMul, broadcast_util, buffer, util } from '@tensorflow/tfjs-core';
import { assertNotComplex } from '../cpu_util';
import { reshape } from './Reshape';
export function batchMatMul(args) {
    const { inputs, backend, attrs } = args;
    const { a, b } = inputs;
    const { transposeA, transposeB } = attrs;
    assertNotComplex([a, b], 'matMul');
    const aRank = a.shape.length;
    const bRank = b.shape.length;
    const innerShapeA = transposeA ? a.shape[aRank - 2] : a.shape[aRank - 1];
    const innerShapeB = transposeB ? b.shape[bRank - 1] : b.shape[bRank - 2];
    const outerShapeA = transposeA ? a.shape[aRank - 1] : a.shape[aRank - 2];
    const outerShapeB = transposeB ? b.shape[bRank - 2] : b.shape[bRank - 1];
    const outerDimsA = a.shape.slice(0, -2);
    const outerDimsB = b.shape.slice(0, -2);
    const batchDimA = util.sizeFromShape(outerDimsA);
    const batchDimB = util.sizeFromShape(outerDimsB);
    const outShapeOuterDims = broadcast_util.assertAndGetBroadcastShape(a.shape.slice(0, -2), b.shape.slice(0, -2));
    const outShape = outShapeOuterDims.concat([outerShapeA, outerShapeB]);
    util.assert(innerShapeA === innerShapeB, () => `Error in matMul: inner shapes (${innerShapeA}) and (` +
        `${innerShapeB}) of Tensors with shapes ${a.shape} and ` +
        `${b.shape} and transposeA=${transposeA}` +
        ` and transposeB=${transposeB} must match.`);
    const a3dShape = transposeA ? [batchDimA, innerShapeA, outerShapeA] :
        [batchDimA, outerShapeA, innerShapeA];
    const b3dShape = transposeB ? [batchDimB, outerShapeB, innerShapeB] :
        [batchDimB, innerShapeB, outerShapeB];
    // The rest of the implementation is designed to operate on rank-3 tensors
    const a3d = reshape({ inputs: { x: a }, backend, attrs: { shape: a3dShape } });
    const b3d = reshape({ inputs: { x: b }, backend, attrs: { shape: b3dShape } });
    const sharedDim = transposeA ? a3d.shape[1] : a3d.shape[2];
    const leftDim = transposeA ? a3d.shape[2] : a3d.shape[1];
    const rightDim = transposeB ? b3d.shape[1] : b3d.shape[2];
    const batchDim = Math.max(batchDimA, batchDimB);
    const a3dValues = backend.data.get(a3d.dataId).values;
    const b3dValues = backend.data.get(b3d.dataId).values;
    const a3dStrides = util.computeStrides(a3d.shape);
    const b3dStrides = util.computeStrides(b3d.shape);
    const [aBatch, aOuterStep, aInnerStep] = transposeA ?
        [a3dStrides[0], 1, a3dStrides[1]] :
        [a3dStrides[0], a3dStrides[1], 1];
    const [bInnerStep, bOuterStep, bBatch] = transposeB ?
        [1, b3dStrides[1], b3dStrides[0]] :
        [b3dStrides[1], 1, b3dStrides[0]];
    const size = leftDim * rightDim;
    const result = buffer([batchDim, leftDim, rightDim], a3d.dtype);
    const resVals = result.values;
    const blockSize = backend.blockSize;
    for (let bi = 0; bi < batchDim; bi++) {
        const batchIndexA = bi % batchDimA;
        const batchIndexB = bi % batchDimB;
        for (let i0 = 0; i0 < leftDim; i0 += blockSize) {
            // for when blockSize doesn't evenly divide the input
            const iBlock = Math.min(i0 + blockSize, leftDim);
            for (let j0 = 0; j0 < rightDim; j0 += blockSize) {
                const jBlock = Math.min(j0 + blockSize, rightDim);
                for (let k0 = 0; k0 < sharedDim; k0 += blockSize) {
                    const kBlock = Math.min(k0 + blockSize, sharedDim);
                    for (let i = i0; i < iBlock; i++) {
                        for (let j = j0; j < jBlock; j++) {
                            let sum = 0.0;
                            for (let k = k0; k < kBlock; k++) {
                                const aVal = 
                                // tslint:disable-next-line: max-line-length
                                a3dValues[batchIndexA * aBatch + i * aOuterStep + k * aInnerStep];
                                const bVal = 
                                // tslint:disable-next-line: max-line-length
                                b3dValues[k * bInnerStep + j * bOuterStep + batchIndexB * bBatch];
                                sum += aVal * bVal;
                            }
                            resVals[bi * size + (i * rightDim + j)] += sum;
                        }
                    }
                }
            }
        }
    }
    backend.disposeIntermediateTensorInfo(a3d);
    backend.disposeIntermediateTensorInfo(b3d);
    // set correct shape on output.
    return backend.makeTensorInfo(outShape, result.dtype, result.values);
}
export const batchMatMulConfig = {
    kernelName: BatchMatMul,
    backendName: 'cpu',
    kernelFunc: batchMatMul,
};
//# sourceMappingURL=data:application/json;base64,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