export type RunResult = {
    /** Total number of affected rows */
    changes: number;
    /** Rowid of the last inserted row */
    lastInsertRowid: number;
};
export type StatementOptions = Readonly<{
    /**
     * If `true` - the statement is assumed to be long-lived and some otherwise
     * costly optimizations are enabled.
     *
     * The default value is controlled by DatabaseOptions.
     *
     * @see {@link DatabaseOptions}
     */
    persistent?: boolean;
    /**
     * If `true` - `.get()` returns a single column and `.all()` returns a list
     * of column values.
     *
     * Note: the statement must not result in multi-column rows.
     */
    pluck?: true;
    /**
     * If `true` - all integers returned by query will be returned as big
     * integers instead of regular (floating-point) numbers.
     */
    bigint?: true;
}>;
/**
 * Parameters accepted by `.run()`/`.get()`/`.all()` methods of the statement.
 */
export type StatementParameters<Options extends StatementOptions> = ReadonlyArray<SqliteValue<Options>> | Readonly<Record<string, SqliteValue<Options>>>;
/**
 * Possible SQL values given statement options.
 */
export type SqliteValue<Options extends StatementOptions> = string | Uint8Array | number | null | (Options extends {
    bigint: true;
} ? bigint : never);
/**
 * Return value type of `.get()` and an element type of `.all()`
 */
export type RowType<Options extends StatementOptions> = Options extends {
    pluck: true;
} ? SqliteValue<Options> : Record<string, SqliteValue<Options>>;
/**
 * A compiled SQL statement class.
 */
declare class Statement<Options extends StatementOptions = object> {
    #private;
    /**
     * Run the statement's query without returning any rows.
     *
     * @param params - Parameters to be bound to query placeholders before
     *                 executing the statement.
     * @returns An object with `changes` and `lastInsertedRowid` integers.
     */
    run(params?: StatementParameters<Options>): RunResult;
    /**
     * Run the statement's query and return the first row of the result or
     * `undefined` if no rows matched.
     *
     * @param params - Parameters to be bound to query placeholders before
     *                 executing the statement.
     * @returns A row object or a single column if `pluck: true` is set in the
     *          statement options.
     */
    get<Row extends RowType<Options> = RowType<Options>>(params?: StatementParameters<Options>): Row | undefined;
    /**
     * Run the statement's query and return the all rows of the result or
     * `undefined` if no rows matched.
     *
     * @param params - Parameters to be bound to query placeholders before
     *                 executing the statement.
     * @returns A list of row objects or single columns if `pluck: true` is set in
     *          the statement options.
     */
    all<Row extends RowType<Options> = RowType<Options>>(params?: StatementParameters<Options>): Array<Row>;
    /**
     * Close the statement and release the used memory.
     */
    close(): void;
}
export { type Statement };
/**
 * Options for `db.pragma()` method.
 *
 * If `simple` is `true` - pragma returns the first column of the first row of
 * the result.
 */
export type PragmaOptions = Readonly<{
    simple?: true;
}>;
/**
 * Result of `db.pragma()` method.
 *
 * Either a list of rows a single column from the first row depending on the
 * options.
 */
export type PragmaResult<Options extends PragmaOptions> = Options extends {
    simple: true;
} ? RowType<{
    pluck: true;
}> | undefined : Array<RowType<object>>;
export type DatabaseOptions = Readonly<{
    /**
     * If `true` - all statements are persistent by default (unless
     * `persistent` is set to `false` in `StatementOptions`, and persistent
     * statements are automatically cached and reused until closed.
     *
     * @see {@link StatementOptions}
     */
    cacheStatements?: boolean;
}>;
/**
 * A sqlite database class.
 */
export default class Database {
    #private;
    /**
     * Constructor
     *
     * @param path - The path to the database file or ':memory:'/'' for opening
     *               the in-memory database.
     */
    constructor(path?: string, { cacheStatements }?: DatabaseOptions);
    initTokenizer(): void;
    /**
     * Execute one or multiple SQL statements in a given `sql` string.
     *
     * @param sql - one or multiple SQL statements
     */
    exec(sql: string): void;
    /**
     * Compile a single SQL statement.
     *
     * @param query - a single SQL statement.
     * @param options - statement options.
     * @returns Statement instance.
     *
     * @see {@link StatementOptions}
     */
    prepare<Options extends StatementOptions = StatementOptions>(query: string, options: Options): Statement<Options>;
    /**
     * Compile a single SQL statement.
     *
     * @param query - a single SQL statement.
     * @returns Statement instance.
     */
    prepare(query: string): Statement<object>;
    /**
     * Close the database and all associated statements.
     */
    close(): void;
    /**
     * Run a pragma statement and return the result.
     *
     * @param source - pragma query source
     * @param options - options to control the return value of `.pragma()`
     * @returns Either multiple rows returned by the statement, or the first
     *          column of the first row (or `undefined`) if `options` has
     *          `simple: true`.
     *
     * @see {@link PragmaOptions}
     */
    pragma<Options extends PragmaOptions>(source: string, { simple }: Options): PragmaResult<Options>;
    /**
     * Run a pragma statement and return the result.
     *
     * @param source - pragma query source
     * @returns Either multiple rows returned by the statement.
     */
    pragma(source: string): PragmaResult<object>;
    /**
     * Wrap `fn()` in a transaction.
     *
     * @param fn - a function to be executed within a transaction.
     * @returns The value returned by `fn()`.
     */
    transaction<Params extends [], Result>(fn: (...params: Params) => Result): typeof fn;
    /**
     * Tokenize a given sentence with a Signal-FTS5-Extension.
     *
     * @param value - a sentence
     * @returns a list of word-like tokens.
     *
     * @see {@link https://github.com/signalapp/Signal-FTS5-Extension}
     */
    signalTokenize(value: string): Array<string>;
}
export { Database };
