import { underline, yellow } from 'colorette';
import { merge, omit } from 'lodash-es';
import { fileExists, readFileAsync } from './misc.js';
import { DEFAULT_TARGETS, parseTriple } from './target.js';
export const SupportedPackageManagers = ['yarn'];
export const SupportedTestFrameworks = ['ava'];
export async function readNapiConfig(path, configPath) {
    if (configPath && !(await fileExists(configPath))) {
        throw new Error(`NAPI-RS config not found at ${configPath}`);
    }
    if (!(await fileExists(path))) {
        throw new Error(`package.json not found at ${path}`);
    }
    // May support multiple config sources later on.
    const content = await readFileAsync(path, 'utf8');
    let pkgJson;
    try {
        pkgJson = JSON.parse(content);
    }
    catch (e) {
        throw new Error(`Failed to parse package.json at ${path}`, {
            cause: e,
        });
    }
    let separatedConfig;
    if (configPath) {
        const configContent = await readFileAsync(configPath, 'utf8');
        try {
            separatedConfig = JSON.parse(configContent);
        }
        catch (e) {
            throw new Error(`Failed to parse NAPI-RS config at ${configPath}`, {
                cause: e,
            });
        }
    }
    const userNapiConfig = pkgJson.napi ?? {};
    if (pkgJson.napi && separatedConfig) {
        const pkgJsonPath = underline(path);
        const configPathUnderline = underline(configPath);
        console.warn(yellow(`Both napi field in ${pkgJsonPath} and [NAPI-RS config](${configPathUnderline}) file are found, the NAPI-RS config file will be used.`));
        Object.assign(userNapiConfig, separatedConfig);
    }
    const napiConfig = merge({
        binaryName: 'index',
        packageName: pkgJson.name,
        targets: [],
        packageJson: pkgJson,
        npmClient: 'npm',
    }, omit(userNapiConfig, 'targets'));
    let targets = userNapiConfig.targets ?? [];
    // compatible with old config
    if (userNapiConfig?.name) {
        console.warn(yellow(`[DEPRECATED] napi.name is deprecated, use napi.binaryName instead.`));
        napiConfig.binaryName = userNapiConfig.name;
    }
    if (!targets.length) {
        let deprecatedWarned = false;
        const warning = yellow(`[DEPRECATED] napi.triples is deprecated, use napi.targets instead.`);
        if (userNapiConfig.triples?.defaults) {
            deprecatedWarned = true;
            console.warn(warning);
            targets = targets.concat(DEFAULT_TARGETS);
        }
        if (userNapiConfig.triples?.additional?.length) {
            targets = targets.concat(userNapiConfig.triples.additional);
            if (!deprecatedWarned) {
                console.warn(warning);
            }
        }
    }
    napiConfig.targets = targets.map(parseTriple);
    return napiConfig;
}
//# sourceMappingURL=config.js.map