import { loadBindings } from './loadBindings';
let bindings;
if (process.platform === 'win32') {
    bindings = loadBindings(import.meta.url, 'simple-windows-notifications');
}
/**
 * Main class
 */
export class Notifier {
    appId;
    /**
     * @constructor
     * @param appId - Application id, typically: 'org.nodejs.node'
     */
    constructor(appId) {
        this.appId = appId;
    }
    /**
     * Show a notification with a given toast XML.
     *
     * @param toastXml - See https://learn.microsoft.com/en-us/previous-versions/windows/apps/hh761494(v=win.10)
     * @param options - Notification data use to identify the notification.
     */
    show(toastXml, { tag, group, expiresOnReboot = false }) {
        if (!bindings) {
            throw new Error('This library works only on Windows');
        }
        bindings.showNotification(this.appId, toastXml, tag, group, expiresOnReboot);
    }
    /**
     * Remove a notification with a given tag/group
     *
     * @param toastXml - See https://learn.microsoft.com/en-us/previous-versions/windows/apps/hh761494(v=win.10)
     * @param options - Notification data use to identify the notification.
     */
    remove({ tag, group }) {
        if (!bindings) {
            throw new Error('This library works only on Windows');
        }
        bindings.removeNotification(this.appId, tag, group);
    }
    /**
     * Remove all notifications sent by this app.
     */
    clearAll() {
        if (!bindings) {
            throw new Error('This library works only on Windows');
        }
        bindings.clearHistory(this.appId);
    }
}
/**
 * Send a dummy keystroke to the app. Needed when starting a second instance
 * of the app from the notification manager and bringing the first instance to
 * the foreground.
 *
 * See: https://chromium.googlesource.com/chromium/src.git/+/5c432815bbb22210f7c995bbb508359f64baadf5/chrome/notification_helper/notification_activator.cc#155
 * See: https://www.npmjs.com/package/windows-dummy-keystroke#but-why
 */
export function sendDummyKeystroke() {
    if (!bindings) {
        throw new Error('This library works only on Windows');
    }
    bindings.sendDummyKeystroke();
}
