/**
 * Options for showing a notification.
 */
export type ShowNotificationOptionsType = Readonly<{
    /**
     * Notification tag string, used to identify notifications when removing
     * them.
     */
    tag: string;
    /**
     * Notification group string, used to identify notifications when removing
     * them.
     */
    group: string;
    /**
     * If true - the notification will be cleared after reboot.
     */
    expiresOnReboot?: boolean;
}>;
/**
 * Options for removing a notification.
 */
export type RemoveNotificationOptionsType = Readonly<{
    /**
     * Notification tag string, used to identify notifications when removing
     * them.
     */
    tag: string;
    /**
     * Notification group string, used to identify notifications when removing
     * them.
     */
    group: string;
}>;
/**
 * Main class
 */
export declare class Notifier {
    private readonly appId;
    /**
     * @constructor
     * @param appId - Application id, typically: 'org.nodejs.node'
     */
    constructor(appId: string);
    /**
     * Show a notification with a given toast XML.
     *
     * @param toastXml - See https://learn.microsoft.com/en-us/previous-versions/windows/apps/hh761494(v=win.10)
     * @param options - Notification data use to identify the notification.
     */
    show(toastXml: string, { tag, group, expiresOnReboot }: ShowNotificationOptionsType): void;
    /**
     * Remove a notification with a given tag/group
     *
     * @param toastXml - See https://learn.microsoft.com/en-us/previous-versions/windows/apps/hh761494(v=win.10)
     * @param options - Notification data use to identify the notification.
     */
    remove({ tag, group }: RemoveNotificationOptionsType): void;
    /**
     * Remove all notifications sent by this app.
     */
    clearAll(): void;
}
/**
 * Send a dummy keystroke to the app. Needed when starting a second instance
 * of the app from the notification manager and bringing the first instance to
 * the foreground.
 *
 * See: https://chromium.googlesource.com/chromium/src.git/+/5c432815bbb22210f7c995bbb508359f64baadf5/chrome/notification_helper/notification_activator.cc#155
 * See: https://www.npmjs.com/package/windows-dummy-keystroke#but-why
 */
export declare function sendDummyKeystroke(): void;
