import 'package:collection/collection.dart';

import '/src/commons/date_time_utils.dart';
import 'osm_note_comment.dart';
import 'osm_note_status.dart';

/**
 * An immutable container class for an OSM note.
 *
 * For more information visit https://wiki.openstreetmap.org/wiki/Notes
 */
class OSMNote {

  /**
   * The unique identifier of this note.
   *
   * This id is generated by the OSM Server.
   * You shouldn't set the [id] on your own.
   */
  final int id;

  /**
   * The latitude of this note.
   */

  final double lat;

  /**
   * The longitude of this note.
   */
  final double lon;

  /**
   * The current status of this note.
   */
  final OSMNoteStatus status;

  /**
   * The date and time the note was created.
   */
  final DateTime createdAt;

  /**
   * The date and time the note was closed.
   * This will be null if the note is still open.
   */
  final DateTime? closedAt;


  final List<OSMNoteComment> _comments;

  /**
   * All the comments of this note as a list.
   */
  UnmodifiableListView<OSMNoteComment> get comments => UnmodifiableListView(_comments);


  /**
   * A short hand for checking whether this note is still open.
   */
  bool get isOpen => closedAt == null || status == OSMNoteStatus.open;


  /**
   * A short hand for checking whether this note has been closed.
   */
  bool get isClosed => closedAt != null || status == OSMNoteStatus.closed;


  /**
   * A short hand for checking whether this note is hidden.
   */
  bool get isHidden => status == OSMNoteStatus.hidden;


  const OSMNote({
    required this.id,
    required this.lat,
    required this.lon,
    required this.status,
    required this.createdAt,
    required comments,
    this.closedAt,
  }) : _comments = comments;


  /**
   * A factory method for constructing an [OSMNote] from a JSON object.
   */
  factory OSMNote.fromJSONObject(Map<String, dynamic> obj) {
    final List<double> coordinates = obj['geometry']['coordinates'].cast<double>();
    final Map<String, dynamic> properties = obj['properties'];

    return OSMNote(
      id: properties['id'],
      lat: coordinates[0],
      lon: coordinates[1],
      status: osmNoteStatusFromString(
        properties['status'],
      ),
      createdAt: parseUTCDate(properties['date_created']),
      closedAt: properties.containsKey('closed_at')
        ? parseUTCDate(properties['closed_at'])
        : null,
      comments: properties['comments']
        .cast<Map<String, dynamic>>()
        .map<OSMNoteComment>(OSMNoteComment.fromJSONObject)
        .toList(),
    );
  }


  @override
  String toString() => '$runtimeType - id: $id; lat: $lat; lon: $lon; status: $status; createdAt: $createdAt; closedAt: $closedAt; _comments: $_comments';


  @override
  int get hashCode =>
    id.hashCode ^
    lat.hashCode ^
    lon.hashCode ^
    status.hashCode ^
    createdAt.hashCode ^
    closedAt.hashCode ^
    Object.hashAll(_comments);


  @override
  bool operator ==(Object other) {
    if (identical(this, other)) return true;

    return other is OSMNote &&
      other.id == id &&
      other.lat == lat &&
      other.lon == lon &&
      other.status == status &&
      other.createdAt == createdAt &&
      other.closedAt == closedAt &&
      ListEquality().equals(other._comments, _comments);
  }
}
